package main

import (
	"container/heap"
	"log"
	"net"
	"sync"
	"time"
)

type connRecord struct {
	Conn     *queueConn
	LastSeen time.Time
}

// connMap manages a set of live queueConns mapped by address. connMap's
// functions are safe to call from multiple goroutines. The Accept function
// returns newly created queueConns.
type connMap struct {
	// We use an inner structure to avoid exposing public heap.Interface
	// functions to users of connMap.
	inner connMap_
	// Synchronizes access to inner.
	lock sync.Mutex
}

// Create a new connMap that expires conns after a timeout.
func newConnMap(timeout time.Duration) *connMap {
	m := &connMap{
		inner: connMap_{
			byAge:    make([]*connRecord, 0),
			byAddr:   make(map[net.Addr]int),
			acceptCh: make(chan *queueConn),
		},
	}
	go func() {
		for {
			time.Sleep(timeout / 2)
			now := time.Now()
			m.lock.Lock()
			m.inner.removeExpired(now, timeout)
			m.lock.Unlock()
		}
	}()
	return m
}

// Returns an existing Conn or creates one if none exists yet.
func (m *connMap) Get(addr net.Addr, now time.Time) *queueConn {
	m.lock.Lock()
	defer m.lock.Unlock()
	return m.inner.Get(addr, now)
}

func (m *connMap) Accept() (*queueConn, error) {
	return <-m.inner.acceptCh, nil
}

// The inner type of connMap, that implements heap.Interface. byAge is the
// backing store, a heap ordered by last-seen time, which facilitates expiring
// old connRecords. byAddr is a map from addresses (i.e., clientID strings) to
// heap indices, to allow looking up by address. Unlike connMap, this structure
// is unsyncronized and requires external synchonization.
type connMap_ struct {
	byAge  []*connRecord
	byAddr map[net.Addr]int
	// Newly created conns are sent on this channel, to be returned by
	// Accept.
	acceptCh chan *queueConn
}

// heap.Interface for connMap.

func (inner *connMap_) Len() int {
	if len(inner.byAge) != len(inner.byAddr) {
		panic("inconsistent connMap")
	}
	return len(inner.byAge)
}

func (inner *connMap_) Less(i, j int) bool {
	return inner.byAge[i].LastSeen.Before(inner.byAge[j].LastSeen)
}

func (inner *connMap_) Swap(i, j int) {
	inner.byAge[i], inner.byAge[j] = inner.byAge[j], inner.byAge[i]
	inner.byAddr[inner.byAge[i].Conn.RemoteAddr()] = i
	inner.byAddr[inner.byAge[j].Conn.RemoteAddr()] = j
}

func (inner *connMap_) Push(x interface{}) {
	record := x.(*connRecord)
	if _, ok := inner.byAddr[record.Conn.RemoteAddr()]; ok {
		panic("duplicate address in connMap")
	}
	// Insert into byAddr map.
	inner.byAddr[record.Conn.RemoteAddr()] = len(inner.byAge)
	// Insert into byAge slice.
	inner.byAge = append(inner.byAge, record)
}

func (inner *connMap_) Pop() interface{} {
	n := len(inner.byAddr)
	// Remove from byAge slice.
	record := inner.byAge[n-1]
	inner.byAge[n-1] = nil
	inner.byAge = inner.byAge[:n-1]
	// Remove from byAddr map.
	delete(inner.byAddr, record.Conn.RemoteAddr())
	return record
}

// Remove all records whose LastSeen timestamp is more than timeout in the past.
func (inner *connMap_) removeExpired(now time.Time, timeout time.Duration) {
	for len(inner.byAge) > 0 && now.Sub(inner.byAge[0].LastSeen) >= timeout {
		record := heap.Pop(inner).(*connRecord)
		err := record.Conn.Close()
		if err != nil {
			log.Printf("error in queueConn.Close %v: %v", record.Conn.RemoteAddr(), err)
		}
	}
}

// Returns an existing queueConn or creates one if none exists yet.
func (inner *connMap_) Get(addr net.Addr, now time.Time) *queueConn {
	var record *connRecord
	i, ok := inner.byAddr[addr]
	if ok {
		record = inner.byAge[i]
		record.LastSeen = time.Now()
		heap.Fix(inner, i)
	} else {
		// Not found, create a new one.
		record = &connRecord{
			LastSeen: now,
			Conn:     newQueueConn(addr),
		}
		heap.Push(inner, record)
		inner.acceptCh <- record.Conn
	}
	return record.Conn
}
