package main

import (
	"container/heap"
	"net"
	"sync"
	"time"
)

// A record of a recently seen client, with the time it was last seen and a send
// queue. In this case, the "address" of the client will actually be a randomly
// generated clientID string.
type clientRecord struct {
	Addr      net.Addr
	LastSeen  time.Time
	SendQueue chan []byte
}

// clientMap manages a set of live clients and their send queues. clientMap's
// functions are safe to call from multiple goroutines.
type clientMap struct {
	// We use an inner structure to avoid exposing public heap.Interface
	// functions to users of clientMap.
	inner clientMap_
	// Synchronizes access to inner.
	lock sync.Mutex
}

// Create a new clientMap that expires client after a timeout.
func newClientMap(timeout time.Duration) *clientMap {
	m := &clientMap{
		inner: clientMap_{
			byAge:  make([]*clientRecord, 0),
			byAddr: make(map[net.Addr]int),
		},
	}
	go func() {
		for {
			time.Sleep(timeout / 2)
			now := time.Now()
			m.lock.Lock()
			m.inner.removeExpired(now, timeout)
			m.lock.Unlock()
		}
	}()
	return m
}

// Returns an existing send queue or creates one if none exists yet.
func (m *clientMap) SendQueue(addr net.Addr, now time.Time) chan []byte {
	m.lock.Lock()
	defer m.lock.Unlock()
	return m.inner.SendQueue(addr, now)
}

// The inner type of clientMap, that implements heap.Interface. byAge is the
// backing store, a heap ordered by last-seen time, which facilitates expiring
// old client records. byAddr is a map from addresses (i.e., clientID strings)
// to heap indices, to allow looking up by address. Unlike clientMap, this
// structure is unsyncronized and requires external synchonization.
type clientMap_ struct {
	byAge  []*clientRecord
	byAddr map[net.Addr]int
}

// heap.Interface for clientMap.

func (inner *clientMap_) Len() int {
	if len(inner.byAge) != len(inner.byAddr) {
		panic("inconsistent clientMap")
	}
	return len(inner.byAge)
}

func (inner *clientMap_) Less(i, j int) bool {
	return inner.byAge[i].LastSeen.Before(inner.byAge[j].LastSeen)
}

func (inner *clientMap_) Swap(i, j int) {
	inner.byAge[i], inner.byAge[j] = inner.byAge[j], inner.byAge[i]
	inner.byAddr[inner.byAge[i].Addr] = i
	inner.byAddr[inner.byAge[j].Addr] = j
}

func (inner *clientMap_) Push(x interface{}) {
	record := x.(*clientRecord)
	if _, ok := inner.byAddr[record.Addr]; ok {
		panic("duplicate address in clientMap")
	}
	// Insert into byAddr map.
	inner.byAddr[record.Addr] = len(inner.byAge)
	// Insert into byAge slice.
	inner.byAge = append(inner.byAge, record)
}

func (inner *clientMap_) Pop() interface{} {
	n := len(inner.byAddr)
	// Remove from byAge slice.
	record := inner.byAge[n-1]
	inner.byAge[n-1] = nil
	inner.byAge = inner.byAge[:n-1]
	// Remove from byAddr map.
	delete(inner.byAddr, record.Addr)
	return record
}

// Remove all clients whose LastSeen timestamp is more than timeout in the past.
func (inner *clientMap_) removeExpired(now time.Time, timeout time.Duration) {
	for len(inner.byAge) > 0 && now.Sub(inner.byAge[0].LastSeen) >= timeout {
		heap.Pop(inner)
	}
}

// Returns an existing send queue or creates one if none exists yet.
func (inner *clientMap_) SendQueue(addr net.Addr, now time.Time) chan []byte {
	var record *clientRecord
	i, ok := inner.byAddr[addr]
	if ok {
		record = inner.byAge[i]
		record.LastSeen = time.Now()
		heap.Fix(inner, i)
	} else {
		// Not found, create a new one.
		record = &clientRecord{
			Addr:      addr,
			LastSeen:  now,
			SendQueue: make(chan []byte, queueCapacity),
		}
		heap.Push(inner, record)
	}
	return record.SendQueue
}
