/*
	sgf.h
*/

#ifndef _SGF_H
#define _SGF_H

#include <stdio.h>

enum {
	SGF_PROPERTY_VALUE_NONE = 1 << 0,
	SGF_PROPERTY_VALUE_SINGLE = 1 << 1,
	SGF_PROPERTY_VALUE_COMPOSED = 1 << 2,
	SGF_PROPERTY_LIST = 1 << 3,
	SGF_PROPERTY_ELIST = 1 << 4,
	SGF_PROPERTY_VALUE_MASK = 0x07,
	SGF_PROPERTY_MASK = 0x18
};

enum {
	SGF_GAME_GO = 1
};

enum {
	SGF_PROPERTY_TYPE_ROOT,
	SGF_PROPERTY_TYPE_GAME_INFO,
	SGF_PROPERTY_TYPE_SETUP,
	SGF_PROPERTY_TYPE_MOVE,
	SGF_PROPERTY_TYPE_NONE
};

enum {
	SGF_FILE_MAX_LINE_LENGTH = 80
};

typedef union sgf_property_value_type sgf_property_value_type;
union sgf_property_value_type {
	int number;
	float real;
	unsigned int binary;
	unsigned int color;
	char *simpletext;
	char *text;
	/* The Stone, Move and Point types are game-specific, so leave them
	   undefined. Property value type handlers can also store these
	   types in any of the other members. */
	void *stone, *move, *point;
	/* Store the values of unknown properties as character strings */
	char *unknown;
};

typedef struct sgf_parser sgf_parser;

typedef struct sgf_property_value_type_handler sgf_property_value_type_handler;
struct sgf_property_value_type_handler {
	void (*free)(sgf_property_value_type *);
	sgf_property_value_type *(*duplicate)(sgf_property_value_type *, const sgf_property_value_type *);
	int (*cmp)(const sgf_property_value_type *, const sgf_property_value_type *);
	sgf_property_value_type *(*read)(sgf_property_value_type *, struct sgf_parser *);
	char *(*compose)(const sgf_property_value_type *, unsigned int);
};

typedef struct sgf_property_type sgf_property_type;
struct sgf_property_type {
	char *name;
	int type;
	unsigned int priority;
	unsigned int flags;
	sgf_property_value_type_handler *handlers[2];
};

typedef struct sgf_property_value sgf_property_value;
struct sgf_property_value {
	unsigned int flags;
	sgf_property_value_type types[2];
	sgf_property_value_type_handler **handlers;
};

typedef struct sgf_property sgf_property;
struct sgf_property {
	char *name;
	sgf_property_type *type;
	unsigned int num_values;
	sgf_property_value *values;
};

typedef struct sgf_node sgf_node;
struct sgf_node {
	unsigned int num_properties;
	sgf_property *properties;
	sgf_node *next;
	sgf_node *variation;
};

typedef sgf_node *sgf_game;

typedef struct sgf_collection sgf_collection;
struct sgf_collection {
	unsigned int num_games;
	sgf_game *games;
};

typedef void (*sgf_parser_handler)(sgf_parser *);

typedef void (*sgf_parser_property_handler)(sgf_parser *, const sgf_property *);

struct sgf_parser {
	FILE *fp;
	sgf_parser_handler game_start;
	sgf_parser_handler game_end;
	sgf_parser_handler gametree_start;
	sgf_parser_handler gametree_end;
	sgf_parser_handler node_start;
	sgf_parser_handler node_end;
	sgf_parser_property_handler property;
	void *user_data;
	unsigned int num_property_types;
	sgf_property_type *property_types;
	unsigned int is_running;
	unsigned int line;
	unsigned int column;
	unsigned int prev_column;
	char *charset;
	unsigned int format;
	unsigned int game;
};

sgf_property_value_type_handler sgf_property_value_type_handler_number;

sgf_property_value_type_handler sgf_property_value_type_handler_real;

sgf_property_value_type_handler sgf_property_value_type_handler_binary;

sgf_property_value_type_handler sgf_property_value_type_handler_color;

sgf_property_value_type_handler sgf_property_value_type_handler_simpletext;

sgf_property_value_type_handler sgf_property_value_type_handler_simpletext_composed;

sgf_property_value_type_handler sgf_property_value_type_handler_text;

sgf_property_value_type_handler sgf_property_value_type_handler_text_composed;

sgf_property_value_type_handler sgf_property_value_type_handler_go_point;

sgf_property_value_type_handler sgf_property_value_type_handler_unknown;

void sgf_property_value_free(sgf_property_value *value);

sgf_property_value *sgf_property_value_duplicate(sgf_property_value *a, const sgf_property_value *b);

sgf_property *sgf_property_init(sgf_property *property);

void sgf_property_free(sgf_property *property);

sgf_property *sgf_property_duplicate(sgf_property *a, const sgf_property *b);

sgf_property *sgf_property_add_value(sgf_property *property, const sgf_property_value *value);

sgf_property *sgf_property_remove_value(sgf_property *property, const sgf_property_value *value);

sgf_node *sgf_node_init(sgf_node *node);

void sgf_node_free(sgf_node *node);

sgf_node *sgf_node_add_property(sgf_node *node, sgf_property *property);

sgf_node *sgf_node_remove_property(sgf_node *node, const sgf_property *property);

sgf_property *sgf_node_search_properties(const sgf_node *node, const sgf_property *property);

sgf_property *sgf_node_search_properties_by_name(const sgf_node *node, const char *name);

void sgf_gametree_free(sgf_node *node);

sgf_game *sgf_game_init(sgf_game *game);

void sgf_game_free(sgf_game *game);

sgf_game *sgf_game_read(sgf_game *game, sgf_parser *parser);

sgf_game *sgf_game_read_shallow(sgf_game *game, sgf_parser *parser);

sgf_game *sgf_game_write(sgf_game *game, sgf_parser *parser);

sgf_collection *sgf_collection_init(sgf_collection *collection);

void sgf_collection_free(sgf_collection *collection);

sgf_collection *sgf_collection_read(sgf_collection *collection, sgf_parser *parser);

sgf_collection *sgf_collection_read_shallow(sgf_collection *collection, sgf_parser *parser);

sgf_collection *sgf_collection_write(sgf_collection *collection, sgf_parser *parser);

sgf_parser *sgf_parser_init(sgf_parser *parser);

void sgf_parser_free(sgf_parser *parser);

sgf_parser *sgf_parser_open_file(sgf_parser *parser, const char *filename, const char *mode);

void sgf_parser_set_game_handlers(sgf_parser *parser, sgf_parser_handler start, sgf_parser_handler end);

void sgf_parser_set_gametree_handlers(sgf_parser *parser, sgf_parser_handler start, sgf_parser_handler end);

void sgf_parser_set_node_handlers(sgf_parser *parser, sgf_parser_handler start, sgf_parser_handler end);

void sgf_parser_set_property_handler(sgf_parser *parser, sgf_parser_property_handler f);

void sgf_parser_set_user_data(sgf_parser *parser, void *user_data);

sgf_parser *sgf_parser_add_property_type(sgf_parser *parser, const sgf_property_type *type);

sgf_parser *sgf_parser_remove_property_type(sgf_parser *parser, const sgf_property_type *type);

sgf_parser *sgf_parser_parse(sgf_parser *parser);

void sgf_parser_stop(sgf_parser *parser);

sgf_property *sgf_parser_property_set_name(const sgf_parser *parser, sgf_property *property, const char *name);

#endif
