/*
	sgf.c
*/

#include <ctype.h>
#include <errno.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "array.h"
#include "sgf.h"

/* Check the value of errno when a read function returns NULL, and skip
   ahead to a known character if it's 0 */

#define SGF_SOFT_LINE_BREAK "\\\n"

enum {
	SGF_FILE_BREAK_STYLE_TEXT,
	SGF_FILE_BREAK_STYLE_SIMPLETEXT,
	SGF_FILE_BREAK_STYLE_UNKNOWN
};

typedef struct sgf_game_read_helper sgf_game_read_helper;
struct sgf_game_read_helper {
	sgf_node *root_node;
	sgf_node *current_node;
	sgf_collection *collection;
	unsigned int stack_size;
	sgf_node **stack;
};

/* I guess these should go back in sgf.h */
static void sgf_property_value_type_free_simpletext(sgf_property_value_type *type);

static void sgf_property_value_type_free_text(sgf_property_value_type *type);

static void sgf_property_value_type_free_unknown(sgf_property_value_type *type);

static sgf_property_value_type *sgf_property_value_type_duplicate_simpletext(sgf_property_value_type *a, const sgf_property_value_type *b);

static sgf_property_value_type *sgf_property_value_type_duplicate_text(sgf_property_value_type *a, const sgf_property_value_type *b);

static sgf_property_value_type *sgf_property_value_type_duplicate_unknown(sgf_property_value_type *a, const sgf_property_value_type *b);

static int sgf_property_value_type_cmp_number(const sgf_property_value_type *a, const sgf_property_value_type *b);

static int sgf_property_value_type_cmp_real(const sgf_property_value_type *a, const sgf_property_value_type *b);

static int sgf_property_value_type_cmp_binary(const sgf_property_value_type *a, const sgf_property_value_type *b);

static int sgf_property_value_type_cmp_color(const sgf_property_value_type *a, const sgf_property_value_type *b);

static int sgf_property_value_type_cmp_simpletext(const sgf_property_value_type *a, const sgf_property_value_type *b);

static int sgf_property_value_type_cmp_text(const sgf_property_value_type *a, const sgf_property_value_type *b);

static int sgf_property_value_type_cmp_go_point(const sgf_property_value_type *a, const sgf_property_value_type *b);

static int sgf_property_value_type_cmp_unknown(const sgf_property_value_type *a, const sgf_property_value_type *b);

static sgf_property_value_type *sgf_property_value_type_read_number(sgf_property_value_type *type, sgf_parser *parser);

static sgf_property_value_type *sgf_property_value_type_read_real(sgf_property_value_type *type, sgf_parser *parser);

static sgf_property_value_type *sgf_property_value_type_read_binary(sgf_property_value_type *type, sgf_parser *parser);

static sgf_property_value_type *sgf_property_value_type_read_color(sgf_property_value_type *type, sgf_parser *parser);

static sgf_property_value_type *sgf_property_value_type_read_simpletext(sgf_property_value_type *type, sgf_parser *parser);

static sgf_property_value_type *sgf_property_value_type_read_simpletext_composed(sgf_property_value_type *type, sgf_parser *parser);

static sgf_property_value_type *sgf_property_value_type_read_text(sgf_property_value_type *type, sgf_parser *parser);

static sgf_property_value_type *sgf_property_value_type_read_text_composed(sgf_property_value_type *type, sgf_parser *parser);

static sgf_property_value_type *sgf_property_value_type_read_go_point(sgf_property_value_type *type, sgf_parser *parser);

static sgf_property_value_type *sgf_property_value_type_read_unknown(sgf_property_value_type *type, sgf_parser *parser);

static char *sgf_property_value_type_compose_number(const sgf_property_value_type *type, unsigned int column);

static char *sgf_property_value_type_compose_real(const sgf_property_value_type *type, unsigned int column);

static char *sgf_property_value_type_compose_binary(const sgf_property_value_type *type, unsigned int column);

static char *sgf_property_value_type_compose_color(const sgf_property_value_type *type, unsigned int column);

static char *sgf_property_value_type_compose_simpletext(const sgf_property_value_type *type, unsigned int column);

static char *sgf_property_value_type_compose_text(const sgf_property_value_type *type, unsigned int column);

static char *sgf_property_value_type_compose_go_point(const sgf_property_value_type *type, unsigned int column);

static char *sgf_property_value_type_compose_unknown(const sgf_property_value_type *type, unsigned int column);

sgf_property_value_type_handler sgf_property_value_type_handler_number = {
	NULL, NULL, sgf_property_value_type_cmp_number, sgf_property_value_type_read_number, sgf_property_value_type_compose_number
}, sgf_property_value_type_handler_real = {
	NULL, NULL, sgf_property_value_type_cmp_real, sgf_property_value_type_read_real, sgf_property_value_type_compose_real
}, sgf_property_value_type_handler_binary = {
	NULL, NULL, sgf_property_value_type_cmp_binary, sgf_property_value_type_read_binary, sgf_property_value_type_compose_binary
}, sgf_property_value_type_handler_color = {
	NULL, NULL, sgf_property_value_type_cmp_color, sgf_property_value_type_read_color, sgf_property_value_type_compose_color
}, sgf_property_value_type_handler_simpletext = {
	sgf_property_value_type_free_simpletext, sgf_property_value_type_duplicate_simpletext, sgf_property_value_type_cmp_simpletext, sgf_property_value_type_read_simpletext, sgf_property_value_type_compose_simpletext
}, sgf_property_value_type_handler_simpletext_composed = {
	sgf_property_value_type_free_simpletext, sgf_property_value_type_duplicate_simpletext, sgf_property_value_type_cmp_simpletext, sgf_property_value_type_read_simpletext_composed, sgf_property_value_type_compose_simpletext
}, sgf_property_value_type_handler_text = {
	sgf_property_value_type_free_text, sgf_property_value_type_duplicate_text, sgf_property_value_type_cmp_text, sgf_property_value_type_read_text, sgf_property_value_type_compose_text
}, sgf_property_value_type_handler_text_composed = {
	sgf_property_value_type_free_text, sgf_property_value_type_duplicate_text, sgf_property_value_type_cmp_text, sgf_property_value_type_read_text_composed, sgf_property_value_type_compose_text
}, sgf_property_value_type_handler_go_point = {
	NULL, NULL, sgf_property_value_type_cmp_go_point, sgf_property_value_type_read_go_point, sgf_property_value_type_compose_go_point
}, sgf_property_value_type_handler_unknown = {
	sgf_property_value_type_free_unknown, sgf_property_value_type_duplicate_unknown, sgf_property_value_type_cmp_unknown, sgf_property_value_type_read_unknown, sgf_property_value_type_compose_unknown
};

/* Check game-info properties (CP, EV, GN, PC and SO) and make sure they're
   really Text. Fix things in game_sgf.c if needed. */
static const sgf_property_type sgf_property_types_common[] = {
	/* The SGF specification says that the value of the AP property must
	   be composed SimpleText ":" SimpleText, but non-composed SimpleText
	   values are used in the files you download from Sensei's Library */
	{ "AP", SGF_PROPERTY_TYPE_ROOT, 50, SGF_PROPERTY_VALUE_SINGLE | SGF_PROPERTY_VALUE_COMPOSED, { &sgf_property_value_type_handler_simpletext_composed, &sgf_property_value_type_handler_simpletext } },
	/* { "BL", SGF_PROPERTY_TYPE_MOVE, 75, SGF_PROPERTY_VALUE_SINGLE, { &sgf_property_value_type_handler_real, NULL } }, */
	/* { "BR", SGF_PROPERTY_TYPE_GAME_INFO, 21, SGF_PROPERTY_VALUE_SINGLE, { } }, */
	/* { "BM", SGF_PROPERTY_TYPE_MOVE, 70, SGF_PROPERTY_VALUE_SINGLE, { &sgf_property_value_type_handler_binary, NULL } }, */
	{ "C", SGF_PROPERTY_TYPE_NONE, 90, SGF_PROPERTY_VALUE_SINGLE, { &sgf_property_value_type_handler_text, NULL } },
	{ "CA", SGF_PROPERTY_TYPE_ROOT, 1, SGF_PROPERTY_VALUE_SINGLE, { &sgf_property_value_type_handler_simpletext, NULL } },
	{ "CP", SGF_PROPERTY_TYPE_GAME_INFO, 55, SGF_PROPERTY_VALUE_SINGLE, { &sgf_property_value_type_handler_text, NULL } },
	{ "EV", SGF_PROPERTY_TYPE_GAME_INFO, 11, SGF_PROPERTY_VALUE_SINGLE, { &sgf_property_value_type_handler_text, NULL } },
	{ "FF", SGF_PROPERTY_TYPE_ROOT, 0, SGF_PROPERTY_VALUE_SINGLE, { &sgf_property_value_type_handler_number, NULL } },
	{ "GM", SGF_PROPERTY_TYPE_ROOT, 2, SGF_PROPERTY_VALUE_SINGLE, { &sgf_property_value_type_handler_number, NULL } },
	{ "GN", SGF_PROPERTY_TYPE_GAME_INFO, 10, SGF_PROPERTY_VALUE_SINGLE, { &sgf_property_value_type_handler_text, NULL } },
	{ "PB", SGF_PROPERTY_TYPE_GAME_INFO, 20, SGF_PROPERTY_VALUE_SINGLE, { &sgf_property_value_type_handler_simpletext, NULL } },
	{ "PC", SGF_PROPERTY_TYPE_GAME_INFO, 50, SGF_PROPERTY_VALUE_SINGLE, { &sgf_property_value_type_handler_text, NULL } },
	{ "PL", SGF_PROPERTY_TYPE_SETUP, 0, SGF_PROPERTY_VALUE_SINGLE, { &sgf_property_value_type_handler_color, NULL } },
	{ "PW", SGF_PROPERTY_TYPE_GAME_INFO, 22, SGF_PROPERTY_VALUE_SINGLE, { &sgf_property_value_type_handler_simpletext, NULL } },
	{ "RE", SGF_PROPERTY_TYPE_GAME_INFO, 25, SGF_PROPERTY_VALUE_SINGLE, { &sgf_property_value_type_handler_simpletext, NULL } },
	{ "RU", SGF_PROPERTY_TYPE_GAME_INFO, 40, SGF_PROPERTY_VALUE_SINGLE, { &sgf_property_value_type_handler_simpletext, NULL } },
	{ "SO", SGF_PROPERTY_TYPE_GAME_INFO, 50, SGF_PROPERTY_VALUE_SINGLE, { &sgf_property_value_type_handler_text, NULL } },
	/* { "ST", SGF_PROPERTY_TYPE_ROOT, 15, SGF_PROPERTY_VALUE_SINGLE, { &sgf_property_value_type_handler_number, NULL } }, */
	{ "SZ", SGF_PROPERTY_TYPE_ROOT, 10, SGF_PROPERTY_VALUE_SINGLE | SGF_PROPERTY_VALUE_COMPOSED, { &sgf_property_value_type_handler_number, &sgf_property_value_type_handler_number } }
	/* { "WL", SGF_PROPERTY_TYPE_MOVE, 76, SGF_PROPERTY_VALUE_SINGLE, { &sgf_property_value_type_handler_real, NULL } } */
	/* { "WR", SGF_PROPERTY_TYPE_GAME_INFO, 23, SGF_PROPERTY_VALUE_SINGLE, { } }, */
};

static const sgf_property_type sgf_property_types_go[] = {
	{ "AB", SGF_PROPERTY_TYPE_SETUP, 10, SGF_PROPERTY_VALUE_SINGLE | SGF_PROPERTY_VALUE_COMPOSED | SGF_PROPERTY_LIST, { &sgf_property_value_type_handler_go_point, &sgf_property_value_type_handler_go_point } },
	{ "AE", SGF_PROPERTY_TYPE_SETUP, 12, SGF_PROPERTY_VALUE_SINGLE | SGF_PROPERTY_VALUE_COMPOSED | SGF_PROPERTY_LIST, { &sgf_property_value_type_handler_go_point, &sgf_property_value_type_handler_go_point } },
	{ "AR", SGF_PROPERTY_TYPE_NONE, 61, SGF_PROPERTY_VALUE_COMPOSED | SGF_PROPERTY_LIST, { &sgf_property_value_type_handler_go_point, &sgf_property_value_type_handler_go_point } },
	{ "AW", SGF_PROPERTY_TYPE_SETUP, 11, SGF_PROPERTY_VALUE_SINGLE | SGF_PROPERTY_VALUE_COMPOSED | SGF_PROPERTY_LIST, { &sgf_property_value_type_handler_go_point, &sgf_property_value_type_handler_go_point } },
	{ "B", SGF_PROPERTY_TYPE_MOVE, 10, SGF_PROPERTY_VALUE_NONE | SGF_PROPERTY_VALUE_SINGLE, { &sgf_property_value_type_handler_go_point, NULL } },
	{ "CR", SGF_PROPERTY_TYPE_NONE, 62, SGF_PROPERTY_VALUE_SINGLE | SGF_PROPERTY_VALUE_COMPOSED | SGF_PROPERTY_LIST, { &sgf_property_value_type_handler_go_point, &sgf_property_value_type_handler_go_point } },
	{ "KM", SGF_PROPERTY_TYPE_GAME_INFO, 90, SGF_PROPERTY_VALUE_SINGLE, { &sgf_property_value_type_handler_real, NULL } },
	{ "HA", SGF_PROPERTY_TYPE_GAME_INFO, 85, SGF_PROPERTY_VALUE_SINGLE, { &sgf_property_value_type_handler_number, NULL } },
	{ "LB", SGF_PROPERTY_TYPE_NONE, 60, SGF_PROPERTY_VALUE_COMPOSED | SGF_PROPERTY_LIST, { &sgf_property_value_type_handler_go_point, &sgf_property_value_type_handler_simpletext } },
	{ "LN", SGF_PROPERTY_TYPE_NONE, 61, SGF_PROPERTY_VALUE_COMPOSED | SGF_PROPERTY_LIST, { &sgf_property_value_type_handler_go_point, &sgf_property_value_type_handler_go_point } },
	{ "MA", SGF_PROPERTY_TYPE_NONE, 62, SGF_PROPERTY_VALUE_SINGLE | SGF_PROPERTY_VALUE_COMPOSED | SGF_PROPERTY_LIST, { &sgf_property_value_type_handler_go_point, &sgf_property_value_type_handler_go_point } },
	{ "SL", SGF_PROPERTY_TYPE_NONE, 62, SGF_PROPERTY_VALUE_SINGLE | SGF_PROPERTY_VALUE_COMPOSED | SGF_PROPERTY_LIST, { &sgf_property_value_type_handler_go_point, &sgf_property_value_type_handler_go_point } },
	{ "SQ", SGF_PROPERTY_TYPE_NONE, 62, SGF_PROPERTY_VALUE_SINGLE | SGF_PROPERTY_VALUE_COMPOSED | SGF_PROPERTY_LIST, { &sgf_property_value_type_handler_go_point, &sgf_property_value_type_handler_go_point } },
	{ "TB", SGF_PROPERTY_TYPE_NONE, 55, SGF_PROPERTY_VALUE_SINGLE | SGF_PROPERTY_VALUE_COMPOSED | SGF_PROPERTY_ELIST, { &sgf_property_value_type_handler_go_point, &sgf_property_value_type_handler_go_point } },
	{ "TR", SGF_PROPERTY_TYPE_NONE, 62, SGF_PROPERTY_VALUE_SINGLE | SGF_PROPERTY_VALUE_COMPOSED | SGF_PROPERTY_ELIST, { &sgf_property_value_type_handler_go_point, &sgf_property_value_type_handler_go_point } },
	{ "TW", SGF_PROPERTY_TYPE_NONE, 56, SGF_PROPERTY_VALUE_SINGLE | SGF_PROPERTY_VALUE_COMPOSED | SGF_PROPERTY_LIST, { &sgf_property_value_type_handler_go_point, &sgf_property_value_type_handler_go_point } },
	{ "W", SGF_PROPERTY_TYPE_MOVE, 11, SGF_PROPERTY_VALUE_NONE | SGF_PROPERTY_VALUE_SINGLE, { &sgf_property_value_type_handler_go_point, NULL } }
};

static const struct {
	unsigned int num_property_types;
	const sgf_property_type *property_types;
} sgf_property_types_game_specific[] = {
	{ 0, NULL },
	{ sizeof(sgf_property_types_go) / sizeof(*sgf_property_types_go), sgf_property_types_go }
};

static char *strlencat(char **s, size_t *length, const char *t);

static char *strlencatchar(char **s, size_t *length, char c);

static char *compose_text(const char *s, const char *escaped_chars, unsigned int column, int break_style);

static int sgf_property_value_cmp(const sgf_property_value *a, const sgf_property_value *b);

static sgf_property_value *sgf_property_value_read(sgf_property_value *value, sgf_parser *parser);

static char *sgf_property_value_compose(const sgf_property_value *value, unsigned int column);

static int sgf_property_cmp(const sgf_property *a, const sgf_property *b);

static int sgf_property_name_cmp(const char *name, const sgf_property *property);

static sgf_property *sgf_property_read(sgf_property *property, sgf_parser *parser);

static sgf_game_read_helper *sgf_game_read_helper_init(sgf_game_read_helper *helper);

static void sgf_game_read_helper_free(sgf_game_read_helper *helper);

static sgf_game_read_helper *sgf_game_read_helper_push(sgf_game_read_helper *helper, sgf_node *node);

static sgf_game_read_helper *sgf_game_read_helper_pop(sgf_game_read_helper *helper, sgf_node **node);

static void sgf_game_read_game_start(sgf_parser *parser);

static void sgf_game_read_game_end(sgf_parser *parser);

static void sgf_game_read_gametree_start(sgf_parser *parser);

static void sgf_game_read_gametree_end(sgf_parser *parser);

static void sgf_game_read_node_start(sgf_parser *parser);

static void sgf_game_read_property(sgf_parser *parser, const sgf_property *property);

static void sgf_game_read_shallow_node_end(sgf_parser *parser);

static sgf_node *sgf_gametree_write(sgf_node *node, sgf_parser *parser);

static sgf_collection *sgf_collection_add_game(sgf_collection *collection, const sgf_game *game);

static void sgf_collection_read_game_end(sgf_parser *parser);

static void sgf_collection_read_shallow_game_end(sgf_parser *parser);

static sgf_parser *sgf_parser_parse_gametree(sgf_parser *parser);

static sgf_parser *sgf_parser_parse_node(sgf_parser *parser);

static int sgf_parser_get_char(sgf_parser *parser);

static int sgf_parser_unget_char(sgf_parser *parser, int c);

static int sgf_parser_put_char(sgf_parser *parser, int c);

static int sgf_parser_write_string(sgf_parser *parser, const char *s);

static void sgf_parser_skip_space(sgf_parser *parser);

static char *sgf_parser_read_text(sgf_parser *parser, const char *terminators, int simple);

static int sgf_property_type_name_cmp(const char *name, const sgf_property_type *property_type);

static sgf_property_type *sgf_property_types_search(const sgf_property_type *array, unsigned int num, const char *name);

static sgf_property_type *sgf_parser_search_property_types(const sgf_parser *parser, const char *name);

static void sgf_property_value_type_free_simpletext(sgf_property_value_type *type)
{
	free(type->simpletext);

	return;
}

static void sgf_property_value_type_free_text(sgf_property_value_type *type)
{
	free(type->text);

	return;
}

static void sgf_property_value_type_free_unknown(sgf_property_value_type *type)
{
	free(type->unknown);

	return;
}

static sgf_property_value_type *sgf_property_value_type_duplicate_simpletext(sgf_property_value_type *a, const sgf_property_value_type *b)
{
	if (b->simpletext == NULL)
		return NULL;
	a->simpletext = (char *) malloc(strlen(b->simpletext) + 1);
	if (a->simpletext == NULL)
		return NULL;
	strcpy(a->simpletext, b->simpletext);

	return a;
}

static sgf_property_value_type *sgf_property_value_type_duplicate_text(sgf_property_value_type *a, const sgf_property_value_type *b)
{
	if (b->text == NULL)
		return NULL;
	a->text = (char *) malloc(strlen(b->text) + 1);
	if (a->text == NULL)
		return NULL;
	strcpy(a->text, b->text);

	return a;
}

static sgf_property_value_type *sgf_property_value_type_duplicate_unknown(sgf_property_value_type *a, const sgf_property_value_type *b)
{
	if (b->unknown == NULL)
		return NULL;
	a->unknown = (char *) malloc(strlen(b->unknown) + 1);
	if (a->unknown == NULL)
		return NULL;
	strcpy(a->unknown, b->unknown);

	return a;
}

static int sgf_property_value_type_cmp_number(const sgf_property_value_type *a, const sgf_property_value_type *b)
{
	return a->number - b->number;
}

static int sgf_property_value_type_cmp_real(const sgf_property_value_type *a, const sgf_property_value_type *b)
{
	return a->real - b->real;
}

static int sgf_property_value_type_cmp_binary(const sgf_property_value_type *a, const sgf_property_value_type *b)
{
	return (int) a->binary - b->binary;
}

static int sgf_property_value_type_cmp_color(const sgf_property_value_type *a, const sgf_property_value_type *b)
{
	return (int) a->color - b->color;
}

static int sgf_property_value_type_cmp_simpletext(const sgf_property_value_type *a, const sgf_property_value_type *b)
{
	return strcmp(a->simpletext, b->simpletext);
}

static int sgf_property_value_type_cmp_text(const sgf_property_value_type *a, const sgf_property_value_type *b)
{
	return strcmp(a->text, b->text);
}

static int sgf_property_value_type_cmp_go_point(const sgf_property_value_type *a, const sgf_property_value_type *b)
{
	return (int) a->number - b->number;
}

static int sgf_property_value_type_cmp_unknown(const sgf_property_value_type *a, const sgf_property_value_type *b)
{
	return strcmp(a->unknown, b->unknown);
}

static sgf_property_value_type *sgf_property_value_type_read_number(sgf_property_value_type *type, sgf_parser *parser)
{
	int c, sign;

	type->number = 0;
	sign = 1;
	sgf_parser_skip_space(parser);
	c = sgf_parser_get_char(parser);
	if ((c == '+') || (c == '-')) {
		if (c == '-')
			sign = -1;
		c = sgf_parser_get_char(parser);
	}
	if (!isdigit(c)) {
		sgf_parser_unget_char(parser, c);
		return NULL;
	}
	do {
		type->number *= 10;
		type->number += c - '0';
		c = sgf_parser_get_char(parser);
	} while (isdigit(c));
	type->number *= sign;
	sgf_parser_unget_char(parser, c);
	sgf_parser_skip_space(parser);

	return type;
}

static sgf_property_value_type *sgf_property_value_type_read_real(sgf_property_value_type *type, sgf_parser *parser)
{
	int c, number;
	float divisor;
	
	if (sgf_property_value_type_read_number((sgf_property_value_type *) &number, parser) == NULL)
		return NULL;
	type->real = (float) number;
	c = sgf_parser_get_char(parser);
	if (c != '.') {
		sgf_parser_unget_char(parser, c);
		return type;
	}
	c = sgf_parser_get_char(parser);
	if (!isdigit(c)) {
		sgf_parser_unget_char(parser, c);
		return NULL;
	}
	if (type->real >= 0)
		divisor = 1.0;
	else
		divisor = -1.0;
	do {
		divisor *= 10;
		type->real += (c - '0') / divisor;
		c = sgf_parser_get_char(parser);
	} while (isdigit(c));
	sgf_parser_unget_char(parser, c);
	sgf_parser_skip_space(parser);

	return type;
}

static sgf_property_value_type *sgf_property_value_type_read_binary(sgf_property_value_type *type, sgf_parser *parser)
{
	int c;

	sgf_parser_skip_space(parser);
	c = sgf_parser_get_char(parser);
	if (c == '1') {
		type->binary = 1;
	} else if (c == '2') {
		type->binary = 2;
	} else {
		sgf_parser_unget_char(parser, c);
		return NULL;
	}
	sgf_parser_skip_space(parser);

	return type;
}

static sgf_property_value_type *sgf_property_value_type_read_color(sgf_property_value_type *type, sgf_parser *parser)
{
	int c;

	sgf_parser_skip_space(parser);
	c = sgf_parser_get_char(parser);
	if (c == 'B') {
		type->color = 1;
	} else if (c == 'W') {
		type->color = 2;
	} else {
		sgf_parser_unget_char(parser, c);
		return NULL;
	}
	sgf_parser_skip_space(parser);

	return type;
}

static sgf_property_value_type *sgf_property_value_type_read_simpletext(sgf_property_value_type *type, sgf_parser *parser)
{
	type->simpletext = sgf_parser_read_text(parser, "]", 1);
	if (type->simpletext == NULL)
		return NULL;

	return type;
}

static sgf_property_value_type *sgf_property_value_type_read_simpletext_composed(sgf_property_value_type *type, sgf_parser *parser)
{
	type->simpletext = sgf_parser_read_text(parser, ":]", 1);
	if (type->simpletext == NULL)
		return NULL;

	return type;
}

static sgf_property_value_type *sgf_property_value_type_read_text(sgf_property_value_type *type, sgf_parser *parser)
{
	type->text = sgf_parser_read_text(parser, "]", 0);
	if (type->text == NULL)
		return NULL;

	return type;
}

static sgf_property_value_type *sgf_property_value_type_read_text_composed(sgf_property_value_type *type, sgf_parser *parser)
{
	type->text = sgf_parser_read_text(parser, ":]", 0);
	if (type->text == NULL)
		return NULL;

	return type;
}

static sgf_property_value_type *sgf_property_value_type_read_go_point(sgf_property_value_type *type, sgf_parser *parser)
{
	unsigned int i, multiplier;
	int c;

	type->number = 0;
	sgf_parser_skip_space(parser);
	multiplier = 1;
	for (i = 0; i < 2; i++) {
		c = sgf_parser_get_char(parser);
		if (!isalpha(c)) {
			sgf_parser_unget_char(parser, c);
			return NULL;
		}
		if (isupper(c))
			type->number += (c - 'A' + 26) * multiplier;
		else
			type->number += (c - 'a') * multiplier;
		multiplier *= 52;
	}
	sgf_parser_skip_space(parser);

	return type;
}

static sgf_property_value_type *sgf_property_value_type_read_unknown(sgf_property_value_type *type, sgf_parser *parser)
{
	static char buffer[128];
	char *p;
	size_t i, length;
	int c;

	type->unknown = NULL;
	length = 0;
	i = 0;
	c = sgf_parser_get_char(parser);
	for (;;) {
		if ((c == EOF) || (c == ']') || (i >= sizeof(buffer))) {
			p = (char *) realloc(type->unknown, length + i + 1);
			if (p == NULL) {
				sgf_parser_unget_char(parser, c);
				free(type->unknown);
				return NULL;
			}
			type->unknown = p;
			memcpy(type->unknown + length, buffer, i);
			length += i;
			i = 0;
			if ((c == EOF) || (c == ']')) {
				sgf_parser_unget_char(parser, c);
				break;
			}
		} else if (c == '\\') {
			c = sgf_parser_get_char(parser);
			if (c == EOF)
				continue;
			buffer[i++] = c;
			c = sgf_parser_get_char(parser);
		} else if (c == '\0') {
			/* Skip null bytes */
			c = sgf_parser_get_char(parser);
		} else {
			buffer[i++] = c;
			c = sgf_parser_get_char(parser);
		}
	}
	type->unknown[length] = '\0';

	return type;
}

static char *sgf_property_value_type_compose_number(const sgf_property_value_type *type, unsigned int column)
{
	char *result;
	size_t length;

	if (type->number == 0)
		length = 1;
	else
		length = (size_t) log10(abs(type->number)) + 1;
	if (type->number < 0)
		length++;
	result = (char *) malloc(length + 1);
	if (result == NULL)
		return NULL;
	sprintf(result, "%*d", length, type->number);

	return result;
}

static char *sgf_property_value_type_compose_real(const sgf_property_value_type *type, unsigned int column)
{
	char *result, *p;
	size_t length;
	double real;
	int num_chars;

	modf(type->real * 10000, &real);
	if (real == 0.0)
		length = 1;
	else
		length = (size_t) log10(fabs(real)) + 1;
	real /= 10000;
	if (real < 0.0)
		length++;
	length++;
	result = (char *) malloc(length + 1);
	if (result == NULL)
		return NULL;
	num_chars = sprintf(result, "%g", real);
	if (length != num_chars + 1) {
		p = (char *) realloc(result, num_chars + 1);
		if (p == NULL) {
			free(result);
			return NULL;
		}
		result = p;
	}

	return result;
}

static char *sgf_property_value_type_compose_binary(const sgf_property_value_type *type, unsigned int column)
{
	char *result;
	char c;

	if (type->binary == 1)
		c = '1';
	else if (type->binary == 2)
		c = '2';
	else
		return NULL;
	result = (char *) malloc(2);
	if (result == NULL)
		return NULL;
	result[0] = c;
	result[1] = '\0';

	return result;
}

static char *sgf_property_value_type_compose_color(const sgf_property_value_type *type, unsigned int column)
{
	char *result;
	char c;

	if (type->binary == 1)
		c = 'B';
	else if (type->binary == 2)
		c = 'W';
	else
		return NULL;
	result = (char *) malloc(2);
	if (result == NULL)
		return NULL;
	result[0] = c;
	result[1] = '\0';

	return result;
}

static char *strlencat(char **s, size_t *length, const char *t)
{
	size_t new_length;
	char *p;

	new_length = *length + strlen(t);
	p = (char *) realloc(*s, new_length + 1);
	if (p == NULL) {
		free(*s);
		*s = NULL;
		return NULL;
	}
	*s = p;
	strcpy(*s + *length, t);
	*length = new_length;

	return *s;
}

static char *strlencatchar(char **s, size_t *length, char c)
{
	char *p;

	p = (char *) realloc(*s, *length + 2);
	if (p == NULL) {
		free(*s);
		*s = NULL;
		return NULL;
	}
	*s = p;
	(*s)[*length] = c;
	(*length)++;
	(*s)[*length] = '\0';

	return *s;
}

/* If we want to get really sick in the future, we can accept an argument
   specifying how many characters will follow the composed text. Right now it
   is always 1: ']' (it could be ':' too, which breaks because it's not followed
   by a newline). Then we could put closing parentheses on the same line and
   other stupid stuff. We'll need to do stuff wherever there's a test for the
   end of text; i.e., *s == '\0'. */
static char *compose_text(const char *s, const char *escaped_chars, unsigned int column, int break_style)
{
	const char *p;
	char *result;
	unsigned int length, word_length, max_line_length;

	result = NULL;
	length = 0;
	max_line_length = SGF_FILE_MAX_LINE_LENGTH;
	if (break_style == SGF_FILE_BREAK_STYLE_TEXT)
		max_line_length--;
	while (*s != '\0') {
		while (*s != '\0') {
			if (strchr(escaped_chars, *s) != NULL) {
				if ((break_style != SGF_FILE_BREAK_STYLE_UNKNOWN) &&
				    (column >= max_line_length - 1)) {
					if (strlencat(&result, &length, SGF_SOFT_LINE_BREAK) == NULL)
						return NULL;
					column = 0;
				}
				if (strlencatchar(&result, &length, '\\') == NULL)
					return NULL;
				if (strlencatchar(&result, &length, *s) == NULL)
					return NULL;
				column += 2;
				s++;
			} else if (!isspace(*s)) {
				if ((break_style != SGF_FILE_BREAK_STYLE_UNKNOWN) &&
				    (column >= max_line_length)) {
					if (strlencat(&result, &length, SGF_SOFT_LINE_BREAK) == NULL)
						return NULL;
					column = 0;
				}
				if (strlencatchar(&result, &length, *s) == NULL)
					return NULL;
				column++;
				s++;
			} else {
				break;
			}
		}
		if (break_style == SGF_FILE_BREAK_STYLE_TEXT) {
			while (isspace(*s) && (strchr(escaped_chars, *s) == NULL)) {
				if (*s == '\n') {
					if (strlencatchar(&result, &length, '\n') == NULL)
						return NULL;
					column = 0;
					s++;
				} else if (column >= max_line_length) {
					if (strlencat(&result, &length, SGF_SOFT_LINE_BREAK) == NULL)
						return NULL;
					column = 0;
				} else {
					if (strlencatchar(&result, &length, *s) == NULL)
						return NULL;
					column++;
					s++;
				}
			}
			/* Get the length of the word following this whitespace */
			word_length = 0;
			for (p = s; *p != '\0'; p++) {
				if (strchr(escaped_chars, *p) != NULL)
					word_length += 2;
				else if (!isspace(*p))
					word_length++;
				else
					break;
			}
			if ((column + word_length >= max_line_length) &&
			    !((column + word_length == max_line_length) &&
			      (*p == '\0'))) {
				if (strlencat(&result, &length, SGF_SOFT_LINE_BREAK) == NULL)
					return NULL;
				column = 0;
			}
		} else if (break_style == SGF_FILE_BREAK_STYLE_SIMPLETEXT) {
			if (isspace(*s) && (strchr(escaped_chars, *s) == NULL)) {
				while (isspace(*(s + 1)) &&
				       (strchr(escaped_chars, *(s + 1)) == NULL)) {
					if (column >= max_line_length) {
						if (strlencatchar(&result, &length, '\n') == NULL)
							return NULL;
						column = 0;
					} else {
						if (strlencatchar(&result, &length, ' ') == NULL)
							return NULL;
						column++;
					}
					s++;
				}
				column++;
				s++;
				/* Get the length of the word following this whitespace */
				word_length = 0;
				for (p = s; *p != '\0'; p++) {
					if (strchr(escaped_chars, *p) != NULL)
						word_length += 2;
					else if (!isspace(*p))
						word_length++;
					else
						break;
				}
				if ((column + word_length >= max_line_length) &&
				    !((column + word_length == max_line_length) &&
				      isspace(*p))) {
					if (strlencatchar(&result, &length, '\n') == NULL)
						return NULL;
					column = 0;
				} else {
					if (strlencatchar(&result, &length, ' ') == NULL)
						return NULL;
				}
			}
		} else {
			while (isspace(*s) && (strchr(escaped_chars, *s) == NULL)) {
				if (strlencatchar(&result, &length, *s) == NULL)
					return NULL;
				column++;
				s++;
			}
		}
	}

	return result;
}

static char *sgf_property_value_type_compose_simpletext(const sgf_property_value_type *type, unsigned int column)
{
	return compose_text(type->simpletext, ":\\]", column, SGF_FILE_BREAK_STYLE_SIMPLETEXT);
}

static char *sgf_property_value_type_compose_text(const sgf_property_value_type *type, unsigned int column)
{
	return compose_text(type->simpletext, ":\\]", column, SGF_FILE_BREAK_STYLE_TEXT);
}

static char *sgf_property_value_type_compose_go_point(const sgf_property_value_type *type, unsigned int column)
{
	char *result;
	unsigned int x, y;

	y = type->number / 52;
	if (y >= 52)
		return NULL;
	x = type->number % 52;
	result = (char *) malloc(3);
	if (result == NULL)
		return NULL;
	if (x < 26)
		result[0] = 'a' + x;
	else
		result[0] = 'A' + x - 26;
	if (y < 26)
		result[1] = 'a' + y;
	else
		result[1] = 'A' + y - 26;
	result[2] = '\0';

	return result;
}

static char *sgf_property_value_type_compose_unknown(const sgf_property_value_type *type, unsigned int column)
{
	static char buffer[128];
	const char *s;
	char *result, *p;
	size_t i, j, length;

	result = NULL;
	length = 0;
	i = 0;
	s = type->unknown;
	for (;;) {
		if ((*s == '\0') || (i >= sizeof(buffer) - 1)) {
			p = (char *) realloc(result, length + i + 1);
			if (p == NULL) {
				free(result);
				return NULL;
			}
			result = p;
			memcpy(result + length, buffer, i);
			length += i;
			i = 0;
			if (*s == '\0')
				break;
		} else if (*s == ']') {
			buffer[i++] = '\\';
			buffer[i++] = *s;
			s++;
		} else {
			p = strchr(s, ']');
			if (p == NULL)
				j = strlen(s);
			else
				j = p - s;
			if (j > sizeof(buffer) - i)
				j = sizeof(buffer) - i;
			memmove(buffer + i, s, j);
			i += j;
			s += j;
		}
	}
	result[length] = '\0';

	return result;
}

void sgf_property_value_free(sgf_property_value *value)
{
	if (!(value->flags & SGF_PROPERTY_VALUE_NONE)) {
	    if (value->handlers[0]->free != NULL)
			value->handlers[0]->free(&value->types[0]);
		if (!(value->flags & SGF_PROPERTY_VALUE_SINGLE)) {
		    if (value->handlers[1]->free != NULL)
				value->handlers[1]->free(&value->types[1]);
		}
	}

	return;
}

sgf_property_value *sgf_property_value_duplicate(sgf_property_value *a, const sgf_property_value *b)
{
	a->flags = b->flags;
	a->handlers = b->handlers;
	if (!(a->flags & SGF_PROPERTY_VALUE_NONE)) {
	    if (a->handlers[0]->duplicate != NULL) {
	    	if (a->handlers[0]->duplicate(&a->types[0], &b->types[0]) == NULL)
				return NULL;
		} else {
			a->types[0] = b->types[0];
		}
		if (!(a->flags & SGF_PROPERTY_VALUE_SINGLE)) {
	    	if (a->handlers[1]->duplicate != NULL) {
	    		if (a->handlers[1]->duplicate(&a->types[1], &b->types[1]) == NULL) {
	    			if (a->handlers[0]->free != NULL)
						a->handlers[0]->free(&a->types[0]);
					return NULL;
				}
			} else {
				a->types[1] = b->types[1];
			}
		}
	}

	return a;
}

static int sgf_property_value_cmp(const sgf_property_value *a, const sgf_property_value *b)
{
	int diff;
	diff = (int) (a->flags & SGF_PROPERTY_VALUE_MASK) - (b->flags & SGF_PROPERTY_VALUE_MASK);
	if (diff != 0)
		return diff;
	if (!(a->flags & SGF_PROPERTY_VALUE_NONE)) {
	    if (a->handlers[0]->cmp != NULL) {
			diff = a->handlers[0]->cmp(&a->types[0], &b->types[0]);
			if (diff != 0)
				return diff;
		}
		if (!(a->flags & SGF_PROPERTY_VALUE_SINGLE)) {
		    if (a->handlers[1]->cmp != NULL) {
				diff = a->handlers[1]->cmp(&a->types[1], &b->types[1]);
				if (diff != 0)
					return diff;
			}
		}
	}

	return 0;
}

static sgf_property_value *sgf_property_value_read(sgf_property_value *value, sgf_parser *parser)
{
	unsigned int flags;
	int c;

	/* On entering this function, value->flags indicates what kinds of
	   values are acceptable (because some property values can be simple
	   or composed, for example). value->flags is set to reflect what was
	   actually read. */
	flags = value->flags;
	value->flags = 0;
	sgf_parser_skip_space(parser);
	c = sgf_parser_get_char(parser);
	if (c != '[') {
		sgf_parser_unget_char(parser, c);
		return NULL;
	}
	if (flags & SGF_PROPERTY_VALUE_NONE) {
		value->flags = SGF_PROPERTY_VALUE_NONE;
		sgf_parser_skip_space(parser);
		c = sgf_parser_get_char(parser);
		if (c == ']')
			return value;
		else
			sgf_parser_unget_char(parser, c);
	}
	if ((flags & SGF_PROPERTY_VALUE_SINGLE) ||
	    (flags & SGF_PROPERTY_VALUE_COMPOSED)) {
		if ((value->handlers[0]->read == NULL) ||
		    (value->handlers[0]->read(&value->types[0], parser) == NULL))
			return NULL;
		value->flags = SGF_PROPERTY_VALUE_SINGLE;
		sgf_parser_skip_space(parser);
		c = sgf_parser_get_char(parser);
		if (flags & SGF_PROPERTY_VALUE_SINGLE) {
			if (c == ']')
				return value;
		}
		if (flags & SGF_PROPERTY_VALUE_COMPOSED) {
			if (c == ':') {
				if ((value->handlers[1]->read == NULL) ||
				    (value->handlers[1]->read(&value->types[1], parser) == NULL)) {
					sgf_property_value_free(value);
					return NULL;
				}
				value->flags = SGF_PROPERTY_VALUE_COMPOSED;
				sgf_parser_skip_space(parser);
				c = sgf_parser_get_char(parser);
				if (c == ']')
					return value;
				else
					sgf_parser_unget_char(parser, c);
			} else {
				sgf_parser_unget_char(parser, c);
			}
		}
	}
	/* Getting to here is a failure */
	sgf_property_value_free(value);

	return NULL;
}

static char *sgf_property_value_compose(const sgf_property_value *value, unsigned int column)
{
	char *result, *p;
	size_t length;
	unsigned int tmp;

	length = 0;
	result = NULL;
	if (strlencat(&result, &length, "[") == NULL)
		return NULL;
	column++;
	if ((value->flags & SGF_PROPERTY_VALUE_SINGLE) ||
	    (value->flags & SGF_PROPERTY_VALUE_COMPOSED)) {
		if (value->handlers[0]->compose != NULL) {
			p = value->handlers[0]->compose(&value->types[0], column);
			if (p == NULL) {
				free(result);
				return NULL;
			}
			strlencat(&result, &length, p);
			free(p);
			if (result == NULL)
				return NULL;
		}
		if (value->flags & SGF_PROPERTY_VALUE_COMPOSED) {
			tmp = 0;
			for (p = result + length - 1; (p >= result) && (*p != '\n'); p--)
				tmp++;
			if (p < result)
				column += tmp;
			else
				column = tmp;
			if (strlencat(&result, &length, ":") == NULL)
				return NULL;
			if (value->handlers[1]->compose != NULL) {
				p = value->handlers[1]->compose(&value->types[1], column);
				if (p == NULL) {
					free(result);
					return NULL;
				}
				strlencat(&result, &length, p);
				free(p);
				if (result == NULL)
					return NULL;
			}
		}
	}
	if (strlencat(&result, &length, "]") == NULL)
		return NULL;

	return result;
}

sgf_property *sgf_property_init(sgf_property *property)
{
	property->name = NULL;
	property->type = NULL;
	array_init((void **) &property->values, &property->num_values);

	return property;
}

void sgf_property_free(sgf_property *property)
{
	free(property->name);
	array_free((void **) &property->values, &property->num_values, sizeof(sgf_property_value), (void (*)(void *)) sgf_property_value_free);

	return;
}

sgf_property *sgf_property_duplicate(sgf_property *a, const sgf_property *b)
{
	a->name = (char *) malloc(strlen(b->name) + 1);
	if (a->name == NULL)
		return NULL;
	strcpy(a->name, b->name);
	if (array_duplicate((void **) &a->values, &a->num_values, sizeof(sgf_property_value), b->values, b->num_values, (void *(*)(void *, const void *)) sgf_property_value_duplicate) == NULL) {
		sgf_property_free(a);
		return NULL;
	}
	a->type = b->type;

	return a;
}

sgf_property *sgf_property_add_value(sgf_property *property, const sgf_property_value *value)
{
	if (array_insert_sorted((void **) &property->values, &property->num_values, sizeof(sgf_property_value), value, (int (*)(const void *, const void *)) sgf_property_value_cmp) == NULL)
		return NULL;

	return property;
}

sgf_property *sgf_property_remove_value(sgf_property *property, const sgf_property_value *value)
{
	if (array_remove_sorted((void **) &property->values, &property->num_values, sizeof(sgf_property_value), value, (int (*)(const void *, const void *)) sgf_property_value_cmp) == NULL)
		return NULL;

	return property;
}

static int sgf_property_cmp(const sgf_property *a, const sgf_property *b)
{
	int diff;

	diff = a->type->type - b->type->type;
	if (diff != 0)
		return diff;
	diff = (int) a->type->priority - (int) b->type->priority;
	if (diff != 0)
		return diff;
	diff = strcmp(a->name, b->name);

	return diff;
}

static int sgf_property_name_cmp(const char *name, const sgf_property *property)
{
	return strcmp(name, property->name);
}

static sgf_property *sgf_property_read(sgf_property *property, sgf_parser *parser)
{
	static char buffer[10];
	char *p;
	size_t i, length;
	int c;
	sgf_property_value value;

	if (sgf_property_init(property) == NULL)
		return NULL;
	sgf_parser_skip_space(parser);
	length = 0;
	i = 0;
	c = sgf_parser_get_char(parser);
	/* Make this read FF[3] property names if appropriate */
	for (;;) {
		if ((c == EOF) || !isupper(c) || (i >= sizeof(buffer))) {
			p = (char *) realloc(property->name, length + i + 1);
			if (p == NULL) {
				sgf_parser_unget_char(parser, c);
				free(property->name);
				return NULL;
			}
			property->name = p;
			memcpy(property->name + length, buffer, i);
			length += i;
			i = 0;
			if ((c == EOF) || !isupper(c)) {
				sgf_parser_unget_char(parser, c);
				break;
			}
		} else if (c == '\0') {
			/* Skip null bytes */
			c = sgf_parser_get_char(parser);
		} else {
			buffer[i++] = c;
			c = sgf_parser_get_char(parser);
		}
	}
	property->name[length] = '\0';
	sgf_parser_skip_space(parser);
	property->type = sgf_parser_search_property_types(parser, property->name);
	/* sgf_parser_search_property_types never returns NULL */
	value.flags = property->type->flags & SGF_PROPERTY_VALUE_MASK;
	value.handlers = property->type->handlers;
	/* If this property can have an elist of values, pretend that the
	   first value can be None */
	if (property->type->flags & SGF_PROPERTY_ELIST)
		value.flags |= SGF_PROPERTY_VALUE_NONE;
	if (sgf_property_value_read(&value, parser) == NULL) {
		sgf_property_free(property);
		return NULL;
	}
	if ((property->type->flags & SGF_PROPERTY_ELIST) &&
	    !(property->type->flags & SGF_PROPERTY_VALUE_NONE) &&
	    (value.flags & SGF_PROPERTY_VALUE_NONE)) {
		/* This is an empty elist ("[]"). Return without adding any
		   property values. */
		return property;
	}
	if (sgf_property_add_value(property, &value) == NULL) {
		sgf_property_free(property);
		sgf_property_value_free(&value);
		return NULL;
	}
	/* Some properties (CA, FF and GM) have to be handled specially */
	/* It might be better to move this to sgf_game_read or somesuch */
	if (strcmp(property->name, "CA") == 0) {
		p = (char *) realloc(parser->charset, strlen(value.types[0].simpletext) + 1);
		if (p != NULL) {
			strcpy(p, value.types[0].simpletext);
			parser->charset = p;
		}
	} else if (strcmp(property->name, "FF") == 0) {
		parser->format = value.types[0].number;
	} else if (strcmp(property->name, "GM") == 0) {
		parser->game = value.types[0].number;
	}
	if ((property->type->flags & SGF_PROPERTY_LIST) ||
	    (property->type->flags & SGF_PROPERTY_ELIST)) {
		for (;;) {
			sgf_parser_skip_space(parser);
			c = sgf_parser_get_char(parser);
			sgf_parser_unget_char(parser, c);
			if (c != '[')
				break;
			value.flags = property->type->flags & SGF_PROPERTY_VALUE_MASK;
			if (sgf_property_value_read(&value, parser) == NULL) {
				sgf_property_free(property);
				return NULL;
			}
			if (sgf_property_add_value(property, &value) == NULL) {
				sgf_property_free(property);
				sgf_property_value_free(&value);
				return NULL;
			}
		}
	}

	return property;
}

sgf_node *sgf_node_init(sgf_node *node)
{
	array_init((void **) &node->properties, &node->num_properties);
	node->next = NULL;
	node->variation = NULL;

	return node;
}

void sgf_node_free(sgf_node *node)
{
	array_free((void **) &node->properties, &node->num_properties, sizeof(sgf_property), (void (*)(void *)) sgf_property_free);

	return;
}

/* property is not declared const because it can be modified if another
   property with the same name is found */
sgf_node *sgf_node_add_property(sgf_node *node, sgf_property *property)
{
	sgf_property *p;
	sgf_property_value value;
	unsigned int i;

	if ((property->type->flags & SGF_PROPERTY_LIST) ||
	    (property->type->flags & SGF_PROPERTY_ELIST)) {
		/* If the property can be a list, see if a property with the same
		   name exists in the node and merge the values */
		p = sgf_node_search_properties(node, property);
		if (p != NULL) {
			for (i = 0; i < property->num_values; i++) {
				if (sgf_property_value_duplicate(&value, &property->values[i]) == NULL)
					return NULL;
				if (sgf_property_add_value(p, &value) == NULL)
					return NULL;
			}
			sgf_property_free(property);
			return node;
		}
	}
	if (array_insert_sorted((void **) &node->properties, &node->num_properties, sizeof(sgf_property), property, (int (*)(const void *, const void *)) sgf_property_cmp) == NULL)
		return NULL;

	return node;
}

sgf_node *sgf_node_remove_property(sgf_node *node, const sgf_property *property)
{
	if (array_remove((void **) &node->properties, &node->num_properties, sizeof(sgf_property), property, (int (*)(const void *, const void *)) sgf_property_cmp) == NULL)
		return NULL;

	return node;
}

sgf_property *sgf_node_search_properties(const sgf_node *node, const sgf_property *property)
{
	return (sgf_property *) array_search_sorted((const void **) &node->properties, &node->num_properties, sizeof(sgf_property), property, (int (*)(const void *, const void *)) sgf_property_cmp);
}

sgf_property *sgf_node_search_properties_by_name(const sgf_node *node, const char *name)
{
	return (sgf_property *) array_search((const void **) &node->properties, &node->num_properties, sizeof(sgf_property), name, (int (*)(const void *, const void *)) sgf_property_name_cmp);
}

void sgf_gametree_free(sgf_node *node)
{
	sgf_node *next;

	for (; node != NULL; node = next) {
		sgf_gametree_free(node->variation);
		next = node->next;
		sgf_node_free(node);
		free(node);
	}

	return;
}

sgf_game *sgf_game_init(sgf_game *game)
{
	*game = NULL;

	return game;
}

void sgf_game_free(sgf_game *game)
{
	sgf_gametree_free(*game);

	return;
}

static sgf_game_read_helper *sgf_game_read_helper_init(sgf_game_read_helper *helper)
{
	helper->root_node = NULL;
	helper->current_node = NULL;
	helper->collection = NULL;
	array_init((void **) &helper->stack, &helper->stack_size);

	return helper;
}

static void sgf_game_read_helper_free(sgf_game_read_helper *helper)
{
	array_free((void **) &helper->stack, &helper->stack_size, sizeof(sgf_node *), NULL);

	return;
}

static sgf_game_read_helper *sgf_game_read_helper_push(sgf_game_read_helper *helper, sgf_node *node)
{
	if (array_insert((void **) &helper->stack, &helper->stack_size, sizeof(sgf_node *), &node, NULL) == NULL)
		return NULL;

	return helper;
}

static sgf_game_read_helper *sgf_game_read_helper_pop(sgf_game_read_helper *helper, sgf_node **node)
{
	if (helper->stack_size == 0)
		return NULL;
	if (node != NULL)
		*node = helper->stack[helper->stack_size - 1];
	if (array_remove_index_keep_order((void **) &helper->stack, &helper->stack_size, sizeof(sgf_node *), helper->stack_size - 1) == NULL)
		return NULL;

	return helper;
}

static void sgf_game_read_game_start(sgf_parser *parser)
{
	sgf_game_read_helper *helper;

	helper = (sgf_game_read_helper *) parser->user_data;
	helper->root_node = NULL;

	return;
}

static void sgf_game_read_game_end(sgf_parser *parser)
{
	sgf_parser_stop(parser);

	return;
}

static void sgf_game_read_gametree_start(sgf_parser *parser)
{
	sgf_game_read_helper *helper;

	helper = (sgf_game_read_helper *) parser->user_data;
	if (sgf_game_read_helper_push(helper, helper->current_node) == NULL) {
		sgf_parser_stop(parser);
		return;
	}

	return;
}

static void sgf_game_read_gametree_end(sgf_parser *parser)
{
	sgf_game_read_helper *helper;

	helper = (sgf_game_read_helper *) parser->user_data;
	if (sgf_game_read_helper_pop(helper, &helper->current_node) == NULL) {
		sgf_parser_stop(parser);
		return;
	}

	return;
}

static void sgf_game_read_node_start(sgf_parser *parser)
{
	sgf_game_read_helper *helper;
	sgf_node *node;

	helper = (sgf_game_read_helper *) parser->user_data;
	node = (sgf_node *) malloc(sizeof(sgf_node));
	if (node == NULL) {
		sgf_parser_stop(parser);
		return;
	}
	if (sgf_node_init(node) == NULL) {
		free(node);
		sgf_parser_stop(parser);
		return;
	}
	if (helper->current_node != NULL) {
		if (helper->current_node->next == NULL) {
			helper->current_node->next = node;
		} else {
			for (helper->current_node = helper->current_node->next; helper->current_node->variation != NULL; helper->current_node = helper->current_node->variation)
				;
			helper->current_node->variation = node;
		}
	}
	helper->current_node = node;
	if (helper->root_node == NULL)
		helper->root_node = helper->current_node;

	return;
}

static void sgf_game_read_property(sgf_parser *parser, const sgf_property *property)
{
	sgf_game_read_helper *helper;
	sgf_property *p;

	helper = (sgf_game_read_helper *) parser->user_data;
	p = (sgf_property *) malloc(sizeof(sgf_property));
	if (p == NULL) {
		sgf_parser_stop(parser);
		return;
	}
	if (sgf_property_duplicate(p, property) == NULL) {
		free(p);
		sgf_parser_stop(parser);
		return;
	}
	if (sgf_node_add_property(helper->current_node, p) == NULL) {
		sgf_property_free(p);
		free(p);
		sgf_parser_stop(parser);
		return;
	}

	return;
}

sgf_game *sgf_game_read(sgf_game *game, sgf_parser *parser)
{
	sgf_game_read_helper helper;

	if (sgf_game_init(game) == NULL)
		return NULL;
	sgf_parser_set_game_handlers(parser, sgf_game_read_game_start, sgf_game_read_game_end);
	sgf_parser_set_gametree_handlers(parser, sgf_game_read_gametree_start, sgf_game_read_gametree_end);
	sgf_parser_set_node_handlers(parser, sgf_game_read_node_start, NULL);
	sgf_parser_set_property_handler(parser, sgf_game_read_property);
	if (sgf_game_read_helper_init(&helper) == NULL)
		return NULL;
	sgf_parser_set_user_data(parser, (void *) &helper);
	/* We should loop here, looking for a valid game */
	if (sgf_parser_parse(parser) == NULL) {
		sgf_game_read_helper_free(&helper);
		sgf_game_free(game);
		return NULL;
	}
	*game = helper.root_node;
	sgf_game_read_helper_free(&helper);

	return game;
}

static void sgf_game_read_shallow_node_end(sgf_parser *parser)
{
	/* We ignore every node but the first (the root) */
	sgf_parser_set_gametree_handlers(parser, NULL, NULL);
	sgf_parser_set_node_handlers(parser, NULL, NULL);
	sgf_parser_set_property_handler(parser, NULL);

	return;
}

sgf_game *sgf_game_read_shallow(sgf_game *game, sgf_parser *parser)
{
	sgf_game_read_helper helper;

	if (sgf_game_init(game) == NULL)
		return NULL;
	sgf_parser_set_game_handlers(parser, sgf_game_read_game_start, sgf_game_read_game_end);
	sgf_parser_set_gametree_handlers(parser, sgf_game_read_gametree_start, sgf_game_read_gametree_end);
	sgf_parser_set_node_handlers(parser, sgf_game_read_node_start, sgf_game_read_shallow_node_end);
	sgf_parser_set_property_handler(parser, sgf_game_read_property);
	if (sgf_game_read_helper_init(&helper) == NULL)
		return NULL;
	sgf_parser_set_user_data(parser, (void *) &helper);
	/* We should loop here, looking for a valid game */
	if (sgf_parser_parse(parser) == NULL) {
		sgf_game_read_helper_free(&helper);
		sgf_game_free(game);
		return NULL;
	}
	*game = helper.root_node;
	sgf_game_read_helper_free(&helper);

	return game;
}

sgf_game *sgf_game_write(sgf_game *game, sgf_parser *parser)
{
	if (sgf_gametree_write(*game, parser) == NULL)
		return NULL;
	sgf_parser_put_char(parser, '\n');

	return game;
}

static sgf_node *sgf_gametree_write(sgf_node *node, sgf_parser *parser)
{
	sgf_node *root;
	char *value, *p;
	unsigned int i, j, length;

	root = node;
	if (parser->column >= SGF_FILE_MAX_LINE_LENGTH)
		sgf_parser_put_char(parser, '\n');
	sgf_parser_put_char(parser, '(');
	for (;;) {
		for (i = 0; i < node->num_properties; i++) {
			if (node->properties[i].num_values == 0) {
				/* For value types of None and empty elists */
				value = "[]";
				length = strlen(value) + strlen(node->properties[i].name);
				if (i == 0)
					length += 1;
				if (parser->column + length > SGF_FILE_MAX_LINE_LENGTH)
					sgf_parser_put_char(parser, '\n');
				if (i == 0)
					sgf_parser_put_char(parser, ';');
				sgf_parser_write_string(parser, node->properties[i].name);
				sgf_parser_write_string(parser, value);
			} else {
				for (j = 0; j < node->properties[i].num_values; j++) {
					length = 0;
					if (j == 0) {
						if (i == 0)
							length += 1;
						length += strlen(node->properties[i].name);
					}
					value = sgf_property_value_compose(&node->properties[i].values[j], length);
					if (value == NULL)
						return NULL;
					p = strchr(value, '\n');
					if (p == NULL)
						length += strlen(value);
					else
						length += p - value;
					if (parser->column + length > SGF_FILE_MAX_LINE_LENGTH)
						sgf_parser_put_char(parser, '\n');
					if (j == 0) {
						if (i == 0)
							sgf_parser_put_char(parser, ';');
						sgf_parser_write_string(parser, node->properties[i].name);
					}
					sgf_parser_write_string(parser, value);
					free(value);
				}
			}
			/* Put each game-info property on a line by itself */
			/* This is still broken when the very first turn after the root has
			   variations. */
			if ((node->properties[i].type->type == SGF_PROPERTY_TYPE_GAME_INFO) ||
			    ((i + 1 < node->num_properties) &&
			     (node->properties[i + 1].type->type == SGF_PROPERTY_TYPE_GAME_INFO)))
				sgf_parser_put_char(parser, '\n');
		}
		node = node->next;
		if (node == NULL)
			break;
		if (node->variation != NULL) {
			for (; node != NULL; node = node->variation) {
				sgf_parser_put_char(parser, '\n');
				if (sgf_gametree_write(node, parser) == NULL)
					return NULL;
			}
			break;
		}
	}
	if (parser->column >= SGF_FILE_MAX_LINE_LENGTH)
		sgf_parser_put_char(parser, '\n');
	sgf_parser_put_char(parser, ')');

	return root;
}

sgf_collection *sgf_collection_init(sgf_collection *collection)
{
	array_init((void **) &collection->games, &collection->num_games);

	return collection;
}

void sgf_collection_free(sgf_collection *collection)
{
	array_free((void **) &collection->games, &collection->num_games, sizeof(sgf_node), (void (*)(void *)) sgf_game_free);

	return;
}

static sgf_collection *sgf_collection_add_game(sgf_collection *collection, const sgf_game *game)
{
	return array_insert((void **) &collection->games, &collection->num_games, sizeof(sgf_game), game, NULL);
}

static void sgf_collection_read_game_end(sgf_parser *parser)
{
	sgf_game_read_helper *helper;

	helper = (sgf_game_read_helper *) parser->user_data;
	if (sgf_collection_add_game(helper->collection, &helper->root_node) == NULL) {
		sgf_collection_free(helper->collection);
		sgf_parser_stop(parser);
		return;
	}

	return;
}

sgf_collection *sgf_collection_read(sgf_collection *collection, sgf_parser *parser)
{
	sgf_game_read_helper helper;

	if (sgf_collection_init(collection) == NULL)
		return NULL;
	sgf_parser_set_game_handlers(parser, sgf_game_read_game_start, sgf_collection_read_game_end);
	sgf_parser_set_gametree_handlers(parser, sgf_game_read_gametree_start, sgf_game_read_gametree_end);
	sgf_parser_set_node_handlers(parser, sgf_game_read_node_start, NULL);
	sgf_parser_set_property_handler(parser, sgf_game_read_property);
	if (sgf_game_read_helper_init(&helper) == NULL)
		return NULL;
	helper.collection = collection;
	sgf_parser_set_user_data(parser, (void *) &helper);
	if (sgf_parser_parse(parser) == NULL) {
		sgf_game_read_helper_free(&helper);
		sgf_collection_free(collection);
		return NULL;
	}
	sgf_game_read_helper_free(&helper);

	return collection;
}

static void sgf_collection_read_shallow_game_end(sgf_parser *parser)
{
	sgf_collection_read_game_end(parser);
	sgf_parser_set_gametree_handlers(parser, sgf_game_read_gametree_start, sgf_game_read_gametree_end);
	sgf_parser_set_node_handlers(parser, sgf_game_read_node_start, NULL);
	sgf_parser_set_property_handler(parser, sgf_game_read_property);

	return;
}

sgf_collection *sgf_collection_read_shallow(sgf_collection *collection, sgf_parser *parser)
{
	sgf_game_read_helper helper;

	if (sgf_collection_init(collection) == NULL)
		return NULL;
	sgf_parser_set_game_handlers(parser, sgf_game_read_game_start, sgf_collection_read_shallow_game_end);
	sgf_parser_set_gametree_handlers(parser, sgf_game_read_gametree_start, sgf_game_read_gametree_end);
	sgf_parser_set_node_handlers(parser, sgf_game_read_node_start, sgf_game_read_shallow_node_end);
	sgf_parser_set_property_handler(parser, sgf_game_read_property);
	if (sgf_game_read_helper_init(&helper) == NULL)
		return NULL;
	helper.collection = collection;
	sgf_parser_set_user_data(parser, (void *) &helper);
	if (sgf_parser_parse(parser) == NULL) {
		sgf_game_read_helper_free(&helper);
		sgf_collection_free(collection);
		return NULL;
	}
	sgf_game_read_helper_free(&helper);

	return collection;
}

sgf_collection *sgf_collection_write(sgf_collection *collection, sgf_parser *parser)
{
	unsigned int i;

	for (i = 0; i < collection->num_games; i++) {
		if (sgf_game_write(&collection->games[i], parser) == NULL)
			return NULL;
	}

	return collection;
}

sgf_parser *sgf_parser_init(sgf_parser *parser)
{
	parser->fp = NULL;
	parser->game_start = NULL;
	parser->game_end = NULL;
	parser->gametree_start = NULL;
	parser->gametree_end = NULL;
	parser->node_start = NULL;
	parser->node_end = NULL;
	parser->property = NULL;
	parser->user_data = NULL;
	array_init((void **) &parser->property_types, &parser->num_property_types);
	parser->is_running = 0;
	parser->line = 0;
	parser->column = 0;
	parser->prev_column = 0;
	parser->charset = NULL;
	parser->format = 1;
	parser->game = SGF_GAME_GO;

	return parser;
}

void sgf_parser_free(sgf_parser *parser)
{
	if (parser->fp != NULL)
		fclose(parser->fp);
	array_free((void **) &parser->property_types, &parser->num_property_types, sizeof(sgf_property_type), NULL);
	free(parser->charset);

	return;
}

sgf_parser *sgf_parser_open_file(sgf_parser *parser, const char *filename, const char *mode)
{
	if (parser->fp != NULL) {
		if (fclose(parser->fp) == EOF)
			return NULL;
	}
	parser->fp = fopen(filename, mode);
	if (parser->fp == NULL)
		return NULL;
	parser->line = 1;
	parser->column = 1;
	parser->prev_column = 0;

	return parser;
}

void sgf_parser_set_game_handlers(sgf_parser *parser, sgf_parser_handler start, sgf_parser_handler end)
{
	parser->game_start = start;
	parser->game_end = end;

	return;
}

void sgf_parser_set_gametree_handlers(sgf_parser *parser, sgf_parser_handler start, sgf_parser_handler end)
{
	parser->gametree_start = start;
	parser->gametree_end = end;

	return;
}

void sgf_parser_set_node_handlers(sgf_parser *parser, sgf_parser_handler start, sgf_parser_handler end)
{
	parser->node_start = start;
	parser->node_end = end;

	return;
}

void sgf_parser_set_property_handler(sgf_parser *parser, sgf_parser_property_handler f)
{
	parser->property = f;

	return;
}

void sgf_parser_set_user_data(sgf_parser *parser, void *user_data)
{
	parser->user_data = user_data;

	return;
}

sgf_parser *sgf_parser_add_property_type(sgf_parser *parser, const sgf_property_type *type)
{
	if (array_insert_sorted((void **) &parser->property_types, &parser->num_property_types, sizeof(sgf_property_type), type, (int (*)(const void *, const void *)) sgf_property_type_name_cmp) == NULL)
		return NULL;

	return parser;
}

sgf_parser *sgf_parser_remove_property_type(sgf_parser *parser, const sgf_property_type *type)
{
	if (array_remove_sorted((void **) &parser->property_types, &parser->num_property_types, sizeof(sgf_property_type), type, (int (*)(const void *, const void *)) sgf_property_type_name_cmp) == NULL)
		return NULL;

	return parser;
}

sgf_parser *sgf_parser_parse(sgf_parser *parser)
{
	int c;

	if (parser->fp == NULL)
		return NULL;
	parser->is_running = 1;
	for (;;) {
		/* Search for a gametree */
		do {
			c = sgf_parser_get_char(parser);
		} while ((c != EOF) && (c != '('));
		if (c == EOF)
			return parser;
		sgf_parser_unget_char(parser, c);
		if (parser->game_start != NULL) {
			parser->game_start(parser);
			if (!parser->is_running)
				return parser;
		}
		if (sgf_parser_parse_gametree(parser) == NULL)
			return NULL;
		if (!parser->is_running)
			return parser;
		if (parser->game_end != NULL) {
			parser->game_end(parser);
			if (!parser->is_running)
				return parser;
		}
	}

	return parser;
}

static sgf_parser *sgf_parser_parse_gametree(sgf_parser *parser)
{
	int c, num_nodes;

	if (parser->fp == NULL)
		return NULL;
	sgf_parser_skip_space(parser);
	c = sgf_parser_get_char(parser);
	if (c != '(') {
		sgf_parser_unget_char(parser, c);
		return NULL;
	}
	if (parser->gametree_start != NULL) {
		parser->gametree_start(parser);
		if (!parser->is_running)
			return parser;
	}
	num_nodes = 0;
	for (;;) {
		sgf_parser_skip_space(parser);
		c = sgf_parser_get_char(parser);
		sgf_parser_unget_char(parser, c);
		/* Having no nodes is illegal */
		if (c != ';') {
			if (num_nodes == 0)
				return NULL;
			else
				break;
		}
		if (sgf_parser_parse_node(parser) == NULL)
			return NULL;
		if (!parser->is_running)
			return parser;
		num_nodes++;
	}
	for (;;) {
		sgf_parser_skip_space(parser);
		c = sgf_parser_get_char(parser);
		sgf_parser_unget_char(parser, c);
		if (c != '(')
			break;
		if (sgf_parser_parse_gametree(parser) == NULL)
			return NULL;
		if (!parser->is_running)
			return parser;
	}
	sgf_parser_skip_space(parser);
	c = sgf_parser_get_char(parser);
	if (c != ')') {
		sgf_parser_unget_char(parser, c);
		return NULL;
	}
	if (parser->gametree_end != NULL) {
		parser->gametree_end(parser);
		if (!parser->is_running)
			return parser;
	}

	return parser;
}

static sgf_parser *sgf_parser_parse_node(sgf_parser *parser)
{
	sgf_property property;
	int c;

	sgf_parser_skip_space(parser);
	c = sgf_parser_get_char(parser);
	if (c != ';') {
		sgf_parser_unget_char(parser, c);
		return NULL;
	}
	if (parser->node_start != NULL) {
		parser->node_start(parser);
		if (!parser->is_running)
			return parser;
	}
	for (;;) {
		sgf_parser_skip_space(parser);
		c = sgf_parser_get_char(parser);
		sgf_parser_unget_char(parser, c);
		if ((c == EOF) || (c == ';') || (c == '(') || (c == ')'))
			break;
		if (sgf_property_read(&property, parser) == NULL)
			return NULL;
		if (parser->property != NULL) {
			parser->property(parser, &property);
			if (!parser->is_running)
				return parser;
		}
		sgf_property_free(&property);
	}
	if (parser->node_end != NULL) {
		parser->node_end(parser);
		if (!parser->is_running)
			return parser;
	}

	return parser;
}

void sgf_parser_stop(sgf_parser *parser)
{
	parser->is_running = 0;

	return;
}

static int sgf_parser_get_char(sgf_parser *parser)
{
	int c;

	c = fgetc(parser->fp);
	putchar(c);
	/* Make this handle "\r\n" and "\n\r" linebreaks */
	if ((c == '\n') || (c == '\r')) {
		parser->line++;
		parser->prev_column = parser->column;
		parser->column = 0;
	} else {
		parser->column++;
	}

	return c;
}

static int sgf_parser_unget_char(sgf_parser *parser, int c)
{
	if (c != EOF)
		ungetc(c, parser->fp);
	if ((c == '\n') || (c == '\r')) {
		parser->line--;
		parser->column = parser->prev_column;
		parser->prev_column = 0;
	} else {
		parser->column--;
	}

	return c;
}

static int sgf_parser_put_char(sgf_parser *parser, int c)
{
	c = fputc(c, parser->fp);
	if (c == '\n') {
		parser->line++;
		parser->prev_column = parser->column;
		parser->column = 0;
	} else {
		parser->column++;
	}

	return c;
}

static int sgf_parser_write_string(sgf_parser *parser, const char *s)
{
	int i;

	for (i = 0; s[i] != '\0'; i++) {
		if (sgf_parser_put_char(parser, s[i]) == EOF)
			return EOF;
	}

	return i;
}

static void sgf_parser_skip_space(sgf_parser *parser)
{
	int c;

	do {
		c = sgf_parser_get_char(parser);
	} while (isspace(c));
	sgf_parser_unget_char(parser, c);

	return;
}

sgf_property *sgf_parser_property_set_name(const sgf_parser *parser, sgf_property *property, const char *name)
{
	char *p;

	p = realloc(property->name, strlen(name) + 1);
	if (p == NULL)
		return NULL;
	property->name = p;
	strcpy(property->name, name);
	property->type = sgf_parser_search_property_types(parser, property->name);

	return property;
}

static char *sgf_parser_read_text(sgf_parser *parser, const char *terminators, int simple)
{
	static char buffer[128];
	char *result, *p;
	size_t i, length;
	int c, next;

	result = NULL;
	length = 0;
	i = 0;
	c = sgf_parser_get_char(parser);
	for (;;) {
		if ((c == EOF) || (strchr(terminators, c) != NULL) || (i >= sizeof(buffer))) {
			p = (char *) realloc(result, length + i + 1);
			if (p == NULL) {
				sgf_parser_unget_char(parser, c);
				free(result);
				return NULL;
			}
			result = p;
			memcpy(result + length, buffer, i);
			length += i;
			i = 0;
			if ((c == EOF) || (strchr(terminators, c) != NULL)) {
				sgf_parser_unget_char(parser, c);
				break;
			}
		} else if (isspace(c)) {
			while (isspace(c) && (i < sizeof(buffer))) {
				next = sgf_parser_get_char(parser);
				if (((c == '\r') && (next == '\n')) ||
				    ((c == '\n') && (next == '\r'))) {
					if (simple)
						/* Treat a two-character line break as one space */
						buffer[i++] = ' ';
					else
						/* Canonicalize two-character line breaks */
						buffer[i++] = '\n';
					c = sgf_parser_get_char(parser);
				} else {
					if (!simple && ((c == '\r') || (c == '\n')))
						buffer[i++] = '\n';
					else
						buffer[i++] = ' ';
					c = next;
				}
			}
		} else if (c == '\\') {
			c = sgf_parser_get_char(parser);
			if (c == EOF)
				continue;
			next = sgf_parser_get_char(parser);
			if ((c == '\r') || (c == '\n')) {
				/* Ignore line breaks following a backslash */
				if (((c == '\r') && (next == '\n')) ||
				    ((c == '\n') && (next == '\r')))
					c = sgf_parser_get_char(parser);
				else
					c = next;
				continue;
			} 
			if (isspace(c))
				buffer[i++] = ' ';
			else
				buffer[i++] = c;
			c = next;
		} else if (c == '\0') {
			/* Skip null bytes */
			c = sgf_parser_get_char(parser);
		} else {
			buffer[i++] = c;
			c = sgf_parser_get_char(parser);
		}
	}
	result[length] = '\0';

	return result;
}

static int sgf_property_type_name_cmp(const char *name, const sgf_property_type *property_type)
{
	return strcmp(name, property_type->name);
}

static sgf_property_type *sgf_property_types_search(const sgf_property_type *array, unsigned int num, const char *name)
{
	return (sgf_property_type *) array_search((const void **) &array, &num, sizeof(sgf_property_type), name, (int (*)(const void *, const void *)) sgf_property_type_name_cmp);
}

static sgf_property_type *sgf_parser_search_property_types(const sgf_parser *parser, const char *name)
{
	static sgf_property_type unknown = { "", SGF_PROPERTY_TYPE_NONE, 99, SGF_PROPERTY_VALUE_SINGLE | SGF_PROPERTY_LIST, { &sgf_property_value_type_handler_unknown, NULL } };
	sgf_property_type *property_type;

	property_type = sgf_property_types_search(parser->property_types, parser->num_property_types, name);
	if (property_type != NULL)
		return property_type;
	if (parser->game < sizeof(sgf_property_types_game_specific) / sizeof(*sgf_property_types_game_specific)) {
		property_type = sgf_property_types_search(sgf_property_types_game_specific[parser->game].property_types, sgf_property_types_game_specific[parser->game].num_property_types, name);
		if (property_type != NULL)
			return property_type;
	}
	property_type = sgf_property_types_search(sgf_property_types_common, sizeof(sgf_property_types_common) / sizeof(*sgf_property_types_common), name);
	if (property_type == NULL)
		property_type = &unknown;

	return property_type;
}
