/*
	terminal.c
*/

#include "config.h"

#include <errno.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/select.h>
#include <readline/readline.h>
#include <readline/history.h>

#include "array.h"
#include "commands.h"
#include "game.h"
#include "terminal.h"

/*
struct {
	char main;
	char alt;
} display_chars[] = {
	{ 'X', 'x' },
	{ 'O', 'o' },
	{ 'R', 'r' },
	{ 'B', 'b' }
};
*/

char display_chars[] = { 'X', 'O', 'R', 'B' };

static interface_state interface;

static void execute_command(char *s);

static void go_game_print(const go_game *game, FILE *fp);

static void go_game_board_print_graph(const go_game *game, FILE *fp);

static void go_game_board_print_rectangular(const go_game *game, FILE *fp);

int interface_terminal(int argc, char *argv[])
{
	fd_set fds;

	rl_readline_name = argv[0];
	if (interface_state_init(&interface) == NULL) {
		fprintf(stderr, "Can't initialize interface.\n");
		return 1;
	}
	rl_callback_handler_install(PACKAGE "> ", execute_command);
	while (interface.is_running) {
		FD_ZERO(&fds);
		FD_SET(0, &fds);
		errno = 0;
		if (select(FD_SETSIZE, &fds, NULL, NULL, NULL) > 0) {
			if (FD_ISSET(0, &fds))
				rl_callback_read_char();
		} else if (errno == EINTR) {
			errno = 0;
		}
		if (errno != 0) {
			perror(argv[0]);
			break;
		}
	}
	interface_state_free(&interface);
	if (errno != 0)
		return 1;

	return 0;
}

static void execute_command(char *s)
{
	const command *cmds[MAX_RETURNED_COMMANDS];
	char *name, *tmp;
	unsigned int i, num_cmds;

	interface.update_display = 0;
	if (s == NULL) {
		putchar('\n');
		interface.is_running = 0;
	} else {
		tmp = s;
		name = get_word((const char **) &tmp);
		if ((name == NULL) || (name[0] == '\0'))
			return;
		command_lookup(name, cmds, &num_cmds, MAX_RETURNED_COMMANDS);
		if (num_cmds == 0) {
			if (command_move(&interface, NULL, s) == NULL)
				printf("Unknown command: %s\n", name);
		} else if (num_cmds == 1) {
			if (cmds[0]->func != NULL)
				cmds[0]->func(&interface, name, tmp);
		} else {
			printf("Ambiguous command: `%s' matches ", name);
			printf("%s", cmds[0]->name);
			for (i = 1; i < num_cmds - 1; i++)
				printf(", %s", cmds[i]->name);
			if (i < num_cmds)
				printf(" and %s", cmds[i]->name);
			putchar('\n');
		}
		add_history(s);
		free(s);
		free(name);
	}
	if (interface.is_running) {
		if ((interface.current_game != (unsigned int) -1) &&
		    interface.update_display) {
			/*
			go_history_print_hash_table(&interface.current_game->history, stdout);
			go_history_turn_print_recursively(interface.current_game->history.history, 0, stdout);
			*/
			interface_state_print(&interface, stdout);
		}
	} else {
		rl_callback_handler_remove();
	}

	return;
}

void go_history_turn_print_recursively(const go_history_turn *turn, int indent, FILE *fp)
{
	for (; turn != NULL; turn = turn->variation) {
		printf("%*s%u (%p)\n", indent, "", turn->game_state.turn_num, (void *) turn);
		go_history_turn_print_recursively(turn->next, indent, fp);
		indent++;
	}

	return;
}

void go_history_print_hash_table(const go_history *history, FILE *fp)
{
	unsigned int i, j;

	for (j = 0; j < history->hash_table_size; j++) {
		if (history->hash_table[j].num_turns != 0) {
			fprintf(fp, "%u:", j);
			for (i = 0; i < history->hash_table[j].num_turns; i++)
				fprintf(fp, " %u", history->hash_table[j].turns[i]->game_state.turn_num);
			putc('\n', fp);
		}
	}

	return;
}

interface_state *interface_state_init(interface_state *interface)
{
	interface->is_running = 1;
	interface->update_display = 0;
	array_init((void **) &interface->games, &interface->num_games);
	interface->current_game = (unsigned int) -1;
	interface->alternate_game = (unsigned int) -1;
	interface->options.show_comments = 1;

	return interface;
}

void interface_state_free(interface_state *interface)
{
	array_free((void **) &interface->games, &interface->num_games, sizeof(go_game), (void (*)(void *)) go_game_free);

	return;
}

go_game *interface_state_add_game(interface_state *interface, const go_game *game)
{
	return (go_game *) array_insert((void **) &interface->games, &interface->num_games, sizeof(go_game), game, NULL);
}

go_game *interface_state_remove_game(interface_state *interface, unsigned int index)
{
	go_game *p;

	p = (go_game *) array_remove_index_keep_order((void **) &interface->games, &interface->num_games, sizeof(go_game), index);
	if (p == NULL)
		return NULL;
	if (interface->current_game >= interface->num_games)
		interface->current_game = interface->num_games - 1;
	if (interface->alternate_game >= interface->num_games)
		interface->alternate_game = interface->num_games - 1;

	return p;
}

void interface_state_set_current_game(interface_state *interface, unsigned int index)
{
	interface->alternate_game = interface->current_game;
	interface->current_game = index;

	return;
}

static void go_game_print(const go_game *game, FILE *fp)
{
	const char *game_name;
	unsigned int i;

	game_name = go_game_info_search(game, "name");
	if (game_name != NULL)
		fprintf(fp, "%s\n", game_name);
	if (game->num_teams > 0) {
		fprintf(fp, "%s", game->players[0].name);
		for (i = 1; i < game->num_players; i++)
			fprintf(fp, " vs. %s", game->players[i].name);
		putc('\n', fp);
	}
	if (game->turn_num == 0)
		fprintf(fp, "Game start.");
	else
		fprintf(fp, "Turn %u.", game->turn_num);
	if (game->num_teams > 0) {
		fprintf(fp, " %s to play.", game->players[game->teams[game->next_team].players[game->teams[game->next_team].next_player]].name);
	}
	putc('\n', fp);
	switch (game->board_info.type) {
		case GO_GAME_BOARD_INFO_TYPE_SQUARE:
		case GO_GAME_BOARD_INFO_TYPE_RECTANGULAR:
			go_game_board_print_rectangular(game, fp);
			break;
		default:
			go_game_board_print_graph(game, fp);
			break;
	}

	return;
}

static void go_game_board_print_graph(const go_game *game, FILE *fp)
{
	unsigned int i, j, color;
	char c;

	for (i = 0; i < game->board_info.size.graph.size; i++) {
		c = '.';
		color = game->board.points[i].color;
		if (color > 0) {
			if (go_game_is_dead(game, i))
				c = isprint(game->teams[color - 1].alt_display_char) ? game->teams[color - 1].alt_display_char : '?';
			else
				c = isprint(game->teams[color - 1].display_char) ? game->teams[color - 1].display_char : '?';
		}
		fprintf(fp, "%3u (%c):", i, c);
		for (j = 0; j < game->board.points[i].num_adjacent; j++)
			fprintf(fp, " %u", game->board.points[i].adjacent[j]);
		putchar('\n');
	}

	return;
}

static void go_game_board_print_rectangular(const go_game *game, FILE *fp)
{
	const static struct {
		int type;
		char start;
		char end;
	} markup_chars[] = {
		{ GO_GAME_MARKUP_CIRCLE, '(', ')' },
		{ GO_GAME_MARKUP_SELECTED, '=', '=' },
		{ GO_GAME_MARKUP_SQUARE, '[', ']' },
		{ GO_GAME_MARKUP_TRIANGLE, '<', '>' },
		{ GO_GAME_MARKUP_X, '>', '<' },
		{ GO_GAME_MARKUP_MOVE, '(', ')' }
	};
	unsigned int i, x, y, width, height, color, star_point_index, markup_index;
	int num_digits;
	char c, markup_char, last_markup_char;

	if (game->board_info.type == GO_GAME_BOARD_INFO_TYPE_SQUARE) {
		width = game->board_info.size.square.size;
		height = game->board_info.size.square.size;
	} else {
		width = game->board_info.size.rectangular.width;
		height = game->board_info.size.rectangular.height;
	}
	num_digits = (int) log10(height) + 1;
	fprintf(fp, "%*s", num_digits, "");
	for (x = 0; (x < width) && (x < 'I' - 'A'); x++)
		fprintf(fp, " %c", x + 'A');
	for (; (x < width) && (x < 'Z' - 'A'); x++)
		fprintf(fp, " %c", x + 'A' + 1);
	for (; x < width; x++)
		fprintf(fp, "  ");
	putc('\n', fp);
	star_point_index = 0;
	markup_index = 0;
	for (y = 0; y < height; y++) {
		fprintf(fp, "%*u", num_digits, height - y);
		last_markup_char = ' ';
		for (x = 0; x < width; x++) {
			markup_char = last_markup_char;
			last_markup_char = ' ';
			if ((star_point_index < game->board_info.num_star_points) &&
			    (game->board_info.star_points[star_point_index] == y * width + x)) {
				c = ',';
				star_point_index++;
			} else {
				c = '.';
			}
			/*
			c = game->board.points[y * width + x].string_index % 26 + 'a';
			*/
			while ((markup_index < game->num_markup) &&
			       (game->markup[markup_index].index == y * width + x)) {
				switch (game->markup[markup_index].type) {
					case GO_GAME_MARKUP_CIRCLE:
					case GO_GAME_MARKUP_SELECTED:
					case GO_GAME_MARKUP_SQUARE:
					case GO_GAME_MARKUP_TRIANGLE:
					case GO_GAME_MARKUP_X:
					case GO_GAME_MARKUP_MOVE:
						for (i = 0; i < sizeof(markup_chars) / sizeof(*markup_chars); i++) {
							if (markup_chars[i].type == game->markup[markup_index].type)
								break;
						}
						if (i >= sizeof(markup_chars) / sizeof(*markup_chars))
							break;
						if (markup_char == ' ')
							markup_char = markup_chars[i].start;
						else if (markup_char != markup_chars[i].start)
							markup_char = '|';
						if (last_markup_char == ' ')
							last_markup_char = markup_chars[i].end;
						else if (last_markup_char != markup_chars[i].end)
							last_markup_char = '|';
						break;
					case GO_GAME_MARKUP_LABEL:
						c = *game->markup[markup_index].data.string;
						break;
				}
				markup_index++;
			}
			color = game->board.points[y * width + x].color;
			if (color > 0) {
				if (go_game_is_dead(game, y * width + x))
					c = isprint(game->teams[color - 1].alt_display_char) ? game->teams[color - 1].alt_display_char : '?';
				else
					c = isprint(game->teams[color - 1].display_char) ? game->teams[color - 1].display_char : '?';
			}
			fprintf(fp, "%c%c", markup_char, c);
		}
		putc(last_markup_char, fp);
		fprintf(fp, "%*u", num_digits, height - y);
		putc('\n', fp);
	}
	fprintf(fp, "%*s", num_digits, "");
	for (x = 0; (x < width) && (x < 'I' - 'A'); x++)
		fprintf(fp, " %c", x + 'A');
	for (; (x < width) && (x < 'Z' - 'A'); x++)
		fprintf(fp, " %c", x + 'A' + 1);
	for (; x < width; x++)
		fprintf(fp, "  ");
	putc('\n', fp);

	return;
}

void interface_state_print(const interface_state *interface, FILE *fp)
{
	const go_history_turn *turn;
	unsigned int i, num_variations;

	if (interface->current_game == (unsigned int) -1)
		return;
	go_game_print(&interface->games[interface->current_game], fp);
	if (interface->options.show_comments) {
		for (i = 0; i < interface->games[interface->current_game].history.current_turn->num_actions; i++) {
			if (interface->games[interface->current_game].history.current_turn->actions[i].type == GO_HISTORY_TURN_ACTION_COMMENT) {
				print_wrapped(fp, interface->games[interface->current_game].history.current_turn->actions[i].value.string, "");
				putc('\n', fp);
			}
		}
	}
	turn = interface->games[interface->current_game].history.current_turn->prev;
	if (turn != NULL) {
		num_variations = 0;
		for (turn = turn->next; turn != NULL; turn = turn->variation) {
			num_variations++;
			if (turn == interface->games[interface->current_game].history.current_turn)
				i = num_variations;
		}
		if (num_variations > 1)
			fprintf(fp, "Variation %u of %u.\n", i, num_variations);
	}
	if (go_game_is_over(&interface->games[interface->current_game])) {
		go_game_count_score(&interface->games[interface->current_game]);
		fprintf(fp, "This game is over.\n");
	}

	return;
}

void print_wrapped(FILE *fp, const char *text, const char *indent)
{
	int rows, cols;
	unsigned int i, indent_length, num_newlines;
	const char *p;

	rl_get_screen_size(&rows, &cols);
	if (indent != NULL)
		indent_length = strlen(indent);
	else
		indent_length = 0;
	p = text;
	i = 0;
	if (indent_length != 0) {
		fprintf(fp, "%s", indent);
		i += indent_length;
	}
	while (*text != '\0') {
		num_newlines = 0;
		while (isspace(*p)) {
			if (*p == '\n')
				num_newlines++;
			p++;
		}
		if (num_newlines > 0) {
			while ((num_newlines > 0) && (*p != '\0')) {
				putc('\n', fp);
				i = 0;
				if (indent_length != 0) {
					fprintf(fp, "%s", indent);
					i += indent_length;
				}
				num_newlines--;
			}
			text = p;
			continue;
		}
		while (!isspace(*p) && (*p != '\0'))
			p++;
		if ((i + p - text > cols)) {
			if (i > indent_length) {
				putc('\n', fp);
				i = 0;
				if (indent_length != 0) {
					fprintf(fp, "%s", indent);
					i += indent_length;
				}
			}
			while (isspace(*text))
				text++;
		}
		fwrite(text, sizeof(char), p - text, fp);
		i += p - text;
		text = p;
	}

	return;
}
