/*
	commands.c
*/

#include "config.h"

#include <ctype.h>
#include <errno.h>
#include <math.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <time.h>
#include <readline/readline.h>

#include "array.h"
#include "commands.h"
#include "game.h"
#include "menu.h"
#include "terminal.h"

const static command commands[] = {
	{ "back", command_back, "back [NUMBER]", "Move back one turn in the current game. With NUMBER, move back NUMBER turns." },
	{ "close", command_close, "close", "Close the current game without saving." },
	{ "color", command_color, "color LOCATION COLOR", "Color the point at LOCATION with the color COLOR. COLOR may be a team index or a display character." },
	{ "comment", command_comment, "comment", "Add a new comment to the current game." },
	{ "dead", command_dead, "dead [LOCATION ...]", "Toggle the alive status of the strings containing the stones at LOCATION." },
	{ "display", command_display, "display", "Display the board of the current game." },
	{ "forward", command_forward, "forward [NUMBER]", "Move forward one turn in the current game. With NUMBER, move forward NUMBER turns." },
	{ "games", command_games, "games", "Display a list of open games." },
	{ "help", command_help, "help [COMMAND]", "Display a list of commands and their usage. With COMMAND, give detailed help on COMMAND." },
	{ "label", command_label, "label LOCATION TEXT", "Label the point at LOCATION with TEXT." },
	{ "last", command_last, "last", "Go to the last turn of the current game." },
	{ "load", command_load, "load FILENAME [STRING]", "Load a game from the file FILENAME. If FILENAME contains more than one game, prompt for which one. With STRING, load the game whose name contains STRING." },
	{ "mark", command_mark, "mark (LOCATION TYPE | clear)", "Place a mark at LOCATION of type TYPE. TYPE must be one of ``circle'', ``selected'', ``square'', ``triangle'' and ``x''. ``mark clear'' removes all markup." },
	{ "move", command_move, "[move] LOCATION", "Place a stone of the current player's color at LOCATION in the current game." },
	{ "new", command_new, "new", "Start a new game. The new game becomes the current game." },
	{ "pass", command_pass, "pass", "Pass in the current game." },
	{ "players", command_players, "players", "Display a list of the players in the current game." },
	{ "quit", command_quit, "quit", "Exits the program, discarding any open games." },
	{ "resign", command_resign, "resign", "Resign in the current game." },
	{ "rewind", command_rewind, "rewind", "Go to the first turn of the current game." },
	{ "save", command_save, "save FILENAME [FORMAT]", "Save the current game as FILENAME, in a format guessed from FILENAME's extension, or in SGF if there is no extension. If FORMAT is given, save in FORMAT. FORMAT must be one of ``xml'', ``sgf'', ``ps'', ``PostScript'' and ``eps'' (case-sensitive)." },
	{ "score", command_score, "score", "Display the score of each team in the current game." },
	{ "select", command_select, "select [STRING]", "Prompt for a game and make it the current game. With STRING, make the game whose name contains STRING the current game." },
	{ "switch", command_switch, "switch", "Switch to the alternate game." },
	{ "undo", command_undo, "undo", "Delete the current turn and all its children from the current game." },
	{ "variation", command_variation, "variation", "Go to the next variation of this turn." }
};

#ifndef HAVE_STRDUP
static char *strdup(const char *s)
{
	char *t;

	t = (char *) malloc(strlen(s) + 1);
	if (t == NULL)
		return NULL;
	strcpy(t, s);

	return t;
}
#endif

static unsigned int parse_location(const go_game *game, const char **s);

static unsigned int parse_location_graph(const char **s, unsigned int size);

static unsigned int parse_location_square(const char **s, unsigned int size);

static unsigned int parse_location_rectangular(const char **s, unsigned int width, unsigned int height);

static char *go_game_make_name(const go_game *game, unsigned int index);

static int go_game_match_string(const go_game *game, const char *s);

static go_game *load_sgf(go_game *game, const char *filename, unsigned int index);

static int go_team_ptr_cmp(const go_team **a, const go_team **b);

void command_lookup(const char *name, const command **results, unsigned int *num_results, unsigned int max_results)
{
	size_t name_length;
	unsigned int i;

	*num_results = 0;
	name_length = strlen(name);
	if (name_length == 0)
		return;
	for (i = 0; i < sizeof(commands) / sizeof(*commands); i++) {
		if (strcmp(commands[i].name, name) == 0) {
			*num_results = 1;
			if (max_results >= 1)
				results[0] = &commands[i];
			return;
		}
		if (strncmp(commands[i].name, name, name_length) == 0) {
			if (*num_results < max_results)
				results[(*num_results)++] = &commands[i];
		}
	}

	return;
}

char *get_word(const char **s)
{
	const char *p;
	char *result, *tmp;
	size_t length;
	unsigned int quoted;

	length = 0;
	result = (char *) malloc(length + 1);
	if (result == NULL)
		return NULL;
	while (isspace(**s))
		(*s)++;
	p = *s;
	quoted = 0;
	for (;;) {
		while ((quoted || !isspace(*p)) && (*p != '"') &&
		       (*p != '\\') && (*p != '\0'))
			p++;
		tmp = realloc(result, length + (p - *s) + 1);
		if (tmp == NULL) {
			free(result);
			return NULL;
		}
		result = tmp;
		memmove(result + length, *s, p - *s);
		length += p - *s;
		if (isspace(*p)) {
			if (quoted)
				*s = p;
			else
				break;
		} else if (*p == '"') {
			*s = p + 1;
			p = p + 1;
			quoted = !quoted;
		} else if (*p == '\\') {
			*s = p + 1;
			if (*s == '\0')
				p = p + 1;
			else
				p = p + 2;
		} else if (*p == '\0') {
			if (quoted) {
				free(result);
				return NULL;
			} else {
				break;
			}
		}
	}
	result[length] = '\0';
	*s = p;
	while (isspace(**s))
		(*s)++;

	return result;
}

static unsigned int parse_location(const go_game *game, const char **s)
{
	unsigned int index;

	switch(game->board_info.type) {
		case GO_GAME_BOARD_INFO_TYPE_SQUARE:
			index = parse_location_square(s, game->board_info.size.square.size);
			break;
		case GO_GAME_BOARD_INFO_TYPE_RECTANGULAR:
			index = parse_location_rectangular(s, game->board_info.size.rectangular.width, game->board_info.size.rectangular.height);
			break;
		default:
			index = parse_location_graph(s, game->board_info.size.graph.size);
			break;
	}

	return index;
}

static unsigned int parse_location_graph(const char **s, unsigned int size)
{
	unsigned int index;

	if (sscanf(*s, "%u", &index) != 1)
		return (unsigned int) -1;
	while (isdigit(**s))
		(*s)++;
	while (isspace(**s))
		(*s)++;
	if (index >= size)
		return (unsigned int) -1;

	return index;
}

static unsigned int parse_location_square(const char **s, unsigned int size)
{
	return parse_location_rectangular(s, size, size);
}

static unsigned int parse_location_rectangular(const char **s, unsigned int width, unsigned int height)
{
	unsigned int i, x, y;
	char type[2];
	char c;

	for (i = 0; i < 2; i++) {
		x = y;
		while (!isalpha(**s) && !isdigit(**s) && (**s != '\0'))
			(*s)++;
		if (isalpha(**s)) {
			c = toupper(**s);
			if (c < 'I')
				y = c - 'A';
			else if (c == 'I')
				return (unsigned int) -1;
			else
				y = c - 'A' - 1;
			if (isupper(**s) && (width > 26))
				y += 26;
			(*s)++;
			type[i] = 'A';
		} else if (isdigit(**s)) {
			y = 0;
			while (isdigit(**s) && (**s != '\0')) {
				y *= 10;
				y += **s - '0';
				(*s)++;
			}
			y--;
			type[i] = '1';
		} else {
			return (unsigned int) -1;
		}
	}
	while (!isalpha(**s) && !isdigit(**s) && (**s != '\0'))
		(*s)++;
	if (isdigit(type[0]) && isalpha(type[1])) {
		i = x;
		x = y;
		y = i;
	}
	if ((x >= width) || (y >= height))
		return (unsigned int) -1;

	return (height - y - 1) * width + x;
}

interface_state *command_back(interface_state *interface, const char *name, const char *args)
{
	unsigned int number;

	if (interface->current_game == (unsigned int) -1) {
		printf("There are no games open\n");
		return interface;
	}
	if ((args == NULL) || (args[0] == '\0')) {
		number = 1;
	} else {
		if (sscanf(args, "%u", &number) != 1)
			return NULL;
	}
	if ((number > 0) &&
	    (go_game_history_back(&interface->games[interface->current_game]) == NULL)) {
		printf("You are at the first turn\n");
		return interface;
	}
	interface->update_display = 1;
	number--;
	for (; number > 0; number--) {
		if (go_game_history_back(&interface->games[interface->current_game]) == NULL)
			return interface;
	}

	return interface;
}

interface_state *command_close(interface_state *interface, const char *name, const char *args)
{
	if (interface->current_game == (unsigned int) -1) {
		printf("There are no games open\n");
		return interface;
	}
	/* Do you want to save? */
	go_game_free(&interface->games[interface->current_game]);
	if (interface_state_remove_game(interface, interface->current_game) == NULL)
		return NULL;
	interface->update_display = 1;

	return interface;
}

interface_state *command_color(interface_state *interface, const char *name, const char *args)
{
	go_game *game;
	const char *tmp;
	unsigned int index, color;

	if (interface->current_game == (unsigned int) -1) {
		printf("There are no games open\n");
		return interface;
	}
	game = &interface->games[interface->current_game];
	tmp = args;
	while (isspace(*tmp))
		tmp++;
	if (*tmp == '\0') {
		printf("You must provide a location\n");
		return interface;
	}
	index = parse_location(game, &tmp);
	if (index == (unsigned int) -1) {
		printf("Invalid location: %s\n", args);
		return interface;
	}
	while (isspace(*tmp))
		tmp++;
	if (*tmp == '\0') {
		printf("You must provide a color\n");
		return interface;
	}
	args = tmp + 1;
	while (isspace(*args))
		args++;
	if (*args == '\0') {
		for (color = 0; color < game->num_teams; color++) {
			if (*tmp == game->teams[color].display_char)
				break;
		}
	}
	if (color >= game->num_teams) {
		if (sscanf(tmp, "%u", &color) != 1) {
			printf("Invalid color: %s\n", tmp);
			return interface;
		}
	}
	/*
	if (interface->games[interface->current_game].history.current_turn->next != NULL) {
		if (go_game_new_turn(&interface->games[interface->current_game]) == NULL)
			return NULL;
	}
	*/
	if (go_game_color_point(game, index, color + 1) == NULL)
		return NULL;
	interface->update_display = 1;

	return interface;
}

interface_state *command_dead(interface_state *interface, const char *name, const char *args)
{
	go_game *game;
	const char *tmp;
	unsigned int index;

	if (interface->current_game == (unsigned int) -1) {
		printf("There are no games open\n");
		return interface;
	}
	game = &interface->games[interface->current_game];
	tmp = args;
	for (;;) {
		while (isspace(*tmp))
			tmp++;
		if (*tmp == '\0')
			break;
		index = parse_location(game, &tmp);
		if (index == (unsigned int) -1) {
			printf("Invalid location ignored: %s\n", args);
			continue;
		}
		if (go_game_toggle_dead(game, index) == NULL)
			return NULL;
	}
	interface->update_display = 1;

	return interface;
}

interface_state *command_comment(interface_state *interface, const char *name, const char *args)
{
	go_game *game;
	char *text, *p;

	if (interface->current_game == (unsigned int) -1) {
		printf("There are no games open\n");
		return interface;
	}
	game = &interface->games[interface->current_game];
	rl_save_prompt();
	text = readline("Comment: ");
	rl_restore_prompt();
	if (text == NULL)
		return NULL;
	/* Don't allow a comment of only white space */
	p = text;
	while (isspace(*p))
		;
	if (*p != '\0') {
		if (go_game_comment(game, text) == NULL)
			return NULL;
	}
	free(text);
	interface->update_display = 1;

	return interface;
}

interface_state *command_display(interface_state *interface, const char *name, const char *args)
{
	if (interface->current_game == (unsigned int) -1) {
		printf("There are no games open\n");
		return interface;
	}
	interface->update_display = 1;

	return interface;
}

interface_state *command_forward(interface_state *interface, const char *name, const char *args)
{
	unsigned int number;

	if (interface->current_game == (unsigned int) -1) {
		printf("There are no games open\n");
		return interface;
	}
	if ((args == NULL) || (args[0] == '\0')) {
		number = 1;
	} else {
		if (sscanf(args, "%u", &number) != 1)
			return NULL;
	}
	if ((number > 0) &&
	    (go_game_history_forward(&interface->games[interface->current_game]) == NULL)) {
		printf("You are at the last turn\n");
		return interface;
	}
	interface->update_display = 1;
	number--;
	for (; number > 0; number--) {
		if (go_game_history_forward(&interface->games[interface->current_game]) == NULL)
			return interface;
	}

	return interface;
}

static char *go_game_make_name(const go_game *game, unsigned int index)
{
	static const char *default_name = "Game ";
	char *result;
	const char *game_name;
	unsigned int num_digits;

	game_name = go_game_info_search(game, "name");
	if ((game_name != NULL) && (game_name[0] != '\0'))
		return strdup(game_name);
	else {
		if (index == 0)
			num_digits = 1;
		else
			num_digits = (unsigned int) log10(index) + 1;
		result = (char *) malloc(strlen(default_name) + num_digits + 1);
		if (result == NULL)
			return NULL;
		sprintf(result, "%s%u", default_name, index);
	}

	return result;
}

interface_state *command_games(interface_state *interface, const char *name, const char *args)
{
	go_game *game;
	unsigned int i;
	char *game_name;

	for (i = 0; i < interface->num_games; i++) {
		game = &interface->games[i];
		game_name = go_game_make_name(game, i);
		if (game_name == NULL)
			continue;
		printf("%u: %s: turn %u", i, game_name, game->turn_num);
		free(game_name);
		if (i == interface->current_game)
			printf(" (current game)");
		else if (i == interface->alternate_game)
			printf(" (alternate game)");
		putchar('\n');
	}

	return interface;
}

interface_state *command_help(interface_state *interface, const char *name, const char *args)
{
	const command *cmds[MAX_RETURNED_COMMANDS];
	char *command_name;
	unsigned int i, num_cmds;

	command_name = get_word((const char **) &args);
	if ((command_name == NULL) || (command_name[0] == '\0')) {
		for (i = 0; i < sizeof(commands) / sizeof(*commands); i++) {
			printf("%s\n", commands[i].usage);
		}
	} else {
		command_lookup(command_name, cmds, &num_cmds, sizeof(cmds) / sizeof(*cmds));
		if (num_cmds == 0) {
			printf("Unknown command: %s\n", command_name);
		} else if (num_cmds == 1) {
			if (cmds[0]->usage != NULL)
				printf("%s: %s\n", command_name, cmds[0]->usage);
			if (cmds[0]->help != NULL) {
				print_wrapped(stdout, cmds[0]->help, "    ");
				putchar('\n');
			} else {
				printf("There is no help for the command `%s'\n", command_name);
			}
		} else {
			printf("%s: `%s' matches ", name, command_name);
			printf("%s", cmds[0]->name);
			for (i = 1; i < num_cmds - 1; i++)
				printf(", %s", cmds[i]->name);
			if (i < num_cmds)
				printf(" and %s", cmds[i]->name);
			putchar('\n');
		}
	}

	return interface;
}

interface_state *command_label(interface_state *interface, const char *name, const char *args)
{
	go_game *game;
	const char *tmp;
	unsigned int index;

	if (interface->current_game == (unsigned int) -1) {
		printf("There are no games open\n");
		return interface;
	}
	game = &interface->games[interface->current_game];
	tmp = args;
	while (isspace(*tmp))
		tmp++;
	if (*tmp == '\0') {
		printf("You must provide a location\n");
		return interface;
	}
	index = parse_location(game, &tmp);
	if (index == (unsigned int) -1) {
		printf("Invalid location: %s\n", args);
		return interface;
	}
	if ((tmp == NULL) || (tmp[0] == '\0')) {
		printf("You must provide a label text\n");
		return interface;
	}
	if (go_game_label(game, index, tmp) == NULL)
		return NULL;
	interface->update_display = 1;

	return interface;
}

interface_state *command_last(interface_state *interface, const char *name, const char *args)
{
	if (interface->current_game == (unsigned int) -1) {
		printf("%s: there are no games open.\n", name);
		return interface;
	}
	if (go_game_history_end(&interface->games[interface->current_game]) == NULL)
		return NULL;
	interface->update_display = 1;

	return interface;
}

static int go_game_match_string(const go_game *game, const char *s)
{
	const char *game_name;

	game_name = go_game_info_search(game, "name");
	if (game_name == NULL)
		return 0;
	if (strstr(game_name, s) != NULL)
		return 1;

	return 0;
}

static go_game *load_xml(go_game *game, const char *filename, unsigned int index)
{
	return go_game_load_xml(game, filename);
}

static go_game *load_sgf(go_game *game, const char *filename, unsigned int index)
{
	return go_game_load_sgf_index(game, filename, index);
}

interface_state *command_load(interface_state *interface, const char *name, const char *args)
{
	static const struct {
		go_game **(*shallow)(go_game **, unsigned int *, const char *);
		go_game *(*loader)(go_game *, const char *, unsigned int);
	} loaders[] = {
		{ NULL, load_xml },
		{ go_games_load_sgf_shallow, load_sgf }
	};
	unsigned int num_games;
	go_game *games;
	go_game new_game, *p;
	char *filename, *game_name, *number;
	unsigned int i, game_index;
	struct stat tmp;
	clock_t start_time;

	filename = get_word(&args);
	if ((filename == NULL) || (filename[0] == '\0')) {
		printf("You must supply a filename\n");
		free(filename);
		return interface;
	}
	if (stat(filename, &tmp) != 0) {
		printf("%s\n", strerror(errno));
		errno = 0;
		return interface;
	}
	/* Try opening the file in different formats */
	for (i = 0; i < sizeof(loaders) / sizeof(*loaders); i++) {
		if ((loaders[i].shallow != NULL) &&
		    (loaders[i].shallow(&games, &num_games, filename) != NULL)) {
			if (num_games == 0)
				continue;
			if ((num_games > 1) && (args != NULL) && (args[0] != '\0')) {
				/* Remove the games that don't match args */
				for (game_index = 0; game_index < num_games; game_index++) {
					if (go_game_match_string(&games[game_index], args) == 0) {
						if (array_remove_index_keep_order((void **) &games, &num_games, sizeof(go_game), game_index) == NULL) {
							free(filename);
							array_free((void **) &games, &num_games, sizeof(go_game), (void (*)(void *)) go_game_free);
							return NULL;
						}
						game_index--;
					}
				}
			}
			if (num_games == 0) {
				printf("No games match `%s'\n", args);
				break;
			} else if (num_games == 1) {
				game_index = 0;
			} else if (num_games > 1) {
				for (;;) {
					for (game_index = 0; game_index < num_games; game_index++) {
						game_name = go_game_make_name(&games[game_index], game_index);
						if (game_name == NULL)
							continue;
						printf("%u: %s\n", game_index, game_name);
						free(game_name);
					}
					rl_save_prompt();
					number = readline("Select a game: ");
					rl_restore_prompt();
					if (number == NULL) {
						free(filename);
						array_free((void **) &games, &num_games, sizeof(go_game), (void (*)(void *)) go_game_free);
						return NULL;
					}
					if (sscanf(number, "%u", &game_index) != 1) {
						printf("`%s' is not a non-negative integer\n", number);
					} else if (game_index >= num_games) {
						printf("Enter a number between 0 and %u\n", num_games);
					} else {
						free(number);
						break;
					}
					free(number);
				}
			}
			array_free((void **) &games, &num_games, sizeof(go_game), (void (*)(void *)) go_game_free);
		} else {
			game_index = 0;
		}
		start_time = clock();
		if ((loaders[i].loader != NULL) &&
		    (loaders[i].loader(&new_game, filename, game_index) != NULL)) {
			printf("Game loaded in %.2f seconds\n", ((double) (clock() - start_time)) / CLOCKS_PER_SEC);
			p = interface_state_add_game(interface, &new_game);
			if (p == NULL) {
				free(filename);
				go_game_free(&new_game);
				return NULL;
			}
			interface_state_set_current_game(interface, p - interface->games);
			interface->update_display = 1;
			break;
		}
	}
	if (i == sizeof(loaders) / sizeof(*loaders)) {
		printf("The file `%s' is not in a recognized format\n", filename);
		free(filename);
	} else {
		free(filename);
		/* If the game has no history, start it */
		if (p->history.current_turn == NULL) {
			if (go_game_start(p) == NULL)
				return NULL;
		}
	}

	return interface;
}

interface_state *command_mark(interface_state *interface, const char *name, const char *args)
{
	go_game *game;
	const char *tmp;
	char *type_string;
	unsigned int index, type;

	if (interface->current_game == (unsigned int) -1) {
		printf("There are no games open\n");
		return interface;
	}
	game = &interface->games[interface->current_game];
	tmp = args;
	type_string = get_word(&tmp);
	if (strcmp(type_string, "clear") == 0) {
		if (go_game_clear_markup(game) == NULL) {
			free(type_string);
			return NULL;
		}
	} else {
		free(type_string);
		tmp = args;
		while (isspace(*tmp))
			tmp++;
		if (*tmp == '\0') {
			printf("You must provide a location\n");
			return interface;
		}
		index = parse_location(game, &tmp);
		if (index == (unsigned int) -1) {
			printf("Invalid location: %s\n", args);
			return interface;
		}
		type_string = get_word(&tmp);
		if ((type_string == NULL) || (type_string[0] == '\0')) {
			free(type_string);
			printf("You must provide a markup type\n");
			return interface;
		}
		if (strcmp(type_string, "circle") == 0) {
			type = GO_GAME_MARKUP_CIRCLE;
		} else if (strcmp(type_string, "selected") == 0) {
			type = GO_GAME_MARKUP_SELECTED;
		} else if (strcmp(type_string, "square") == 0) {
			type = GO_GAME_MARKUP_SQUARE;
		} else if (strcmp(type_string, "triangle") == 0) {
			type = GO_GAME_MARKUP_TRIANGLE;
		} else if (strcmp(type_string, "x") == 0) {
			type = GO_GAME_MARKUP_X;
		} else {
			free(type_string);
			printf("Invalid markup type\n");
			return interface;
		}
		free(type_string);
		if (go_game_mark(game, index, type) == NULL)
			return NULL;
	}
	interface->update_display = 1;

	return interface;
}

interface_state *command_move(interface_state *interface, const char *name, const char *args)
{
	const char *tmp;
	unsigned int index;

	if (interface->current_game == (unsigned int) -1) {
		/* If name is NULL, the user probably mistyped a command */
		if (name == NULL)
			return NULL;
		printf("There are no games open\n");
		return interface;
	}
	tmp = args;
	index = parse_location(&interface->games[interface->current_game], &tmp);
	while (isspace(*tmp))
		tmp++;
	if ((index == (unsigned int) -1) || (*tmp != '\0')) {
		/* If name is NULL or *tmp is not '\0', the user probably
		   mistyped a command */
		if ((name == NULL) || (*tmp != '\0')) {
			return NULL;
		} else {
			printf("Invalid move location: %s\n", args);
			return interface;
		}
	}
	if (go_game_move_is_legal(&interface->games[interface->current_game], index) == 0) {
		printf("Illegal move\n");
		return interface;
	}
	if (go_game_new_turn(&interface->games[interface->current_game]) == NULL)
		return NULL;
	if (go_game_move(&interface->games[interface->current_game], index) == NULL)
		return NULL;
	interface->update_display = 1;

	return interface;
}

interface_state *command_new(interface_state *interface, const char *name, const char *args)
{
	static const char *player_names[] = { "Black", "White"/*, "Red", "Blue"*/ };
	static const menu_entry menu[] = {
		{ '1', "Board type and size" },
		{ '2', "Number of players" },
		{ '3', "Start game" },
		{ '\0', NULL }
	};
	go_game game;
	go_game *new_game;
	go_player *player;
	unsigned int i;

	if (go_game_init(&game) == NULL)
		return NULL;
	if (go_game_board_set_size_rectangular(&game, 19, 19) == NULL) {
		go_game_free(&game);
		return NULL;
	}
	for (i = 0; i < sizeof(player_names) / sizeof(*player_names); i++) {
		player = go_game_new_player(&game);
		if (player == NULL) {
			go_game_free(&game);
			return NULL;
		}
		if (go_player_set_name(player, player_names[i]) == NULL) {
			go_game_free(&game);
			return NULL;
		}
	}
	i = display_menu(menu);
	/*
	switch(show_menu("Select: ", "Board type and size", "Number of players", "Start game", NULL)) {
	*/
	switch (i) {
		case 0:
			printf("Board size\n");
			break;
		case 1:
			printf("Players\n");
			break;
		case 2:
			printf("Start game\n");
			break;
		default:
			printf("Wakka wakka\n");
			break;
	}
	go_game_start(&game);
	new_game = interface_state_add_game(interface, &game);
	if (new_game == NULL) {
		go_game_free(&game);
		return NULL;
	}
	interface_state_set_current_game(interface, new_game - interface->games);
	interface->update_display = 1;

	return interface;
}

interface_state *command_pass(interface_state *interface, const char *name, const char *args)
{
	if (interface->current_game == (unsigned int) -1) {
		printf("There are no games open\n");
		return interface;
	}
	if (go_game_new_turn(&interface->games[interface->current_game]) == NULL)
		return NULL;
	if (go_game_pass_is_legal(&interface->games[interface->current_game]) == 0) {
		printf("Illegal pass\n");
		return interface;
	}
	if (go_game_pass(&interface->games[interface->current_game]) == NULL)
		return NULL;
	interface->update_display = 1;

	return interface;
}

interface_state *command_players(interface_state *interface, const char *name, const char *args)
{
	go_game *game;
	unsigned int i, j;

	if (interface->current_game == (unsigned int) -1) {
		printf("There are no games open\n");
		return interface;
	}
	game = &interface->games[interface->current_game];
	/* See if any teams have more than one player */
	for (i = 0; i < game->num_teams; i++) {
		if (game->teams[i].num_players > 1)
			break;
	}
	if (i < game->num_teams) {
		/* If any team has more than one player, list the teams */
		for (i = 0; i < game->num_teams; i++) {
			if (game->teams[i].num_players == 0)
				continue;
			printf("%c: ", game->teams[i].display_char);
			if (game->teams[i].name != NULL)
				printf("%s\n", game->teams[i].name);
			else
				printf("Team %u\n", i);
			for (j = 0; j < game->teams[i].num_players; j++) {
				printf("   ");
				if (game->players[game->teams[i].players[j]].name != NULL)
					printf("%s", game->players[game->teams[i].players[j]].name);
				else
					printf("Player %u", j);
				if ((i == game->next_team) && (j == game->teams[game->next_team].next_player))
					printf(" (next player)");
				putchar('\n');
			}
		}
	} else {
		for (i = 0; i < game->num_teams; i++) {
			if (game->teams[i].num_players == 0)
				continue;
			printf("%c: ", game->teams[i].display_char);
			if (game->teams[i].name != NULL)
				printf("%s", game->teams[i].name);
			else if (game->players[game->teams[i].players[0]].name != NULL)
				printf("%s", game->players[game->teams[i].players[0]].name);
			else
				printf("Player %u", i);
			if ((i == game->next_team) && (game->teams[game->next_team].next_player == 0))
				printf(" (next player)");
			putchar('\n');
		}
	}

	return interface;
}

interface_state *command_quit(interface_state *interface, const char *name, const char *args)
{
	interface->is_running = 0;

	return interface;
}

interface_state *command_resign(interface_state *interface, const char *name, const char *args)
{
	if (interface->current_game == (unsigned int) -1) {
		printf("There are no games open\n");
		return interface;
	}
	if (go_game_new_turn(&interface->games[interface->current_game]) == NULL)
		return NULL;
	if (go_game_resign(&interface->games[interface->current_game]) == NULL) {
		if (go_game_history_undo(&interface->games[interface->current_game]) == NULL)
			return NULL;
		printf("Illegal resignation\n");
		return interface;
	}
	interface->update_display = 1;

	return interface;
}

static int go_team_ptr_cmp(const go_team **a, const go_team **b)
{
	if (((*a)->is_resigned && !(*b)->is_resigned) ||
	    ((*a)->score < (*b)->score))
		return 1;
	if (((*b)->is_resigned && !(*a)->is_resigned) ||
	    ((*b)->score < (*a)->score))
		return -1;

	return 0;
}

interface_state *command_rewind(interface_state *interface, const char *name, const char *args)
{
	if (interface->current_game == (unsigned int) -1) {
		printf("There are no games open\n");
		return interface;
	}
	if (go_game_history_rewind(&interface->games[interface->current_game]) == NULL)
		return NULL;
	interface->update_display = 1;

	return interface;
}

interface_state *command_save(interface_state *interface, const char *name, const char *args)
{
	const go_game *result;
	char *filename, *format, *p;

	if (interface->current_game == (unsigned int) -1) {
		printf("There are no games open\n");
		return interface;
	}
	filename = get_word(&args);
	if ((filename == NULL) || (filename[0] == '\0')) {
		/* Make a filename from the game name or player names and the date */
		/* Prompt the user */
		printf("You must supply a filename\n");
		free(filename);
		return interface;
	}
	format = get_word(&args);
	if ((format == NULL) || (format[0] == '\0')) {
		free(format);
		p = strrchr(filename, '.');
		if (p != NULL) {
			p++;
			format = (char *) malloc(strlen(p) + 1);
			if (format == NULL) {
				free(filename);
				return NULL;
			}
			strcpy(format, p);
		}
	}
	if ((format == NULL) || (format[0] == '\0') ||
	/*  (strcasecmp(format, "xml") == 0)) { */
		(strcmp(format, "xml") == 0)) {
		result = go_game_save_xml(&interface->games[interface->current_game], filename);
	} else if (strcmp(format, "sgf") == 0) {
		result = go_game_save_sgf(&interface->games[interface->current_game], filename);
	} else if ((strcmp(format, "ps") == 0) ||
	           (strcmp(format, "PostScript") == 0) ||
	           (strcmp(format, "eps") == 0)) {
		result = go_game_save_postscript(&interface->games[interface->current_game], filename);
	} else {
		printf("Invalid file format: %s\n", format);
		free(filename);
		free(format);
		return interface;
	}
	free(filename);
	free(format);
	if (result == NULL) {
		printf("There was an error saving the game\n");
		return NULL;
	}
	printf("Game saved\n");

	return interface;
}

interface_state *command_score(interface_state *interface, const char *name, const char *args)
{
	static struct {
		float num;
		const char *singular;
		const char *plural;
	} scores[4];
	unsigned int num_scores;
	go_game *game;
	go_team **teams;
	unsigned int i, j, num_winners;
	float score;

	if (interface->current_game == (unsigned int) -1) {
		printf("There are no games open\n");
		return interface;
	}
	game = &interface->games[interface->current_game];
	teams = (go_team **) malloc(sizeof(go_team *) * game->num_teams);
	if (teams == NULL)
		return NULL;
	if (go_game_count_score(game) == NULL)
		return NULL;
	for (i = 0; i < game->num_teams; i++)
		teams[i] = &game->teams[i];
	for (i = 0; i < game->num_teams; i++) {
		if (teams[i]->name != NULL) {
			printf("%s:", teams[i]->name);
		} else {
			if (teams[i]->num_players == 1)
				printf("%s:", game->players[teams[i]->players[0]].name);
			else
				printf("Team %u:", i);
		}
		num_scores = 0;
		if (game->ruleset.score_territory) {
			scores[num_scores].num = teams[i]->territory;
			scores[num_scores].singular = "territory";
			scores[num_scores++].plural = "territory";
		}
		if (game->ruleset.score_occupied_points) {
			scores[num_scores].num = teams[i]->num_occupied_points;
			scores[num_scores].singular = "occupied point";
			scores[num_scores++].plural = "occupied points";
		}
		if (game->ruleset.score_captures || game->ruleset.score_passes) {
			scores[num_scores].num = teams[i]->num_prisoners;
			scores[num_scores].singular = "prisoner";
			scores[num_scores++].plural = "prisoners";
		}
		if (teams[i]->komi != 0.0) {
			scores[num_scores].num = teams[i]->komi;
			scores[num_scores].singular = "komi";
			scores[num_scores++].plural = "komi";
		}
		for (j = 0; j < num_scores; j++) {
			if (j == 0)
				printf(" %g", scores[j].num);
			else
				printf(" %c %g", (scores[j].num >= 0.0) ? '+' : '-', fabs(scores[j].num));
			if (scores[j].num == 1.0)
				printf(" %s", scores[j].singular);
			else
				printf(" %s", scores[j].plural);
		}
		if (num_scores > 0)
			printf(" =");
		printf(" %g", teams[i]->score);
		if (teams[i]->is_resigned)
			printf(" (resigned)");
		putchar('\n');
	}
	array_sort((void **) &teams, &game->num_teams, sizeof(go_team *), (int (*)(const void *, const void *)) go_team_ptr_cmp);
	printf("Result:");
	for (num_winners = game->num_teams - 1; num_winners > 0; num_winners--) {
		if ((teams[num_winners]->is_resigned != teams[num_winners - 1]->is_resigned) ||
		    (teams[num_winners]->score != teams[num_winners - 1]->score))
			break;
	}
	if (num_winners == 0) {
		printf(" Tie");
	} else {
		i = 0;
		while (i < num_winners) {
			if (teams[i]->name != NULL) {
				printf(" %s", teams[i]->name);
			} else {
				if (teams[i]->num_players == 1)
					printf(" %s", game->players[teams[i]->players[0]].name);
				else
					printf(" Team %u", i);
			}
			score = teams[i]->score;
			i++;
			if (teams[i]->is_resigned) {
				printf("+R");
				break;
			} else if (score == teams[i]->score) {
				printf(",");
			} else {
				printf("+%g", score - teams[i]->score);
			}
		}
	}
	free(teams);
	putchar('\n');

	return interface;
}

interface_state *command_select(interface_state *interface, const char *name, const char *args)
{
	char *game_name, *number;
	unsigned int i;

	if (interface->current_game == (unsigned int) -1) {
		printf("There are no games open\n");
		return interface;
	}
	for (;;) {
		for (i = 0; i < interface->num_games; i++) {
			if (go_game_match_string(&interface->games[i], args) == 0) {
				game_name = go_game_make_name(&interface->games[i], i);
				if (game_name == NULL)
					continue;
				printf("%u: %s", i, game_name);
				free(game_name);
				if (i == interface->current_game)
					printf(" (current game)");
				else if (i == interface->alternate_game)
					printf(" (alternate game)");
				putchar('\n');
			}
		}
		rl_save_prompt();
		number = readline("Select a game: ");
		rl_restore_prompt();
		if (sscanf(number, "%u", &i) != 1) {
			printf("`%s' is not a non-negative integer\n", number);
		} else if (i >= interface->num_games) {
			printf("Enter a number between 0 and %u\n", interface->num_games);
		} else {
			free(number);
			break;
		}
		free(number);
	}
	interface_state_set_current_game(interface, i);
	interface->update_display = 1;

	return interface;
}

interface_state *command_switch(interface_state *interface, const char *name, const char *args)
{
	if (interface->current_game == (unsigned int) -1) {
		printf("There are no games open\n");
		return interface;
	} else if (interface->num_games == 1) {
		printf("There are no other games open\n");
		return interface;
	}
	if (interface->alternate_game == (unsigned int) -1)
		interface->alternate_game = (interface->current_game + 1) % interface->num_games;
	interface_state_set_current_game(interface, interface->alternate_game);
	interface->update_display = 1;

	return interface;
}

interface_state *command_undo(interface_state *interface, const char *name, const char *args)
{
	if (interface->current_game == (unsigned int) -1) {
		printf("There are no games open\n");
		return interface;
	}
	if (go_game_history_undo(&interface->games[interface->current_game]) == NULL)
		return NULL;
	interface->update_display = 1;

	return interface;
}

interface_state *command_variation(interface_state *interface, const char *name, const char *args)
{
	go_history_turn *turn;

	if (interface->current_game == (unsigned int) -1) {
		printf("There are no games open\n");
		return interface;
	}
	turn = interface->games[interface->current_game].history.current_turn;
	if (turn == NULL)
		return NULL;
	if (turn->variation == NULL) {
		if (turn->prev != NULL)
			turn = turn->prev->next;
	} else {
		turn = turn->variation;
	}
	if (go_game_history_seek(&interface->games[interface->current_game], turn) == NULL)
		return NULL;
	interface->update_display = 1;

	return interface;
}
