/*
	history.c
*/

#include "config.h"

#include <stdlib.h>
#include <string.h>

#include "array.h"
#include "board.h"
#include "game.h"

#define HASH_MULTIPLIER 9301
#define HASH_INCREMENT 49297
#define HASH(x, y) ((x) * HASH_MULTIPLIER + HASH_INCREMENT + (y))
#define HASH_TABLE_GROW (.75)
#define HASH_TABLE_SHRINK (HASH_TABLE_GROW / 4)

static void go_history_turn_action_free_string(go_history_turn_action *action);

static int go_history_turn_action_cmp_integer(const go_history_turn_action *a, const go_history_turn_action *b);

static int go_history_turn_action_cmp_index_color(const go_history_turn_action *a, const go_history_turn_action *b);

static int go_history_turn_action_cmp_string(const go_history_turn_action *a, const go_history_turn_action *b);

static int go_history_turn_action_cmp_markup(const go_history_turn_action *a, const go_history_turn_action *b);

/* Get rid of cmp, if possible */
typedef struct go_history_turn_action_handler go_history_turn_action_handler;
struct go_history_turn_action_handler {
	unsigned int type;
	void (*free)(go_history_turn_action *);
	int (*cmp)(const go_history_turn_action *, const go_history_turn_action *);
};

static const go_history_turn_action_handler action_handlers[] = {
	{ GO_HISTORY_TURN_ACTION_COLOR_POINT, NULL, go_history_turn_action_cmp_index_color },
	{ GO_HISTORY_TURN_ACTION_MOVE, NULL, go_history_turn_action_cmp_integer },
	{ GO_HISTORY_TURN_ACTION_PASS, NULL, NULL },
	{ GO_HISTORY_TURN_ACTION_RESIGN, NULL, NULL },
	{ GO_HISTORY_TURN_ACTION_SET_PLAYER, NULL, go_history_turn_action_cmp_integer },
	{ GO_HISTORY_TURN_ACTION_COMMENT, go_history_turn_action_free_string, go_history_turn_action_cmp_string },
	{ GO_HISTORY_TURN_ACTION_MARKUP, NULL, go_history_turn_action_cmp_markup }
};

static go_board_state *go_board_state_init(go_board_state *board_state);

static void go_board_state_free(go_board_state *board_state);

static go_board *go_board_from_state(go_board *board, const go_board_state *board_state);

static go_board_state *go_board_state_from_board(go_board_state *board_state, const go_board *board);

static unsigned int go_board_state_hash(const go_board_state *board_state);

static go_team *go_team_from_state(go_team *team, const go_team_state *team_state);

static go_team_state *go_team_state_from_team(go_team_state *team_state, const go_team *team);

static go_game_state *go_game_state_init(go_game_state *game_state);

static void go_game_state_free(go_game_state *game_state);

go_game *go_game_from_state(go_game *game, const go_game_state *game_state);

go_game_state *go_game_state_from_game(go_game_state *game_state, const go_game *game);

static int go_history_turn_action_handler_type_cmp(const unsigned int *type, const go_history_turn_action_handler *handler);

static go_history_turn_action_handler *action_handlers_search(unsigned int type);

static void go_history_turn_action_free(go_history_turn_action *action);

static go_history_turn *go_history_turn_init(go_history_turn *turn);

static void go_history_turn_free(go_history_turn *turn);

static void go_history_turn_free_recursively(go_history_turn *turn);

static void go_history_hash_table_bucket_free(go_history_hash_table_bucket *bucket);

go_history *go_history_init(go_history *history);

void go_history_free(go_history *history);

go_history_turn *go_history_new_turn(go_history *history);

static go_history *go_history_delete_recursively(go_history *history, go_history_turn *turn);

go_history *go_history_delete_turn(go_history *history, go_history_turn *turn);

static void go_history_search(const go_history *history, const void *elem, int (*cmp)(const void *, const go_history_turn *), unsigned int (*hash_func)(const void *), go_history_turn ***array, unsigned int *num);

static int go_board_turn_cmp(const go_board *board, const go_history_turn *turn);

static unsigned int go_board_hash(const go_board *board);

void go_history_search_board(const go_history *history, const go_board *board, go_history_turn ***array, unsigned int *num);

static go_history *go_history_hash_table_resize(go_history *history, unsigned int size);

go_history *go_history_add_action(go_history *history, const go_history_turn_action *action);

go_history *go_history_remove_action(go_history *history, unsigned int index);

go_history *go_history_hash_table_add(go_history *history, const go_history_turn *turn);

static int ptr_cmp(const void *a, const void *b);

go_history *go_history_hash_table_remove(go_history *history, const go_history_turn *turn);

static go_board_state *go_board_state_init(go_board_state *board_state)
{
	array_init((void **) &board_state->colors, &board_state->num_points);

	return board_state;
}

static void go_board_state_free(go_board_state *board_state)
{
	array_free((void **) &board_state->colors, &board_state->num_points, sizeof(unsigned int), NULL);

	return;
}

static go_board *go_board_from_state(go_board *board, const go_board_state *board_state)
{
	unsigned int i;

	for (i = 0; i < board->num_points; i++)
		board->points[i].color = board_state->colors[i];
	if (go_board_calculate_strings(board) == NULL)
		return NULL;

	return board;
}

static go_board_state *go_board_state_from_board(go_board_state *board_state, const go_board *board)
{
	unsigned int i;

	board_state->colors = (unsigned int *) realloc(board_state->colors, board->num_points * sizeof(unsigned int));
	if (board_state->colors == NULL)
		return NULL;
	board_state->num_points = board->num_points;
	for (i = 0; i < board->num_points; i++)
		board_state->colors[i] = board->points[i].color;

	return board_state;
}

static unsigned int go_board_state_hash(const go_board_state *board_state)
{
	unsigned int i, hash;

	hash = 0;
	for (i = 0; i < board_state->num_points; i++)
		hash = HASH(hash, board_state->colors[i]);

	return hash;
}

static go_team *go_team_from_state(go_team *team, const go_team_state *team_state)
{
	team->current_player = team_state->current_player;
	team->next_player = team_state->next_player;
	team->is_resigned = team_state->is_resigned;
	team->num_prisoners = team_state->num_prisoners;
	team->score = team_state->score;

	return team;
}

static go_team_state *go_team_state_from_team(go_team_state *team_state, const go_team *team)
{
	team_state->current_player = team->current_player;
	team_state->next_player = team->next_player;
	team_state->is_resigned = team->is_resigned;
	team_state->num_prisoners = team->num_prisoners;
	team_state->score = team->score;

	return team_state;
}

void go_game_markup_free(go_game_markup *markup)
{
	if (markup->type == GO_GAME_MARKUP_LABEL)
		free(markup->data.string);

	return;
}

static go_game_state *go_game_state_init(go_game_state *game_state)
{
	if (go_board_state_init(&game_state->board_state) == NULL)
		return NULL;
	game_state->team_states = NULL;
	array_init((void **) &game_state->dead_strings, &game_state->num_dead_strings);
	array_init((void **) &game_state->markup, &game_state->num_markup);

	return game_state;
}

static void go_game_state_free(go_game_state *game_state)
{
	go_board_state_free(&game_state->board_state);
	free(game_state->team_states);
	array_free((void **) &game_state->dead_strings, &game_state->num_dead_strings, sizeof(unsigned int), NULL);
	array_free((void **) &game_state->markup, &game_state->num_markup, sizeof(go_game_markup), (void (*)(void *)) go_game_markup_free);

	return;
}

go_game *go_game_from_state(go_game *game, const go_game_state *game_state)
{
	unsigned int i;

	if (go_board_from_state(&game->board, &game_state->board_state) == NULL)
		return NULL;
	for (i = 0; i < game->num_teams; i++) {
		if (go_team_from_state(&game->teams[i], &game_state->team_states[i]) == NULL)
			return NULL;
	}
	game->current_team = game_state->current_team;
	game->next_team = game_state->next_team;
	game->turn_num = game_state->turn_num;
	game->num_passes = game_state->num_passes;
	game->num_resignations = game_state->num_resignations;
	game->handicap_team = game_state->handicap_team;
	game->handicap_counter = game_state->handicap_counter;
	game->num_dead_strings = game_state->num_dead_strings;
	game->dead_strings = game_state->dead_strings;
	game->num_markup = game_state->num_markup;
	game->markup = game_state->markup;
#ifdef GO_MASK
	game->mask = game_state->mask;
#endif

	return game;
}

go_game_state *go_game_state_from_game(go_game_state *game_state, const go_game *game)
{
	unsigned int i;

	game_state->team_states = (go_team_state *) realloc(game_state->team_states, game->num_players * sizeof(go_team_state));
	if (game_state->team_states == NULL)
		return NULL;
	if (go_board_state_from_board(&game_state->board_state, &game->board) == NULL) {
		free(game_state->team_states);
		return NULL;
	}
	for (i = 0; i < game->num_teams; i++) {
		if (go_team_state_from_team(&game_state->team_states[i], &game->teams[i]) == NULL)
			return NULL;
	}
	game_state->current_team = game->current_team;
	game_state->next_team = game->next_team;
	game_state->turn_num = game->turn_num;
	game_state->num_passes = game->num_passes;
	game_state->num_resignations = game->num_resignations;
	game_state->handicap_team = game->handicap_team;
	game_state->handicap_counter = game->handicap_counter;
	game_state->num_dead_strings = game->num_dead_strings;
	game_state->dead_strings = game->dead_strings;
	game_state->num_markup = game->num_markup;
	game_state->markup = game->markup;
#ifdef GO_MASK
	game_state->mask = game->mask;
#endif

	return game_state;
}

static int go_history_turn_action_handler_type_cmp(const unsigned int *type, const go_history_turn_action_handler *handler)
{
	return *(int *) type - (int) handler->type;
}

static go_history_turn_action_handler *action_handlers_search(unsigned int type)
{
	static unsigned int num = sizeof(action_handlers) / sizeof(*action_handlers);
	static const void *array = action_handlers;

	return (go_history_turn_action_handler *) array_search_sorted(&array, &num, sizeof(go_history_turn_action_handler), &type, (int (*)(const void *, const void *)) go_history_turn_action_handler_type_cmp);
}

static void go_history_turn_action_free(go_history_turn_action *action)
{
	go_history_turn_action_handler *handler;

	handler = action_handlers_search(action->type);
	if ((handler != NULL) && (handler->free != NULL))
		handler->free(action);

	return;
}

static void go_history_turn_action_free_string(go_history_turn_action *action)
{
	free(action->value.string);

	return;
}

static int go_history_turn_action_cmp_integer(const go_history_turn_action *a, const go_history_turn_action *b)
{
	return (int) a->value.integer - (int) b->value.integer;
}

static int go_history_turn_action_cmp_index_color(const go_history_turn_action *a, const go_history_turn_action *b)
{
	int diff;

	diff = (int) a->value.index_color.index - (int) b->value.index_color.index;
	if (diff == 0)
		diff = (int) a->value.index_color.color - (int) a->value.index_color.color;

	return diff;
}

static int go_history_turn_action_cmp_string(const go_history_turn_action *a, const go_history_turn_action *b)
{
	return strcmp(a->value.string, b->value.string);
}

static int go_history_turn_action_cmp_markup(const go_history_turn_action *a, const go_history_turn_action *b)
{
	int diff;

	diff = a->value.markup.type - b->value.markup.type;
	if (diff != 0)
		return diff;
	switch (a->value.markup.type) {
		case GO_GAME_MARKUP_CIRCLE:
		case GO_GAME_MARKUP_SELECTED:
		case GO_GAME_MARKUP_SQUARE:
		case GO_GAME_MARKUP_TRIANGLE:
		case GO_GAME_MARKUP_X:
			diff = (int) a->value.markup.index - (int) b->value.markup.index;
			break;
		case GO_GAME_MARKUP_LINE:
		case GO_GAME_MARKUP_ARROW:
			diff = (int) a->value.markup.index - (int) b->value.markup.index;
			if (diff != 0)
				return 0;
			diff = (int) a->value.markup.data.endpoint - (int) b->value.markup.data.endpoint;
			break;
		case GO_GAME_MARKUP_LABEL:
			diff = (int) a->value.markup.index - (int) b->value.markup.index;
			if (diff != 0)
				return diff;
			diff = strcmp(a->value.markup.data.string, b->value.markup.data.string);
			break;
	}

	return diff;
}

static go_history_turn *go_history_turn_init(go_history_turn *turn)
{
	if (go_game_state_init(&turn->game_state) == NULL)
		return NULL;
	array_init((void **) &turn->actions, &turn->num_actions);
	turn->prev = NULL;
	turn->next = NULL;
	turn->variation = NULL;

	return turn;
}

static void go_history_turn_free(go_history_turn *turn)
{
	go_game_state_free(&turn->game_state);
	array_free((void **) &turn->actions, &turn->num_actions, sizeof(go_history_turn_action), (void (*)(void *)) go_history_turn_action_free);

	return;
}

static void go_history_turn_free_recursively(go_history_turn *turn)
{
	go_history_turn *next;

	for (; turn != NULL; turn = next) {
		go_history_turn_free_recursively(turn->variation);
		next = turn->next;
		go_history_turn_free(turn);
		free(turn);
		turn = next;
	}

	return;
}

static void go_history_hash_table_bucket_free(go_history_hash_table_bucket *bucket)
{
	array_free((void **) &bucket->turns, &bucket->num_turns, sizeof(go_history_turn *), NULL);

	return;
}

go_history *go_history_init(go_history *history)
{
	history->history = NULL;
	history->current_turn = NULL;
	history->hash_table_size = 0;
	history->hash_table_num_nodes = 0;
	history->hash_table = NULL;
	/* This leaks memory if a game is initialized more than once without being
	   freed in between. Try to fix it. */
	go_history_hash_table_resize(history, 1);

	return history;
}

void go_history_free(go_history *history)
{
	go_history_turn_free_recursively(history->history);
	array_free((void **) &history->hash_table, &history->hash_table_size, sizeof(go_history_hash_table_bucket), (void (*)(void *)) go_history_hash_table_bucket_free);

	return;
}

go_history_turn *go_history_new_turn(go_history *history)
{
	go_history_turn *turn, *variation;

	turn = (go_history_turn *) malloc(sizeof(go_history_turn));
	if (turn == NULL)
		return NULL;
	go_history_turn_init(turn);
	if (history->current_turn == NULL) {
		history->history = turn;
		history->current_turn = turn;
	} else {
		if (history->current_turn->next == NULL) {
			history->current_turn->next = turn;
		} else {
			variation = history->current_turn->next;
			while (variation->variation != NULL)
				variation = variation->variation;
			variation->variation = turn;
		}
		turn->prev = history->current_turn;
	}

	return turn;
}

static go_history *go_history_delete_recursively(go_history *history, go_history_turn *turn)
{
	for (; turn != NULL; turn = turn->next) {
		if (go_history_delete_recursively(history, turn->variation) == NULL)
			return NULL;
		if (go_history_hash_table_remove(history, turn) == NULL)
			return NULL;
	}

	return history;
}

go_history *go_history_delete_turn(go_history *history, go_history_turn *turn)
{
	go_history_turn *prev, *variation;

	if (turn == NULL)
		return history;
	prev = turn->prev;
	if (prev != NULL) {
		if (prev->next == turn) {
			prev->next = turn->variation;
		} else {
			variation = prev->next;
			while (variation->variation != turn)
				variation = variation->variation;
			variation->variation = turn->variation;
		}
	}
	turn->variation = NULL;
	/* Remove from the hash table all descendants of this turn */
	if (go_history_delete_recursively(history, turn) == NULL)
		return NULL;
	go_history_turn_free_recursively(turn);

	return history;
}

static void go_history_search(const go_history *history, const void *elem, int (*cmp)(const void *, const go_history_turn *), unsigned int (*hash_func)(const void *), go_history_turn ***array, unsigned int *num)
{
	go_history_turn **turn_ptr, *tmp;
	unsigned int i, hash;

	/* Instead of allocating a new array to hold the turn pointers, we
	   modify the hash table in place and put a slice of one of the
	   buckets in *array and *num */
	array_init((void **) array, num);
	hash = hash_func(elem) % history->hash_table_size;
	for (i = 0; i < history->hash_table[hash].num_turns; i++) {
		turn_ptr = &history->hash_table[hash].turns[i];
		if (cmp(elem, *turn_ptr) == 0) {
			if (*array == NULL) {
				*array = turn_ptr;
			} else if (*array + *num != turn_ptr) {
				tmp = (*array)[*num];
				(*array)[*num] = *turn_ptr;
				*turn_ptr = tmp;
			}
			(*num)++;
		}
	}

	return;
}

static int go_board_turn_cmp(const go_board *board, const go_history_turn *turn)
{
	unsigned int i;
	int diff;

	diff = 0;
	for (i = 0; (i < board->num_points) && (diff == 0); i++)
		diff = (int) board->points[i].color - turn->game_state.board_state.colors[i];

	return diff;
}

static unsigned int go_board_hash(const go_board *board)
{
	unsigned int i, hash;

	hash = 0;
	for (i = 0; i < board->num_points; i++)
		hash = HASH(hash, board->points[i].color);

	return hash;
}

void go_history_search_board(const go_history *history, const go_board *board, go_history_turn ***array, unsigned int *num)
{
	go_history_search(history, board, (int (*)(const void *, const go_history_turn *)) go_board_turn_cmp, (unsigned int (*)(const void *)) go_board_hash, array, num);

	return;
}

static go_history *go_history_hash_table_resize(go_history *history, unsigned int size)
{
	go_history new;
	unsigned int i, j;

	new.hash_table = (go_history_hash_table_bucket *) malloc(size * sizeof(go_history_hash_table_bucket));
	if (new.hash_table == NULL)
		return NULL;
	new.hash_table_num_nodes = 0;
	new.hash_table_size = size;
	for (i = 0; i < new.hash_table_size; i++)
		array_init((void **) &new.hash_table[i].turns, &new.hash_table[i].num_turns);
	for (i = 0; i < history->hash_table_size; i++) {
		for (j = 0; j < history->hash_table[i].num_turns; j++) {
			if (go_history_hash_table_add(&new, history->hash_table[i].turns[j]) == NULL) {
				array_free((void **) &new.hash_table, &new.hash_table_size, sizeof(go_history_hash_table_bucket), (void (*)(void *)) go_history_hash_table_bucket_free);
				return NULL;
			}
		}
	}
	array_free((void **) &history->hash_table, &history->hash_table_size, sizeof(go_history_hash_table_bucket), (void (*)(void *)) go_history_hash_table_bucket_free);
	history->hash_table_size = new.hash_table_size;
	history->hash_table = new.hash_table;

	return history;
}

go_history *go_history_add_action(go_history *history, const go_history_turn_action *action)
{
	if (array_insert((void **) &history->current_turn->actions, &history->current_turn->num_actions, sizeof(go_history_turn_action), (void *) action, NULL) == NULL)
		return NULL;

	return history;
}

go_history *go_history_remove_action(go_history *history, unsigned int index)
{
	go_history_turn_action_free(&history->current_turn->actions[index]);
	if (array_remove_index_keep_order((void **) &history->current_turn->actions, &history->current_turn->num_actions, sizeof(go_history_turn_action), index) == NULL)
		return NULL;

	return history;
}

go_history *go_history_hash_table_add(go_history *history, const go_history_turn *turn)
{
	unsigned int size, hash;

	if ((float) (history->hash_table_num_nodes + 1) / history->hash_table_size >= HASH_TABLE_GROW) {
		if (go_history_hash_table_resize(history, history->hash_table_size * 2) == NULL)
			return NULL;
	}
	hash = go_board_state_hash(&turn->game_state.board_state) % history->hash_table_size;
	size = history->hash_table[hash].num_turns;
	if (array_insert((void **) &history->hash_table[hash].turns, &history->hash_table[hash].num_turns, sizeof(go_history_turn *), &turn, NULL) == NULL)
		return NULL;
	history->hash_table_num_nodes += history->hash_table[hash].num_turns - size;

	return history;
}

static int ptr_cmp(const void *a, const void *b)
{
	return *(char **) a - *(char **) b;
}

go_history *go_history_hash_table_remove(go_history *history, const go_history_turn *turn)
{
	unsigned int size, hash;

	if ((float) (history->hash_table_num_nodes - 1) / history->hash_table_size < HASH_TABLE_SHRINK) {
		if (go_history_hash_table_resize(history, history->hash_table_size / 2) == NULL)
			return NULL;
	}
	hash = go_board_state_hash(&turn->game_state.board_state) % history->hash_table_size;
	size = history->hash_table[hash].num_turns;
	if (array_remove((void **) &history->hash_table[hash].turns, &history->hash_table[hash].num_turns, sizeof(go_history_turn *), &turn, ptr_cmp) == NULL)
		return NULL;
	history->hash_table_num_nodes -= size - history->hash_table[hash].num_turns;

	return history;
}
