/*
	go.c
*/

#include "config.h"

#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <readline/readline.h>

#include "array.h"
#include "game.h"
#include "sgf.h"
#include "interface/interface.h"

#define INTERACTIVE
#define SGF
/*
*/

enum {
	BOARD_WIDTH = 5,
	BOARD_HEIGHT = 5,
	NUM_MOVES = 200,
	NUM_PLAYERS = 4
};

#ifndef SGF
static char *player_names[] = {
	"Black",
	"White",
	"Red",
	"Blue"
};
#endif

static unsigned int parse_move(const go_game *game, const char *s);

static unsigned int parse_move_graph(const char *s, unsigned int size);

static unsigned int parse_move_square(const char *s, unsigned int size);

static unsigned int parse_move_rectangular(const char *s, unsigned int width, unsigned int height);

/*
static int go_player_score_cmp_reverse(const void *a, const void *b);

static void print_result(const go_game *game, FILE *fp);
*/

static int loose_strcmp(const char *s, const char *t);

int main(int argc, char *argv[])
{
	go_game game;
#ifndef SGF
	go_player *player;
#endif
#ifdef INTERACTIVE
	char *buffer;
	unsigned int i;
#endif
	time_t random_seed;

/*
	go_game_load_xml(&game, "test.xml");

	return 0;
*/

	random_seed = time(NULL);
	srand(random_seed);
	return interface_terminal(argc, argv);
	/*
	*/
#ifdef SGF
	if (go_game_load_sgf(&game, "test.sgf") == NULL) {
		fprintf(stderr, "Can't read game from collection.\n");
		return 1;
	}
#else
	if (go_game_init(&game) == NULL) {
		fprintf(stderr, "Can't initialize game.\n");
		return 1;
	}
	if (go_game_board_set_size_rectangular(&game, BOARD_WIDTH, BOARD_HEIGHT) == NULL) {
		fprintf(stderr, "Can't initialize %u-by-%u rectangular board.\n", BOARD_WIDTH, BOARD_HEIGHT);
		return 1;
	}
	for (i = 0; i < NUM_PLAYERS; i++) {
		player = go_game_new_player(&game);
		if (player == NULL) {
			go_game_free(&game);
			fprintf(stderr, "Can't make a new player.\n");
			return 1;
		}
		if (go_player_set_name(player, player_names[i]) == NULL) {
			go_game_free(&game);
			fprintf(stderr, "Can't set player name: %s.\n", player_names[i]);
			return 1;
		}
	}
	go_game_start(&game);
#endif
#ifdef INTERACTIVE
	while (!go_game_is_over(&game)) {
		/*
		go_history_print(&game.history, stdout);
		printf("%u / %u = %.1f%%\n", game.history.hash_table_num_nodes, game.history.hash_table_size, (float) game.history.hash_table_num_nodes * 100 / game.history.hash_table_size);
		go_game_history_print_hash_table(&game.history, stdout);
		go_game_print(&game, stdout);
		*/
		buffer = readline("go> ");
		errno = 0;
		if (errno != 0) {
			perror(NULL);
			exit(1);
		}
		if (buffer == NULL)
			break;
		/*
		printf("go> ");
		if (fgets(buffer, sizeof(buffer) / sizeof(*buffer), stdin) == NULL)
			break;
		*/
		if (loose_strcmp(buffer, "pass") == 0) {
			if ((go_game_new_turn(&game) == NULL) || (go_game_pass(&game) == NULL))
				printf("Illegal pass.\n");
		} else if (loose_strcmp(buffer, "resign") == 0) {
			if ((go_game_new_turn(&game) == NULL) || (go_game_resign(&game) == NULL))
				printf("Illegal resignation.\n");
		} else if (loose_strcmp(buffer, "rewind") == 0) {
			if (go_game_history_rewind(&game) == NULL)
				printf("Can't rewind.\n");
		} else if (loose_strcmp(buffer, "end") == 0) {
			if (go_game_history_end(&game) == NULL)
				printf("Can't end.\n");
		} else if ((loose_strcmp(buffer, "back") == 0) ||
		           (loose_strcmp(buffer, "previous") == 0)) {
			if (go_game_history_back(&game) == NULL)
				printf("Can't back.\n");
		} else if ((loose_strcmp(buffer, "forward") == 0) ||
		           (loose_strcmp(buffer, "next") == 0)) {
			if (go_game_history_forward(&game) == NULL)
				printf("Can't forward.\n");
		} else if (loose_strcmp(buffer, "undo") == 0) {
			if (go_game_history_undo(&game) == NULL)
				printf("Can't undo.\n");
		} else {
			i = parse_move(&game, buffer);
			if (i == (unsigned int) -1) {
				printf("DOES NOT COMPUTE\n");
			} else {
				if ((go_game_new_turn(&game) == NULL) || (go_game_move(&game, i) == NULL))
					printf("Illegal move.\n");
			}
		}
		if (errno != 0) {
			perror(NULL);
			exit(1);
		}
		free(buffer);
	}
	go_game_end(&game);
#else
	/*
	for (i = 0; i < NUM_MOVES; i++) {
		while (go_game_move_turn(&game, rand() % game.board.num_points) == NULL)
			;
		printf("%u\n", i);
	}
	go_game_print(&game, stdout);
	go_game_end(&game);
	*/
#endif
	/*
	printf("Game over.\n");
	go_game_history_print(&game.history, stdout);
	go_game_history_print_hash_table(&game.history, stdout);
	for (i = 0; i < game.num_players; i++)
		go_player_print_score(&game.players[i], stdout);
	print_result(&game, stdout);
	*/
	if (go_game_save_sgf(&game, "board.sgf") == NULL)
		fprintf(stderr, "Can't save SGF.\n");
	if (go_game_save_postscript(&game, "board.ps") == NULL)
		fprintf(stderr, "Can't save PostScript.\n");
	go_game_free(&game);

	return 0;
}

static unsigned int parse_move(const go_game *game, const char *s)
{
	unsigned int index;

	switch(game->board_info.type) {
		case GO_GAME_BOARD_INFO_TYPE_SQUARE:
			index = parse_move_square(s, game->board_info.size.square.size);
			break;
		case GO_GAME_BOARD_INFO_TYPE_RECTANGULAR:
			index = parse_move_rectangular(s, game->board_info.size.rectangular.width, game->board_info.size.rectangular.height);
			break;
		default:
			index = parse_move_graph(s, game->board_info.size.graph.size);
			break;
	}

	return index;
}

static unsigned int parse_move_graph(const char *s, unsigned int size)
{
	unsigned int index;

	if (sscanf(s, "%u", &index) != 1)
		return (unsigned int) -1;
	while (!isalpha(*s) && !isdigit(*s) && (*s != '\0'))
		s++;
	if (*s != '\0')
		return (unsigned int) -1;
	if (index >= size)
		return (unsigned int) -1;

	return index;
}

static unsigned int parse_move_square(const char *s, unsigned int size)
{
	return parse_move_rectangular(s, size, size);
}

static unsigned int parse_move_rectangular(const char *s, unsigned int width, unsigned int height)
{
	unsigned int i, x, y;
	char type[2];
	char c;

	for (i = 0; i < 2; i++) {
		x = y;
		while (!isalpha(*s) && !isdigit(*s) && (*s != '\0'))
			s++;
		if (isalpha(*s)) {
			c = toupper(*s);
			if (c < 'I')
				y = c - 'A';
			else if (c == 'I')
				return (unsigned int) -1;
			else
				y = c - 'A' - 1;
			if (isupper(*s) && (width > 26))
				y += 26;
			s++;
			type[i] = 'A';
		} else if (isdigit(*s)) {
			y = 0;
			while (isdigit(*s) && (*s != '\0')) {
				y *= 10;
				y += *s - '0';
				s++;
			}
			y--;
			type[i] = '1';
		} else {
			return (unsigned int) -1;
		}
	}
	while (!isalpha(*s) && !isdigit(*s) && (*s != '\0'))
		s++;
	if (*s != '\0')
		return (unsigned int) -1;
	if (isdigit(type[0]) && isalpha(type[1])) {
		i = x;
		x = y;
		y = i;
	}
	if ((x >= width) || (y >= height))
		return (unsigned int) -1;

	return (height - y - 1) * width + x;
}

/*
static int go_player_score_cmp_reverse(const void *a, const void *b)
{
	if (((go_player *) a)->resigned && !((go_player *) b)->resigned)
		return 1;
	if (((go_player *) b)->resigned && !((go_player *) a)->resigned)
		return -1;

	return ((go_player *) b)->score - ((go_player *) a)->score;
}

static void print_result(const go_game *game, FILE *fp)
{
	go_player *array;
	unsigned int i;
	float score;

	if (game->game.num_players == 0)
		return;
	array = (go_player *) array_duplicate((void **) &array, NULL, sizeof(go_player), (void *) game->game.players, game->game.num_players);
	if (array == NULL)
		return;
	array_sort((void **) &array, &game->game.num_players, sizeof(go_player), go_player_score_cmp_reverse);
	fprintf(fp, "Result: %s", array[0].name);
	score = array[0].score;
	for (i = 1; i < game->game.num_players; i++) {
		if (array[i].resigned) {
			fprintf(fp, "+R");
			break;
		} else if (score == array[i].score)
			fprintf(fp, ",");
		else
			fprintf(fp, "+%.1f", score - array[i].score);
		if (((i < game->game.num_players - 1) && !array[i].resigned) ||
		    (score == array[i].score))
			fprintf(fp, " %s", array[i].name);
		score = array[i].score;
	}
	free(array);
	putc('\n', fp);

	return;
}
*/

static int loose_strcmp(const char *s, const char *t)
{
	int diff;

	for (;;) {
		while (isspace(*s) && (*s != '\0'))
			s++;
		while (isspace(*t) && (*t != '\0'))
			t++;
		if ((*s == '\0') || (*t == '\0'))
			break;
		while (!isspace(*s) && (*s != '\0') && !isspace(*t) && (*t != '\0')) {
			diff = toupper(*s) - toupper(*t);
			if (diff != 0)
				return diff;
			s++;
			t++;
		}
	}

	return *s - *t;
}
