/*
	game_xml.c
*/

#include "config.h"

#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <iksemel.h>

#include "array.h"
#include "game.h"
#include "ruleset.h"

#define DEFAULT_RULESET (go_ruleset_logical)
/*
#define PRINT_DTD
*/

static const struct {
	char *name;
	const go_ruleset *ruleset;
} ruleset_names[] = {
	{ "aga", &go_ruleset_aga },
	{ "chinese", &go_ruleset_chinese },
	{ "japanese", &go_ruleset_japanese },
	{ "logical", &go_ruleset_logical },
	{ "new zealand", &go_ruleset_new_zealand }
};

typedef struct go_game_load_xml_helper go_game_load_xml_helper;
struct go_game_load_xml_helper {
	go_game *game;
	int game_found;
	unsigned int current_point;
	unsigned int turn_stack_size;
	go_history_turn **turn_stack;
	unsigned int element_stack_size;
	char **element_stack;
};

extern go_history *go_history_remove_action(go_history *history, unsigned int index);

static void strptr_free(char **s);

static go_game_load_xml_helper *go_game_load_xml_helper_init(go_game_load_xml_helper *helper);

static void go_game_load_xml_helper_free(go_game_load_xml_helper *helper);

static go_game_load_xml_helper *go_game_load_xml_helper_turn_push(go_game_load_xml_helper *helper, const go_history_turn *turn);

static go_game_load_xml_helper *go_game_load_xml_helper_turn_pop(go_game_load_xml_helper *helper, go_history_turn **turn);

static go_game_load_xml_helper *go_game_load_xml_helper_element_push(go_game_load_xml_helper *helper, const char *s);

static go_game_load_xml_helper *go_game_load_xml_helper_element_pop(go_game_load_xml_helper *helper, const char **s);

static const char *go_game_load_xml_helper_element_nth(go_game_load_xml_helper *helper, unsigned int n);

static const char *go_game_load_xml_helper_element_current(go_game_load_xml_helper *helper);

static const char *go_game_load_xml_helper_element_parent(go_game_load_xml_helper *helper);

static const char *attributes_search(const char **atts, const char *name);

static go_game_load_xml_helper *go_game_xml_tag_game(go_game_load_xml_helper *helper, const char *name, const char **atts, int type);

static go_game_load_xml_helper *go_game_xml_tag_team(go_game_load_xml_helper *helper, const char *name, const char **atts, int type);

static go_game_load_xml_helper *go_game_xml_tag_player(go_game_load_xml_helper *helper, const char *name, const char **atts, int type);

static go_game_load_xml_helper *go_game_xml_tag_ruleset(go_game_load_xml_helper *helper, const char *name, const char **atts, int type);

static go_game_load_xml_helper *go_game_xml_tag_board(go_game_load_xml_helper *helper, const char *name, const char **atts, int type);

static go_game_load_xml_helper *go_game_xml_tag_point(go_game_load_xml_helper *helper, const char *name, const char **atts, int type);

static go_game_load_xml_helper *go_game_xml_tag_history(go_game_load_xml_helper *helper, const char *name, const char **atts, int type);

static go_game_load_xml_helper *go_game_xml_tag_turn(go_game_load_xml_helper *helper, const char *name, const char **atts, int type);

static unsigned int parse_location(const go_game *game, const char *location);

static go_game_load_xml_helper *go_game_xml_tag_color(go_game_load_xml_helper *helper, const char *name, const char **atts, int type);

static go_game_load_xml_helper *go_game_xml_tag_move(go_game_load_xml_helper *helper, const char *name, const char **atts, int type);

static go_game_load_xml_helper *go_game_xml_tag_pass(go_game_load_xml_helper *helper, const char *name, const char **atts, int type);

static go_game_load_xml_helper *go_game_xml_tag_resign(go_game_load_xml_helper *helper, const char *name, const char **atts, int type);

static go_game_load_xml_helper *go_game_xml_tag_set_player(go_game_load_xml_helper *helper, const char *name, const char **atts, int type);

static go_game_load_xml_helper *go_game_xml_tag_comment(go_game_load_xml_helper *helper, const char *name, const char **atts, int type);

static go_game_load_xml_helper *go_game_xml_tag_dead(go_game_load_xml_helper *helper, const char *name, const char **atts, int type);

static go_game_load_xml_helper *go_game_xml_tag_mark(go_game_load_xml_helper *helper, const char *name, const char **atts, int type);

static go_game_load_xml_helper *go_game_xml_cdata_comment(go_game_load_xml_helper *helper, const char *data, size_t len);

typedef struct xml_tag_handler xml_tag_handler;
struct xml_tag_handler {
	const char *name;
	go_game_load_xml_helper *(*func)(go_game_load_xml_helper *, const char *, const char **, int);
};

typedef struct xml_cdata_handler xml_cdata_handler;
struct xml_cdata_handler {
	const char *name;
	go_game_load_xml_helper *(*func)(go_game_load_xml_helper *, const char *data, size_t len);
};

static const xml_tag_handler tag_handlers[] = {
	{ "game", go_game_xml_tag_game },
	{ "team", go_game_xml_tag_team },
	{ "player", go_game_xml_tag_player },
	{ "ruleset", go_game_xml_tag_ruleset },
	{ "board", go_game_xml_tag_board },
	{ "point", go_game_xml_tag_point },
	{ "history", go_game_xml_tag_history },
	{ "turn", go_game_xml_tag_turn },
	{ "color", go_game_xml_tag_color },
	{ "move", go_game_xml_tag_move },
	{ "pass", go_game_xml_tag_pass },
	{ "resign", go_game_xml_tag_resign },
	{ "set-player", go_game_xml_tag_set_player },
	{ "comment", go_game_xml_tag_comment },
	{ "dead", go_game_xml_tag_dead },
	{ "mark", go_game_xml_tag_mark }
};

static const xml_cdata_handler cdata_handlers[] = {
	{ "comment", go_game_xml_cdata_comment }
};

static const char *search_comma_separated(const char *s, const char *t);

static int xml_tag_handler_name_cmp(const char *name, const xml_tag_handler *handler);

static xml_tag_handler *tag_handlers_search(const char *name);

static int xml_cdata_handler_name_cmp(const char *name, const xml_cdata_handler *handler);

static xml_cdata_handler *cdata_handlers_search(const char *name);

static void go_game_color_print(const go_game *game, const go_history_turn *turn, const go_history_turn_action *action, FILE *fp);

static void go_game_move_print(const go_game *game, const go_history_turn *turn, const go_history_turn_action *action, FILE *fp);

static void go_game_pass_print(const go_game *game, const go_history_turn *turn, const go_history_turn_action *action, FILE *fp);

static void go_game_resign_print(const go_game *game, const go_history_turn *turn, const go_history_turn_action *action, FILE *fp);

static void go_game_set_player_print(const go_game *game, const go_history_turn *turn, const go_history_turn_action *action, FILE *fp);

static void go_game_comment_print(const go_game *game, const go_history_turn *turn, const go_history_turn_action *action, FILE *fp);

static void go_game_dead_print(const go_game *game, const go_history_turn *turn, const go_history_turn_action *action, FILE *fp);

static void go_game_markup_print(const go_game *game, const go_history_turn *turn, const go_history_turn_action *action, FILE *fp);

typedef struct go_history_turn_action_handler go_history_turn_action_handler;
struct go_history_turn_action_handler {
	unsigned int type;
	void (*print)(const go_game *, const go_history_turn *, const go_history_turn_action *, FILE *);
};

static const go_history_turn_action_handler action_handlers[] = {
	{ GO_HISTORY_TURN_ACTION_COLOR_POINT, go_game_color_print },
	{ GO_HISTORY_TURN_ACTION_MOVE, go_game_move_print },
	{ GO_HISTORY_TURN_ACTION_PASS, go_game_pass_print },
	{ GO_HISTORY_TURN_ACTION_RESIGN, go_game_resign_print },
	{ GO_HISTORY_TURN_ACTION_SET_PLAYER, go_game_set_player_print },
	{ GO_HISTORY_TURN_ACTION_COMMENT, go_game_comment_print },
	{ GO_HISTORY_TURN_ACTION_DEAD, go_game_dead_print },
	{ GO_HISTORY_TURN_ACTION_MARKUP, go_game_markup_print }
};

static int go_game_load_xml_tag(void *user_data, char *name, char **atts, int type);

static int go_game_load_xml_cdata(void *user_data, char *data, size_t len);

static void print_xml_escaped(const char *s, FILE *fp);

static void go_game_print_xml(const go_game *game, FILE *fp);

static void go_game_ruleset_print_xml(const go_game *game, FILE *fp);

static void go_game_board_location_print_xml(const go_game *game, unsigned int index, FILE *fp);

static void go_game_board_location_print_xml_graph(const go_game *game, unsigned int index, FILE *fp);

static void go_game_board_location_print_xml_rectangular(const go_game *game, unsigned int index, FILE *fp);

static void (*go_game_board_location_print_xml_square)(const go_game *, unsigned int, FILE *) = go_game_board_location_print_xml_rectangular;

static void go_game_board_print_xml_graph(const go_game *game, FILE *fp);

static void go_game_board_print_xml_square(const go_game *game, FILE *fp);

static void go_game_board_print_xml_rectangular(const go_game *game, FILE *fp);

static void go_game_turn_print_recursively(const go_game *game, const go_history_turn *turn, unsigned int level, FILE *fp);

static void go_game_turn_print(const go_game *game, const go_history_turn *turn, unsigned int level, FILE *fp);

static int go_history_turn_action_handler_type_cmp(unsigned int *type, const go_history_turn_action_handler *handler);

static const go_history_turn_action_handler *action_handlers_search(unsigned int type);

#ifdef PRINT_DTD
static void print_dtd(FILE *fp);
#endif

static void strptr_free(char **s)
{
	free(*s);

	return;
}

static go_game_load_xml_helper *go_game_load_xml_helper_init(go_game_load_xml_helper *helper)
{
	helper->game = NULL;
	helper->game_found = 0;
	helper->current_point = 0;
	array_init((void **) &helper->turn_stack, &helper->turn_stack_size);
	array_init((void **) &helper->element_stack, &helper->element_stack_size);

	return helper;
}

static void go_game_load_xml_helper_free(go_game_load_xml_helper *helper)
{
	array_free((void **) &helper->turn_stack, &helper->turn_stack_size, sizeof(go_history_turn *), NULL);
	array_free((void **) &helper->element_stack, &helper->element_stack_size, sizeof(char *), (void (*)(void *)) strptr_free);

	return;
}

static go_game_load_xml_helper *go_game_load_xml_helper_turn_push(go_game_load_xml_helper *helper, const go_history_turn *turn)
{
	if (array_insert((void **) &helper->turn_stack, &helper->turn_stack_size, sizeof(go_history_turn *), &turn, NULL) == NULL)
		return NULL;

	return helper;
}

static go_game_load_xml_helper *go_game_load_xml_helper_turn_pop(go_game_load_xml_helper *helper, go_history_turn **turn)
{
	if (helper->turn_stack_size == 0)
		return NULL;
	if (turn != NULL)
		*turn = helper->turn_stack[helper->turn_stack_size - 1];
	if (array_remove_index_keep_order((void **) &helper->turn_stack, &helper->turn_stack_size, sizeof(go_history_turn *), helper->turn_stack_size - 1) == NULL)
		return NULL;

	return helper;
}

static go_game_load_xml_helper *go_game_load_xml_helper_element_push(go_game_load_xml_helper *helper, const char *name)
{
	char *p;

	p = (char *) malloc(strlen(name) + 1);
	if (p == NULL)
		return NULL;
	strcpy(p, name);
	if (array_insert((void **) &helper->element_stack, &helper->element_stack_size, sizeof(char *), &p, NULL) == NULL)
		return NULL;

	return helper;
}

static go_game_load_xml_helper *go_game_load_xml_helper_element_pop(go_game_load_xml_helper *helper, const char **name)
{
	char *p;

	if (helper->element_stack_size == 0)
		return NULL;
	p = (char *) go_game_load_xml_helper_element_nth(helper, 0);
	if (name == NULL)
		free(p);
	else
		*name = p;
	if (array_remove_index_keep_order((void **) &helper->element_stack, &helper->element_stack_size, sizeof(char *), helper->element_stack_size - 1) == NULL)
		return NULL;

	return helper;
}

static const char *go_game_load_xml_helper_element_nth(go_game_load_xml_helper *helper, unsigned int n)
{
	if (n >= helper->element_stack_size)
		return NULL;

	return (const char *) helper->element_stack[helper->element_stack_size - n - 1];
}

static const char *go_game_load_xml_helper_element_current(go_game_load_xml_helper *helper)
{
	return go_game_load_xml_helper_element_nth(helper, 0);
}

static const char *go_game_load_xml_helper_element_parent(go_game_load_xml_helper *helper)
{
	return go_game_load_xml_helper_element_nth(helper, 1);
}

static const char *attributes_search(const char **atts, const char *name)
{
	const char **att;

	if (atts != NULL) {
		for (att = atts; *att != NULL; att += 2) {
			if (strcmp(*att, name) == 0)
				return *(att + 1);
		}
	}

	return NULL;
}

static go_game_load_xml_helper *go_game_xml_tag_game(go_game_load_xml_helper *helper, const char *name, const char **atts, int type)
{
	if (type != IKS_OPEN)
		return helper;
	if (go_game_load_xml_helper_element_parent(helper) != NULL) {
		fprintf(stderr, "game element is not the root element.\n");
		return NULL;
	}
	if (helper->game_found) {
		fprintf(stderr, "More than one game element found.\n");
		return NULL;
	}
	helper->game_found = 1;

	return helper;
}

static go_game_load_xml_helper *go_game_xml_tag_team(go_game_load_xml_helper *helper, const char *name, const char **atts, int type)
{
	go_team *team;
	const char *team_name, *tmp;
	unsigned int handicap;

	if (type == IKS_OPEN) {
		if (iks_strcmp(go_game_load_xml_helper_element_parent(helper), "game") != 0) {
			fprintf(stderr, "team element is not a child of game.\n");
			return NULL;
		}
		team = go_game_new_team(helper->game);
		if (team == NULL)
			return NULL;
		team_name = attributes_search(atts, "name");
		if (team_name != NULL) {
			if (go_team_set_name(team, team_name) == NULL)
				return NULL;
		}
		tmp = attributes_search(atts, "handicap");
		if (tmp != NULL) {
			if (sscanf(tmp, "%u%*1s", &handicap) != 1) {
				fprintf(stderr, "Invalid handicap attribute in team element.\n");
				return NULL;
			}
			if (go_game_team_set_handicap(helper->game, team - helper->game->teams, handicap) == NULL)
				return NULL;
		}
		tmp = attributes_search(atts, "komi");
		if (tmp != NULL) {
			if (sscanf(tmp, "%f%*1s", &team->komi) != 1) {
				fprintf(stderr, "Invalid komi attribute in team element.\n");
				return NULL;
			}
		}
	} else if (type == IKS_CLOSE) {
		/* If this team doesn't have any players, remove it */
		team = &helper->game->teams[helper->game->num_teams - 1];
		if (team->num_players == 0) {
			if (go_game_remove_team(helper->game, team - helper->game->teams) == NULL)
				return NULL;
		}
	}

	return helper;
}

static go_game_load_xml_helper *go_game_xml_tag_player(go_game_load_xml_helper *helper, const char *name, const char **atts, int type)
{
	go_player *player;
	const char *player_name;

	if (type == IKS_CLOSE)
		return helper;
	if (iks_strcmp(go_game_load_xml_helper_element_parent(helper), "team") != 0) {
		fprintf(stderr, "player element is not a child of team.\n");
		return NULL;
	}
	if (helper->game->num_teams == 0)
		return NULL;
	player = go_game_new_player_team(helper->game, helper->game->num_teams - 1);
	if (player == NULL)
		return NULL;
	player_name = attributes_search(atts, "name");
	if (player_name != NULL) {
		if (go_player_set_name(player, player_name) == NULL)
			return NULL;
	}

	return helper;
}

static const char *search_comma_separated(const char *s, const char *t)
{
	const char *p, *result;

	for (;;) {
		result = s;
		p = t;
		for (;;) {
			if ((*t == '\0') && ((*s == ',') || (*s == '\0')))
				return result;
			if (*t != *s)
				break;
			s++;
			t++;
		}
		s = result;
		t = p;
		while ((*s != '\0') && (*s != ','))
			s++;
		if (*s == '\0')
			return NULL;
		s++;
	}

	return NULL;
}

static go_game_load_xml_helper *go_game_xml_tag_ruleset(go_game_load_xml_helper *helper, const char *name, const char **atts, int type)
{
	const char *ruleset_name, *tmp;
	unsigned int i;

	if (iks_strcmp(go_game_load_xml_helper_element_parent(helper), "game") != 0) {
		fprintf(stderr, "ruleset element is not a child of game.\n");
		return NULL;
	}
	helper->game->ruleset = DEFAULT_RULESET;
	ruleset_name = attributes_search(atts, "name");
	if (ruleset_name != NULL) {
		for (i = 0; i < sizeof(ruleset_names) / sizeof(*ruleset_names); i++) {
			if (strcmp(ruleset_name, ruleset_names[i].name) == 0) {
				helper->game->ruleset = *ruleset_names[i].ruleset;
				break;
			}
		}
	}
	tmp = attributes_search(atts, "passing");
	if (tmp != NULL) {
		if (strcmp(tmp, "no") == 0) {
			helper->game->ruleset.pass_allowed = 0;
		} else if (strcmp(tmp, "yes") == 0) {
			helper->game->ruleset.pass_allowed = 1;
		} else {
			fprintf(stderr, "Bad passing attribute in ruleset element.\n");
			return NULL;
		}
	}
	tmp = attributes_search(atts, "suicide");
	if (tmp != NULL) {
		if (strcmp(tmp, "no") == 0) {
			helper->game->ruleset.suicide_allowed = 0;
		} else if (strcmp(tmp, "yes") == 0) {
			helper->game->ruleset.suicide_allowed = 1;
		} else {
			fprintf(stderr, "Bad suicide attribute in ruleset element.\n");
			return NULL;
		}
	}
	tmp = attributes_search(atts, "ko");
	if (tmp != NULL) {
		if (strcmp(tmp, "none") == 0) {
			helper->game->ruleset.ko = GO_RULESET_KO_NONE;
		} else if (strcmp(tmp, "simple") == 0) {
			helper->game->ruleset.ko = GO_RULESET_KO_SIMPLE;
		} else if (strcmp(tmp, "positional-superko") == 0) {
			helper->game->ruleset.ko = GO_RULESET_KO_POSITIONAL_SUPERKO;
		} else if (strcmp(tmp, "situational-superko") == 0) {
			helper->game->ruleset.ko = GO_RULESET_KO_SITUATIONAL_SUPERKO;
		} else {
			fprintf(stderr, "Bad ko attribute in ruleset element.\n");
			return NULL;
		}
	}
	tmp = attributes_search(atts, "handicap");
	if (tmp != NULL) {
		if (strcmp(tmp, "free") == 0) {
			helper->game->ruleset.handicap_placement = GO_RULESET_HANDICAP_FREE;
		} else if (strcmp(tmp, "fixed") == 0) {
			helper->game->ruleset.handicap_placement = GO_RULESET_HANDICAP_FIXED;
		} else {
			fprintf(stderr, "Bad handicap attribute in ruleset element.\n");
			return NULL;
		}
	}
	tmp = attributes_search(atts, "score");
	if (tmp != NULL) {
		if (search_comma_separated(tmp, "territory") == NULL)
			helper->game->ruleset.score_territory = 0;
		else
			helper->game->ruleset.score_territory = 1;
		if (search_comma_separated(tmp, "stones") == NULL)
			helper->game->ruleset.score_occupied_points = 0;
		else
			helper->game->ruleset.score_occupied_points = 1;
		if (search_comma_separated(tmp, "captures") == NULL)
			helper->game->ruleset.score_captures = 0;
		else
			helper->game->ruleset.score_captures = 1;
		if (search_comma_separated(tmp, "passes") == NULL)
			helper->game->ruleset.score_passes = 0;
		else
			helper->game->ruleset.score_passes = 1;
	}
	tmp = attributes_search(atts, "play-on-colored-points");
	if (tmp != NULL) {
		if (strcmp(tmp, "no") == 0) {
			helper->game->ruleset.play_on_colored_points = 0;
		} else if (strcmp(tmp, "yes") == 0) {
			helper->game->ruleset.play_on_colored_points = 1;
		} else {
			fprintf(stderr, "Bad play-on-colored-points attribute in ruleset element.\n");
			return NULL;
		}
	}
	tmp = attributes_search(atts, "simultaneous-capture");
	if (tmp != NULL) {
		if (strcmp(tmp, "no") == 0) {
			helper->game->ruleset.simultaneous_capture = 0;
		} else if (strcmp(tmp, "yes") == 0) {
			helper->game->ruleset.simultaneous_capture = 1;
		} else {
			fprintf(stderr, "Bad simultaneous-capture attribute in ruleset element.\n");
			return NULL;
		}
	}
	tmp = attributes_search(atts, "othello-capture");
	if (tmp != NULL) {
		if (strcmp(tmp, "no") == 0) {
			helper->game->ruleset.othello_capture = 0;
		} else if (strcmp(tmp, "yes") == 0) {
			helper->game->ruleset.othello_capture = 1;
		} else {
			fprintf(stderr, "Bad othello-capture attribute in ruleset element.\n");
			return NULL;
		}
	}

	return helper;
}

static go_game_load_xml_helper *go_game_xml_tag_board(go_game_load_xml_helper *helper, const char *name, const char **atts, int type)
{
	unsigned int size, width, height;
	const char *board_type, *tmp;

	if (type == IKS_CLOSE)
		return helper;
	if (iks_strcmp(go_game_load_xml_helper_element_parent(helper), "game") != 0) {
		fprintf(stderr, "board element is not a child of game.\n");
		return NULL;
	}
	board_type = attributes_search(atts, "type");
	if (board_type == NULL) {
		fprintf(stderr, "No type attribute in board element.\n");
		return NULL;
	}
	if (strcmp(board_type, "graph") == 0)
		helper->current_point = 0;
	if ((strcmp(board_type, "graph") == 0) || (strcmp(board_type, "square") == 0)) {
		tmp = attributes_search(atts, "size");
		if (tmp == NULL) {
			fprintf(stderr, "No size given for board type %s.\n", board_type);
			return NULL;
		}
		if (sscanf(tmp, "%u%*1s", &size) != 1) {
			fprintf(stderr, "Invalid size given for board type %s.\n", board_type);
			return NULL;
		}
		if (strcmp(board_type, "graph") == 0) {
			if (go_game_board_set_size_graph(helper->game, size) == NULL)
				return NULL;
		} else {
			if (go_game_board_set_size_square(helper->game, size) == NULL)
				return NULL;
		}
	} else if (strcmp(board_type, "rectangular") == 0) {
		tmp = attributes_search(atts, "width");
		if (tmp == NULL) {
			fprintf(stderr, "No width given for board type %s.\n", board_type);
			return NULL;
		}
		if (sscanf(tmp, "%u%*1s", &width) != 1) {
			fprintf(stderr, "Invalid width given for board type %s.\n", board_type);
			return NULL;
		}
		tmp = attributes_search(atts, "height");
		if (tmp == NULL) {
			fprintf(stderr, "No height given for board type %s.\n", board_type);
			return NULL;
		}
		if (sscanf(tmp, "%u%*1s", &height) != 1) {
			fprintf(stderr, "Invalid height given for board type %s.\n", board_type);
			return NULL;
		}
		if (go_game_board_set_size_rectangular(helper->game, width, height) == NULL)
			return NULL;
	} else {
		fprintf(stderr, "Bad type attribute in board element.\n");
		return NULL;
	}

	return helper;
}

static go_game_load_xml_helper *go_game_xml_tag_point(go_game_load_xml_helper *helper, const char *name, const char **atts, int type)
{
	const char *connections, *tmp;
	float x, y, z;
	unsigned int index;

	if (type == IKS_CLOSE)
		return helper;
	if (iks_strcmp(go_game_load_xml_helper_element_parent(helper), "board") != 0) {
		fprintf(stderr, "point element is not a child of board.\n");
		return NULL;
	}
	if (helper->game->board_info.type != GO_GAME_BOARD_INFO_TYPE_GRAPH) {
		fprintf(stderr, "point element is not allowed for a board type other than graph.\n");
		return NULL;
	}
	if (helper->current_point >= helper->game->board.num_points) {
		fprintf(stderr, "Too many point children of board element.\n");
		return NULL;
	}
	connections = attributes_search(atts, "connections");
	if (connections != NULL) {
		while (*connections != '\0') {
			if ((sscanf(connections, "%u,", &index) != 1) ||
			    (sscanf(connections, "%u%*1s", &index) != 1)) {
				fprintf(stderr, "Bad connections attribute in point element.\n");
				return NULL;
			}
			if (go_board_connect_points(&helper->game->board, helper->current_point, index) == NULL)
				return NULL;
			while (isdigit(*connections))
				connections++;
			while (*connections == ',')
				connections++;
		}
	}
	tmp = attributes_search(atts, "star");
	if (tmp != NULL) {
		if (strcmp(tmp, "yes") == 0) {
			if (go_game_board_add_star_point(helper->game, helper->current_point) == NULL)
				return NULL;
		} else if (strcmp(tmp, "no") != 0) {
			fprintf(stderr, "Bad star attribute in point element.\n");
			return NULL;
		}
	}
	x = 0.0;
	y = 0.0;
	z = 0.0;
	tmp = attributes_search(atts, "x");
	if ((tmp != NULL) && (sscanf(tmp, "%f%*1s", &x) != 1)) {
		fprintf(stderr, "Bad x attribute in color element.\n");
		return NULL;
	}
	tmp = attributes_search(atts, "y");
	if ((tmp != NULL) && (sscanf(tmp, "%f%*1s", &y) != 1)) {
		fprintf(stderr, "Bad y attribute in color element.\n");
		return NULL;
	}
	tmp = attributes_search(atts, "z");
	if ((tmp != NULL) && (sscanf(tmp, "%f%*1s", &z) != 1)) {
		fprintf(stderr, "Bad z attribute in color element.\n");
		return NULL;
	}
	if (go_game_board_set_point_location(helper->game, helper->current_point, x, y, z) == NULL)
		return NULL;
	helper->current_point++;

	return helper;
}

static go_game_load_xml_helper *go_game_xml_tag_history(go_game_load_xml_helper *helper, const char *name, const char **atts, int type)
{
	go_history_turn *turn;

	if ((iks_strcmp(go_game_load_xml_helper_element_parent(helper), "game") != 0) &&
	    (iks_strcmp(go_game_load_xml_helper_element_parent(helper), "history") != 0)) {
		fprintf(stderr, "history element is not a child of game or history.\n");
		return NULL;
	}
	if (type == IKS_OPEN) {
		/* Save the current turn. Before the game starts (at the root history),
		   it will be NULL. */
		if (go_game_load_xml_helper_turn_push(helper, helper->game->history.current_turn) == NULL)
			return NULL;
		if (helper->game->history.current_turn != NULL) {
			if (helper->game->history.current_turn->prev == NULL) {
				fprintf(stderr, "Variations of the root turn are not allowed.\n");
				return NULL;
			}
			/* Step back one turn in preparation for adding variations */
			if (go_game_history_back(helper->game) == NULL)
				return NULL;
		}
	} else if (type == IKS_CLOSE) {
		if (go_game_load_xml_helper_turn_pop(helper, &turn) == NULL)
			return NULL;
		if (turn != NULL) {
			if (go_game_history_seek(helper->game, turn) == NULL)
				return NULL;
		} else {
			if (go_game_history_rewind(helper->game) == NULL)
				return NULL;
		}
	}

	return helper;
}

static go_game_load_xml_helper *go_game_xml_tag_turn(go_game_load_xml_helper *helper, const char *name, const char **atts, int type)
{
	if (iks_strcmp(go_game_load_xml_helper_element_parent(helper), "history") != 0) {
		fprintf(stderr, "turn element is not a child of history.\n");
		return NULL;
	}
	if ((type == IKS_OPEN) ||
	    ((type == IKS_SINGLE) && (helper->game->history.current_turn == NULL))) {
		if (go_game_new_turn(helper->game) == NULL)
			return NULL;
	} else if (type == IKS_CLOSE) {
		/* Delete the turn if it's empty, unless it's the root */
		if ((helper->game->history.current_turn->num_actions == 0) &&
		    (helper->game->history.current_turn->prev != NULL)) {
			if (go_game_history_undo(helper->game) == NULL)
				return NULL;
		}
	}

	return helper;
}

static unsigned int parse_location(const go_game *game, const char *location)
{
	static char c[2];
	unsigned int x, y, width, height, index;

	index = (unsigned int) -1;
	if ((game->board_info.type == GO_GAME_BOARD_INFO_TYPE_SQUARE) ||
	    (game->board_info.type == GO_GAME_BOARD_INFO_TYPE_RECTANGULAR)) {
		if (game->board_info.type == GO_GAME_BOARD_INFO_TYPE_SQUARE) {
			width = game->board_info.size.square.size;
			height = game->board_info.size.square.size;
		} else {
			width = game->board_info.size.rectangular.width;
			height = game->board_info.size.rectangular.height;
		}
		if (sscanf(location, "%1s%u%*1s", c, &y) != 2)
			goto graph;
		if (!isupper(*c))
			goto graph;
		if (*c < 'I')
			x = *c - 'A';
		else if (*c == 'I')
			goto graph;
		else
			x = *c - 'A' - 1;
		if ((x >= width) || (y == 0) || (y > height))
			goto graph;
		y = height - y;
		index = y * width + x;
	}
	graph:
	if ((game->board_info.type == GO_GAME_BOARD_INFO_TYPE_GRAPH) ||
	    (index == (unsigned int) -1)) {
		if (sscanf(location, "%u%*1s", &index) != 1)
			index = (unsigned int) -1;
	}

	return index;
}

static go_game_load_xml_helper *go_game_xml_tag_color(go_game_load_xml_helper *helper, const char *name, const char **atts, int type)
{
	const char *location, *tmp;
	unsigned int index, color;

	if (iks_strcmp(go_game_load_xml_helper_element_parent(helper), "turn") != 0) {
		fprintf(stderr, "color element is not a child of turn.\n");
		return NULL;
	}
	if (type == IKS_CLOSE)
		return helper;
	location = attributes_search(atts, "location");
	if (location == NULL) {
		fprintf(stderr, "No location attribute in color element.\n");
		return NULL;
	}
	index = parse_location(helper->game, location);
	if (index == (unsigned int) -1) {
		fprintf(stderr, "Bad location attribute in color element.\n");
		return NULL;
	}
	tmp = attributes_search(atts, "color");
	if (tmp == NULL) {
		fprintf(stderr, "No color attribute in color element.\n");
		return NULL;
	}
	if ((sscanf(tmp, "%u%*1s", &color) != 1) ||
	    (color > helper->game->num_teams)) {
		fprintf(stderr, "Bad color attribute in color element.\n");
		return NULL;
	}
	if (go_game_color_point(helper->game, index, color) == NULL)
		return NULL;

	return helper;
}

static go_game_load_xml_helper *go_game_xml_tag_move(go_game_load_xml_helper *helper, const char *name, const char **atts, int type)
{
	const char *location;
	unsigned int index;

	if (iks_strcmp(go_game_load_xml_helper_element_parent(helper), "turn") != 0) {
		fprintf(stderr, "move element is not a child of turn.\n");
		return NULL;
	}
	if (type == IKS_CLOSE)
		return helper;
	location = attributes_search(atts, "location");
	if (location == NULL) {
		fprintf(stderr, "No location attribute in move element.\n");
		return NULL;
	}
	index = parse_location(helper->game, location);
	if (index == (unsigned int) -1) {
		fprintf(stderr, "Bad location attribute in move element.\n");
		return NULL;
	}
	if (go_game_move(helper->game, index) == NULL)
		return NULL;

	return helper;
}

static go_game_load_xml_helper *go_game_xml_tag_pass(go_game_load_xml_helper *helper, const char *name, const char **atts, int type)
{
	if (iks_strcmp(go_game_load_xml_helper_element_parent(helper), "turn") != 0) {
		fprintf(stderr, "pass element is not a child of turn.\n");
		return NULL;
	}
	if (type == IKS_CLOSE)
		return helper;
	if (go_game_pass(helper->game) == NULL)
		return NULL;

	return helper;
}

static go_game_load_xml_helper *go_game_xml_tag_resign(go_game_load_xml_helper *helper, const char *name, const char **atts, int type)
{
	if (iks_strcmp(go_game_load_xml_helper_element_parent(helper), "turn") != 0) {
		fprintf(stderr, "resign element is not a child of turn.\n");
		return NULL;
	}
	if (type == IKS_CLOSE)
		return helper;
	if (go_game_resign(helper->game) == NULL)
		return NULL;

	return helper;
}

static go_game_load_xml_helper *go_game_xml_tag_set_player(go_game_load_xml_helper *helper, const char *name, const char **atts, int type)
{
	const char *tmp;
	unsigned int team, player;

	if (iks_strcmp(go_game_load_xml_helper_element_parent(helper), "turn") != 0) {
		fprintf(stderr, "set-player element is not a child of turn.\n");
		return NULL;
	}
	if (type == IKS_CLOSE)
		return helper;
	tmp = attributes_search(atts, "team");
	if (tmp == NULL) {
		fprintf(stderr, "No team attribute in color element.\n");
		return NULL;
	}
	if ((sscanf(tmp, "%u%*1s", &team) != 1) ||
	    (team >= helper->game->num_teams)) {
		fprintf(stderr, "Bad team attribute in color element.\n");
		return NULL;
	}
	tmp = attributes_search(atts, "player");
	if (tmp == NULL) {
		fprintf(stderr, "No player attribute in color element.\n");
		return NULL;
	}
	if ((sscanf(tmp, "%u%*1s", &player) != 1) ||
	    (player >= helper->game->teams[team].num_players)) {
		fprintf(stderr, "Bad player attribute in color element.\n");
		return NULL;
	}
	if (go_game_set_current_player(helper->game, team, player) == NULL)
		return NULL;

	return helper;
}

static go_game_load_xml_helper *go_game_xml_tag_comment(go_game_load_xml_helper *helper, const char *name, const char **atts, int type)
{
	go_history_turn_action *action;

	if (iks_strcmp(go_game_load_xml_helper_element_parent(helper), "turn") != 0) {
		fprintf(stderr, "comment element is not a child of turn.\n");
		return NULL;
	}
	if (type == IKS_OPEN) {
		/* Start a blank comment, which will be added to in
		   go_game_xml_cdata_comment */
		if (go_game_comment(helper->game, "") == NULL)
			return NULL;
	} else if (type == IKS_CLOSE) {
		/* If the comment is still blank, remove it */
		action = &helper->game->history.current_turn->actions[helper->game->history.current_turn->num_actions - 1];
		if (strcmp(action->value.string, "") == 0) {
			if (go_history_remove_action(&helper->game->history, action - helper->game->history.current_turn->actions) == NULL)
				return NULL;
		}
	}

	return helper;
}

static go_game_load_xml_helper *go_game_xml_tag_dead(go_game_load_xml_helper *helper, const char *name, const char **atts, int type)
{
	const char *location;
	unsigned int index;

	if (iks_strcmp(go_game_load_xml_helper_element_parent(helper), "turn") != 0) {
		fprintf(stderr, "dead element is not a child of turn.\n");
		return NULL;
	}
	if (type == IKS_CLOSE)
		return helper;
	location = attributes_search(atts, "location");
	if (location == NULL) {
		fprintf(stderr, "No location attribute in dead element.\n");
		return NULL;
	}
	index = parse_location(helper->game, location);
	if (index == (unsigned int) -1) {
		fprintf(stderr, "Bad location attribute in dead element.\n");
		return NULL;
	}
	if (go_game_toggle_dead(helper->game, index) == NULL)
		return NULL;

	return helper;
}

static go_game_load_xml_helper *go_game_xml_tag_mark(go_game_load_xml_helper *helper, const char *name, const char **atts, int type)
{
	const char *type_string, *location, *endpoint, *text;
	unsigned int markup_type, index, endpoint_index;

	if (iks_strcmp(go_game_load_xml_helper_element_parent(helper), "turn") != 0) {
		fprintf(stderr, "mark element is not a child of turn.\n");
		return NULL;
	}
	if (type == IKS_CLOSE)
		return helper;
	type_string = attributes_search(atts, "type");
	if (type_string == NULL) {
		fprintf(stderr, "No type attribute in mark element.\n");
		return NULL;
	}
	if (strcmp(type_string, "clear") == 0) {
		markup_type = GO_GAME_MARKUP_CLEAR;
	} else {
		location = attributes_search(atts, "location");
		if (location == NULL) {
			fprintf(stderr, "No location attribute in mark element.\n");
			return NULL;
		}
		index = parse_location(helper->game, location);
		if (index == (unsigned int) -1) {
			fprintf(stderr, "Bad location attribute in mark element.\n");
			return NULL;
		}
		if (strcmp(type_string, "circle") == 0) {
			markup_type = GO_GAME_MARKUP_CIRCLE;
		} else if (strcmp(type_string, "selected") == 0) {
			markup_type = GO_GAME_MARKUP_SELECTED;
		} else if (strcmp(type_string, "square") == 0) {
			markup_type = GO_GAME_MARKUP_SQUARE;
		} else if (strcmp(type_string, "triangle") == 0) {
			markup_type = GO_GAME_MARKUP_TRIANGLE;
		} else if (strcmp(type_string, "x") == 0) {
			markup_type = GO_GAME_MARKUP_X;
		} else if ((strcmp(type_string, "line") == 0) ||
	           (strcmp(type_string, "arrow") == 0)) {
			if (strcmp(type_string, "line") == 0)
				markup_type = GO_GAME_MARKUP_LINE;
			else if (strcmp(type_string, "arrow") == 0)
				markup_type = GO_GAME_MARKUP_ARROW;
			endpoint = attributes_search(atts, "endpoint");
			if (location == NULL) {
				fprintf(stderr, "No endpoint attribute in mark element.\n");
				return NULL;
			}
			endpoint_index = parse_location(helper->game, endpoint);
			if (endpoint_index == (unsigned int) -1) {
				fprintf(stderr, "Bad endpoint attribute in mark element.\n");
				return NULL;
			}
		} else if (strcmp(type_string, "label") == 0) {
			markup_type = GO_GAME_MARKUP_LABEL;
			text = attributes_search(atts, "text");
			if (text == NULL) {
				fprintf(stderr, "No text attribute in mark element.\n");
				return NULL;
			}
		} else {
			fprintf(stderr, "Bad type attribute in mark element.\n");
			return NULL;
		}
	}
	switch (markup_type) {
		case GO_GAME_MARKUP_CLEAR:
			if (go_game_clear_markup(helper->game) == NULL)
				return NULL;
			break;
		case GO_GAME_MARKUP_CIRCLE:
		case GO_GAME_MARKUP_SELECTED:
		case GO_GAME_MARKUP_SQUARE:
		case GO_GAME_MARKUP_TRIANGLE:
		case GO_GAME_MARKUP_X:
			if (go_game_mark(helper->game, index, markup_type) == NULL)
				return NULL;
			break;
		case GO_GAME_MARKUP_LINE:
		case GO_GAME_MARKUP_ARROW:
			if (go_game_line(helper->game, index, endpoint_index, markup_type) == NULL)
				return NULL;
			break;
		case GO_GAME_MARKUP_LABEL:
			if (go_game_label(helper->game, index, text) == NULL)
				return NULL;
			break;
	}

	return helper;
}

static go_game_load_xml_helper *go_game_xml_cdata_comment(go_game_load_xml_helper *helper, const char *data, size_t len)
{
	go_history_turn_action *action;
	char *p, *end;

	action = &helper->game->history.current_turn->actions[helper->game->history.current_turn->num_actions - 1];
	if (action->type != GO_HISTORY_TURN_ACTION_COMMENT)
		return NULL;
	/* Find the end of the string */
	end = strchr(action->value.string, '\0');
	p = (char *) realloc(action->value.string, end - action->value.string + len + 1);
	if (p == NULL)
		return NULL;
	end += p - action->value.string;
	action->value.string = p;
	memmove(end, data, len);
	end[len] = '\0';

	return helper;
}

static int go_game_load_xml_tag(void *user_data, char *name, char **atts, int type)
{
	go_game_load_xml_helper *helper;
	xml_tag_handler *handler;
	char **att;

	if (type == IKS_CLOSE)
		printf("</%s", name);
	else
		printf("<%s", name);
	if (atts != NULL) {
		for (att = atts; *att != NULL; att += 2)
			printf(" %s=\"%s\"", *att, *(att + 1));
	}
	if (type == IKS_SINGLE)
		printf("/>");
	else
		printf(">");

	helper = (go_game_load_xml_helper *) user_data;
	handler = tag_handlers_search(name);
	if ((type == IKS_OPEN) || (type == IKS_SINGLE)) {
		if (go_game_load_xml_helper_element_push(helper, name) == NULL)
			return IKS_HOOK;
	}
	if ((handler != NULL) && (handler->func != NULL)) {
		if (handler->func(helper, name, (const char **) atts, type) == NULL)
			return IKS_HOOK;
	}
	if ((type == IKS_CLOSE) || (type == IKS_SINGLE)) {
		if (go_game_load_xml_helper_element_pop(helper, NULL) == NULL)
			return IKS_HOOK;
	}

	return IKS_OK;
}

static int go_game_load_xml_cdata(void *user_data, char *data, size_t len)
{
	go_game_load_xml_helper *helper;
	xml_cdata_handler *handler;
	const char *element;

	fwrite(data, sizeof(char), len, stdout);

	helper = (go_game_load_xml_helper *) user_data;
	element = go_game_load_xml_helper_element_current(helper);
	if (element != NULL) {
		handler = cdata_handlers_search(element);
		if ((handler != NULL) && (handler->func != NULL)) {
			if (handler->func(helper, data, len) == NULL)
				return IKS_HOOK;
		}
	}

	return IKS_OK;
}

go_game *go_game_load_xml(go_game *game, const char *filename)
{
	static char buffer[1024];
	FILE *fp;
	iksparser *parser;
	go_game *result;
	go_game_load_xml_helper helper;
	int parser_result, is_eof;
	size_t length;

	if (go_game_init(game) == NULL)
		return NULL;
	game->ruleset = DEFAULT_RULESET;
	if (go_game_load_xml_helper_init(&helper) == NULL)
		return NULL;
	helper.game = game;
	parser = iks_sax_new((void *) &helper, go_game_load_xml_tag, go_game_load_xml_cdata);
	if (parser == NULL)
		return NULL;
	fp = fopen(filename, "rb");
	if (fp == NULL) {
		iks_parser_delete(parser);
		return NULL;
	}
	for (;;) {
		length = fread(buffer, sizeof(char), sizeof(buffer) / sizeof(*buffer), fp);
		if (errno != 0) {
			printf("Read error.\n");
			go_game_free(game);
			break;
		}
		is_eof = feof(fp);
		if (length == 0)
			parser_result = iks_parse(parser, NULL, 0, is_eof);
		else
			parser_result = iks_parse(parser, buffer, length, 0);
		if (parser_result != IKS_OK) {
			printf("Parse error.\n");
			result = NULL;
			break;
		}
		if (is_eof) {
			result = game;
			break;
		}
	}
	fclose(fp);
	if (!helper.game_found) {
		go_game_free(game);
		result = NULL;
	}
	go_game_load_xml_helper_free(&helper);
	iks_parser_delete(parser);

	return result;
}

static int xml_tag_handler_name_cmp(const char *name, const xml_tag_handler *handler)
{
	return strcmp(name, handler->name);
}

static xml_tag_handler *tag_handlers_search(const char *name)
{
	static const void *array = (void *) tag_handlers;
	static unsigned int num = sizeof(tag_handlers) / sizeof(*tag_handlers);

	return (xml_tag_handler *) array_search((const void **) &array, &num, sizeof(xml_tag_handler), name, (int (*)(const void *, const void *)) xml_tag_handler_name_cmp);
}

static int xml_cdata_handler_name_cmp(const char *name, const xml_cdata_handler *handler)
{
	return strcmp(name, handler->name);
}

static xml_cdata_handler *cdata_handlers_search(const char *name)
{
	static const void *array = (void *) cdata_handlers;
	static unsigned int num = sizeof(cdata_handlers) / sizeof(*cdata_handlers);

	return (xml_cdata_handler *) array_search((const void **) &array, &num, sizeof(xml_cdata_handler), name, (int (*)(const void *, const void *)) xml_cdata_handler_name_cmp);
}

static void go_game_color_print(const go_game *game, const go_history_turn *turn, const go_history_turn_action *action, FILE *fp)
{
	fprintf(fp, "<color location=\"");
	go_game_board_location_print_xml(game, action->value.index_color.index, fp);
	fprintf(fp, "\" color=\"%u\"/>", action->value.index_color.color);

	return;
}

static void go_game_move_print(const go_game *game, const go_history_turn *turn, const go_history_turn_action *action, FILE *fp)
{
	fprintf(fp, "<move location=\"");
	go_game_board_location_print_xml(game, action->value.integer, fp);
	fprintf(fp, "\"/>");

	return;
}

static void go_game_pass_print(const go_game *game, const go_history_turn *turn, const go_history_turn_action *action, FILE *fp)
{
	fprintf(fp, "<pass/>");

	return;
}

static void go_game_resign_print(const go_game *game, const go_history_turn *turn, const go_history_turn_action *action, FILE *fp)
{
	fprintf(fp, "<resign/>");

	return;
}

static void go_game_set_player_print(const go_game *game, const go_history_turn *turn, const go_history_turn_action *action, FILE *fp)
{
	fprintf(fp, "<set-player team=\"%u\" player=\"%u\"/>", action->value.player.team_index, action->value.player.player_index);

	return;
}

static void go_game_comment_print(const go_game *game, const go_history_turn *turn, const go_history_turn_action *action, FILE *fp)
{
	fprintf(fp, "<comment>");
	print_xml_escaped(action->value.string, fp);
	fprintf(fp, "</comment>");

	return;
}

static void go_game_dead_print(const go_game *game, const go_history_turn *turn, const go_history_turn_action *action, FILE *fp)
{
	fprintf(fp, "<dead location=\"");
	go_game_board_location_print_xml(game, action->value.integer, fp);
	fprintf(fp, "\"/>");

	return;
}

static void go_game_markup_print(const go_game *game, const go_history_turn *turn, const go_history_turn_action *action, FILE *fp)
{
	static const struct {
		unsigned int type;
		char *string;
	} types[] = {
		{ GO_GAME_MARKUP_CLEAR, "clear" },
		{ GO_GAME_MARKUP_CIRCLE, "circle" },
		{ GO_GAME_MARKUP_SELECTED, "selected" },
		{ GO_GAME_MARKUP_SQUARE, "square" },
		{ GO_GAME_MARKUP_TRIANGLE, "triangle" },
		/* { GO_GAME_MARKUP_TERRITORY, "territory" }, */
		{ GO_GAME_MARKUP_X, "x" },
		{ GO_GAME_MARKUP_LINE, "line" },
		{ GO_GAME_MARKUP_ARROW, "arrow" },
		{ GO_GAME_MARKUP_LABEL, "label" }
	};
	unsigned int i;

	for (i = 0; i < sizeof(types) / sizeof(*types); i++) {
		if (action->value.markup.type == types[i].type)
			break;
	}
	if (i >= sizeof(types) / sizeof(*types))
		return;
	fprintf(fp, "<mark type=\"%s\"", types[i].string);
	if (action->value.markup.type != GO_GAME_MARKUP_CLEAR) {
		fprintf(fp, " location=\"");
		go_game_board_location_print_xml(game, action->value.markup.index, fp);
		fprintf(fp, "\"");
		if ((action->value.markup.type == GO_GAME_MARKUP_LINE) ||
		    (action->value.markup.type == GO_GAME_MARKUP_ARROW)) {
			fprintf(fp, " endpoint=\"");
			go_game_board_location_print_xml(game, action->value.markup.data.endpoint, fp);
			fprintf(fp, "\"");
		} else if (action->value.markup.type == GO_GAME_MARKUP_LABEL) {
			fprintf(fp, " text=\"");
			print_xml_escaped(action->value.markup.data.string, fp);
			fprintf(fp, "\"");
		}
	}
	fprintf(fp, "/>");

	return;
}

const go_game *go_game_save_xml(const go_game *game, const char *filename)
{
	FILE *fp;

	fp = fopen(filename, "wb");
	if (fp == NULL)
		return NULL;
	fprintf(fp, "<?xml version=\"1.0\"?>\n");
	putc('\n', fp);
#ifdef PRINT_DTD
	print_dtd(fp);
	putc('\n', fp);
#endif
	go_game_print_xml(game, fp);
	if (fclose(fp) == EOF)
		return NULL;

	return game;
}

static void print_xml_escaped(const char *s, FILE *fp)
{
	const char *escapes = "&<>\"'";
	static const struct {
		char c;
		const char *replacement;
	} replacements[] = {
		{ '&', "&amp;" },
		{ '<', "&lt;" },
		{ '>', "&gt;" },
		{ '"', "&quot;" },
		{ '\'', "&apos;" }
	};
	size_t length;
	unsigned int i;

	for (;;) {
		length = strcspn(s, escapes);
		if (length > 0)
			fwrite(s, sizeof(char), length, fp);
		s += length;
		if (*s == '\0')
			break;
		for (i = 0; i < sizeof(replacements) / sizeof(*replacements); i++) {
			if (replacements[i].c == *s) {
				fprintf(fp, "%s", replacements[i].replacement);
				break;
			}
		}
		s++;
	}

	return;
}

static void go_game_print_xml(const go_game *game, FILE *fp)
{
	unsigned int i, j;

	fprintf(fp, "<game>\n");
	for (j = 0; j < game->num_teams; j++) {
		if (game->teams[j].num_players == 0)
			continue;
		fprintf(fp, "\t<team");
		if (game->teams[j].name != NULL) {
			fprintf(fp, " name=\"");
			print_xml_escaped(game->teams[j].name, fp);
			fprintf(fp, "\"");
		}
		if (game->teams[j].handicap > 0)
			fprintf(fp, " handicap=\"%u\"", game->teams[j].handicap);
		if (game->teams[j].komi != 0.0)
			fprintf(fp, " komi=\"%g\"", game->teams[j].komi);
		/*
		fprintf(fp, " display-color=\"#%02X%02X%02X\"",
		        (unsigned int) (game->teams[j].display_color.r * 255),
		        (unsigned int) (game->teams[j].display_color.g * 255),
		        (unsigned int) (game->teams[j].display_color.b * 255));
		*/
		fprintf(fp, ">\n");
		for (i = 0; i < game->teams[j].num_players; i++) {
			fprintf(fp, "\t\t<player");
			if (game->players[game->teams[j].players[i]].name != NULL) {
				fprintf(fp, " name=\"");
				print_xml_escaped(game->players[game->teams[j].players[i]].name, fp);
				fprintf(fp, "\"");
			}
			fprintf(fp, "/>\n");
		}
		fprintf(fp, "\t</team>\n");
	}
	go_game_ruleset_print_xml(game, fp);
	switch (game->board_info.type) {
		case GO_GAME_BOARD_INFO_TYPE_SQUARE:
			go_game_board_print_xml_square(game, fp);
			break;
		case GO_GAME_BOARD_INFO_TYPE_RECTANGULAR:
			go_game_board_print_xml_rectangular(game, fp);
			break;
		default:
			go_game_board_print_xml_graph(game, fp);
			break;
	}
	go_game_turn_print_recursively(game, game->history.history, 0, fp);
	fprintf(fp, "</game>\n");

	return;
}

static void go_game_ruleset_print_xml(const go_game *game, FILE *fp)
{
	const char *scores[4];
	unsigned int num_scores;
	unsigned int i;

	fprintf(fp, "\t<ruleset");
	for (i = 0; i < sizeof(ruleset_names) / sizeof(*ruleset_names); i++) {
		if (go_ruleset_cmp(&game->ruleset, ruleset_names[i].ruleset) == 0) {
			fprintf(fp, " name=\"");
			print_xml_escaped(ruleset_names[i].name, fp);
			fprintf(fp, "\"");
			break;
		}
	}
	/* If there was no exact match */
	if (i >= sizeof(ruleset_names) / sizeof(*ruleset_names)) {
		if (!game->ruleset.pass_allowed)
			fprintf(fp, " passing=\"no\"");
		if (game->ruleset.suicide_allowed)
			fprintf(fp, " suicide=\"yes\"");
		else
			fprintf(fp, " suicide=\"no\"");
		if (game->ruleset.ko == GO_RULESET_KO_NONE)
			fprintf(fp, " ko=\"none\"");
		else if (game->ruleset.ko == GO_RULESET_KO_SIMPLE)
			fprintf(fp, " ko=\"simple\"");
		else if (game->ruleset.ko == GO_RULESET_KO_POSITIONAL_SUPERKO)
			fprintf(fp, " ko=\"positional-superko\"");
		else if (game->ruleset.ko == GO_RULESET_KO_SITUATIONAL_SUPERKO)
			fprintf(fp, " ko=\"situational-superko\"");
		if (game->ruleset.handicap_placement == GO_RULESET_HANDICAP_FREE)
			fprintf(fp, " handicap=\"free\"");
		else if (game->ruleset.handicap_placement == GO_RULESET_HANDICAP_FIXED)
			fprintf(fp, " handicap=\"fixed\"");
		num_scores = 0;
		if (game->ruleset.score_territory)
			scores[num_scores++] = "territory";
		if (game->ruleset.score_occupied_points)
			scores[num_scores++] = "stones";
		if (game->ruleset.score_captures)
			scores[num_scores++] = "captures";
		if (game->ruleset.score_passes)
			scores[num_scores++] = "passes";
		fprintf(fp, " score=\"");
		for (i = 0; i < num_scores; i++) {
			if (i > 0)
				putc(',', fp);
			fprintf(fp, "%s", scores[i]);
		}
		fprintf(fp, "\"");
		if (game->ruleset.play_on_colored_points)
			fprintf(fp, " play-on-colored-points=\"yes\"");
		if (game->ruleset.simultaneous_capture)
			fprintf(fp, " simultaneous-capture=\"yes\"");
		if (game->ruleset.othello_capture)
			fprintf(fp, " othello-capture=\"yes\"");
	}
	fprintf(fp, "/>\n");

	return;
}

static void go_game_board_location_print_xml(const go_game *game, unsigned int index, FILE *fp)
{
	switch (game->board_info.type) {
		case GO_GAME_BOARD_INFO_TYPE_GRAPH:
			go_game_board_location_print_xml_graph(game, index, fp);
			break;
		case GO_GAME_BOARD_INFO_TYPE_SQUARE:
			go_game_board_location_print_xml_square(game, index, fp);
			break;
		case GO_GAME_BOARD_INFO_TYPE_RECTANGULAR:
			go_game_board_location_print_xml_rectangular(game, index, fp);
			break;
		default:
			fprintf(fp, "%u", index);
			break;
	}

	return;
}

static void go_game_board_location_print_xml_graph(const go_game *game, unsigned int index, FILE *fp)
{
	fprintf(fp, "%u", index);

	return;
}

static void go_game_board_location_print_xml_rectangular(const go_game *game, unsigned int index, FILE *fp)
{
	unsigned int width, height;
	char c;

	if (game->board_info.type == GO_GAME_BOARD_INFO_TYPE_SQUARE) {
		width = game->board_info.size.square.size;
		height = game->board_info.size.square.size;
	} else {
		width = game->board_info.size.rectangular.width;
		height = game->board_info.size.rectangular.height;
	}
	if ((width == 0) || (width > 'Z' - 'A' - 1)) {
		fprintf(fp, "%u", index);
		return;
	}
	if (index % width < 'I' - 'A')
		c = index % width + 'A';
	else
		c = index % width + 'A' + 1;
	fprintf(fp, "%c%u", c, height - index / width);

	return;
}

static void go_game_board_print_xml_graph(const go_game *game, FILE *fp)
{
	unsigned int i, j, star_point_index;
	float x, y, z;

	fprintf(fp, "\t<board type=\"graph\" size=\"%u\">\n",
	        game->board_info.size.graph.size);
	star_point_index = 0;
	for (i = 0; i < game->board.num_points; i++) {
		fprintf(fp, "\t\t<point");
		if (game->board.points[i].num_adjacent > 0) {
			fprintf(fp, " connections=\"");
			for (j = 0; j < game->board.points[i].num_adjacent - 1; j++)
				fprintf(fp, "%u,", game->board.points[i].adjacent[j]);
			fprintf(fp, "%u\"", game->board.points[i].adjacent[j]);
		}
		go_game_board_get_point_location(game, i, &x, &y, &z);
		fprintf(fp, " x=\"%g\" y=\"%g\"", x, y);
		if (z != 0.0)
			fprintf(fp, " z=\"%g\"", z);
		if ((star_point_index < game->board_info.num_star_points) &&
		    (game->board_info.star_points[star_point_index] == i)) {
			fprintf(fp, " star=\"yes\"");
			star_point_index++;
		}
		fprintf(fp, "/>\n");
	}
	fprintf(fp, "\t</board>\n");

	return;
}

static void go_game_board_print_xml_square(const go_game *game, FILE *fp)
{
	fprintf(fp, "\t<board type=\"square\" size=\"%u\"/>\n",
	        game->board_info.size.square.size);

	return;
}

static void go_game_board_print_xml_rectangular(const go_game *game, FILE *fp)
{
	fprintf(fp, "\t<board type=\"rectangular\" width=\"%u\" height=\"%u\"/>\n",
	        game->board_info.size.rectangular.width,
	        game->board_info.size.rectangular.height);

	return;
}

static void go_game_turn_print_recursively(const go_game *game, const go_history_turn *turn, unsigned int level, FILE *fp)
{
	go_history_turn *variation;
	unsigned int i;

	if (turn == NULL)
		return;
	for (i = 0; i < level + 1; i++)
		putc('\t', fp);
	fprintf(fp, "<history>\n");
	go_game_turn_print(game, turn, level, fp);
	for (turn = turn->next; turn != NULL; turn = turn->next) {
		go_game_turn_print(game, turn, level, fp);
		for (variation = turn->variation; variation != NULL; variation = variation->variation)
			go_game_turn_print_recursively(game, variation, level + 1, fp);
	}
	for (i = 0; i < level + 1; i++)
		putc('\t', fp);
	fprintf(fp, "</history>\n");

	return;
}

static void go_game_turn_print(const go_game *game, const go_history_turn *turn, unsigned int level, FILE *fp)
{
	const go_history_turn_action_handler *handler;
	unsigned int i;

	for (i = 0; i < level + 2; i++)
		putc('\t', fp);
	fprintf(fp, "<turn>");
	/*
	fprintf(fp, "<turn number=\"%u\">", turn->game_state.turn_num);
	*/
	for (i = 0; i < turn->num_actions; i++) {
		handler = action_handlers_search(turn->actions[i].type);
		if (handler != NULL) {
			handler->print(game, turn, &turn->actions[i], fp);
		}
	}
	fprintf(fp, "</turn>");
	putc('\n', fp);

	return;
}

static int go_history_turn_action_handler_type_cmp(unsigned int *type, const go_history_turn_action_handler *handler)
{
	return (int) *type - (int) handler->type;
}

static const go_history_turn_action_handler *action_handlers_search(unsigned int type)
{
	static const void *array = action_handlers;
	static unsigned int num = sizeof(action_handlers) / sizeof(*action_handlers);

	return (const go_history_turn_action_handler *) array_search_sorted(&array, &num, sizeof(go_history_turn_action_handler), &type, (int (*)(const void *, const void *)) go_history_turn_action_handler_type_cmp);
}

#ifdef PRINT_DTD
static void print_dtd(FILE *fp)
{
	fprintf(fp, "\
<!DOCTYPE game [\n\
\t<!ELEMENT game (team*, board, history?)>\n\
\t<!ELEMENT team (player+)>\n\
\t<!ATTLIST team\n\
\t          name CDATA #IMPLIED\n\
\t          handicap CDATA #IMPLIED\n\
\t          komi CDATA #IMPLIED\n\
\t          display-color CDATA #IMPLIED>\n\
\t<!ELEMENT player EMPTY>\n\
\t<!ATTLIST player\n\
\t          name CDATA #IMPLIED>\n\
");
	fprintf(fp, "\
\t<!ELEMENT ruleset EMPTY>\n\
\t<!ATTLIST ruleset\n\
\t          name CDATA #IMPLIED\n\
\t          passing (yes | no) #IMPLIED\n\
\t          suicide (yes | no) #IMPLIED\n\
\t          ko (none | simple | positional-superko | situational-superko) #IMPLIED\n\
\t          handicap (free | fixed) #IMPLIED\n\
\t          score CDATA #IMPLIED\n\
\t          play-on-colored-points (yes | no) #IMPLIED\n\
\t          simultaneous-capture (yes | no) #IMPLIED\n\
\t          othello-capture (yes | no) #IMPLIED>\n\
");
	fprintf(fp, "\
\t<!ELEMENT board (point*)>\n\
\t<!ATTLIST board\n\
\t          type (graph | square | rectangular) #REQUIRED\n\
\t          size CDATA #IMPLIED\n\
\t          width CDATA #IMPLIED\n\
\t          height CDATA #IMPLIED>\n\
\t<!ELEMENT point EMPTY>\n\
\t<!ATTLIST point\n\
\t          connections CDATA #IMPLIED\n\
\t          star (yes | no) #IMPLIED\n\
\t          x CDATA #IMPLIED\n\
\t          y CDATA #IMPLIED\n\
\t          z CDATA #IMPLIED>\n\
");
	fprintf(fp, "\
\t<!ELEMENT history (turn, (turn, history*)*)>\n\
\t<!ELEMENT turn (color | move | pass | resign | set-player | comment | mark)*>\n\
\t<!ELEMENT color EMPTY>\n\
\t<!ATTLIST color\n\
\t          location CDATA #REQUIRED\n\
\t          color CDATA #REQUIRED>\n\
\t<!ELEMENT move EMPTY>\n\
\t<!ATTLIST move\n\
\t          location CDATA #REQUIRED>\n\
\t<!ELEMENT pass EMPTY>\n\
\t<!ELEMENT resign EMPTY>\n\
\t<!ELEMENT set-player EMPTY>\n\
\t<!ATTLIST set-player\n\
\t          team CDATA #REQUIRED\n\
\t          player CDATA #REQUIRED>\n\
\t<!ELEMENT comment (#PCDATA)>\n\
\t<!ELEMENT dead EMPTY>\n\
\t<!ATTLIST dead\n\
\t          location CDATA #REQUIRED>\n\
");
	fprintf(fp, "\
\t<!ELEMENT mark EMPTY>\n\
\t<!ATTLIST mark\n\
\t          type CDATA #REQUIRED\n\
\t          location CDATA #REQUIRED\n\
\t          endpoint CDATA #IMPLIED\n\
\t          text CDATA #IMPLIED>\n\
]>\n\
");

	return;
}
#endif
