/*
	game_sgf.c
*/

#include "config.h"

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "array.h"
#include "game.h"
#include "ruleset.h"
#include "sgf.h"

#define DEFAULT_RULESET (go_ruleset_japanese)

enum {
	DEFAULT_BOARD_SIZE = 19
};

typedef struct go_game_load_sgf_helper go_game_load_sgf_helper;
struct go_game_load_sgf_helper {
	go_game *game;
	unsigned int index;
	sgf_node root_node;
	unsigned int num_games;
	go_game *games;
	unsigned int stack_size;
	go_history_turn **stack;
};

static const struct {
	char *name;
	const go_ruleset *ruleset;
} sgf_ruleset_names[] = {
	{ "AGA", &go_ruleset_aga },
	/*
	{ "GOE", &go_ruleset_ing },
	*/
	{ "Japanese", &go_ruleset_japanese },
	{ "New Zealand", &go_ruleset_new_zealand },
	{ "NZ", &go_ruleset_new_zealand }
};

static go_game *go_game_sgf_copyright(go_game *game, const sgf_property *property);

static go_game *go_game_sgf_event(go_game *game, const sgf_property *property);

static go_game *go_game_sgf_check_game(go_game *game, const sgf_property *property);

static go_game *go_game_sgf_game_name(go_game *game, const sgf_property *property);

static go_game *go_game_sgf_handicap(go_game *game, const sgf_property *property);

static go_game *go_game_sgf_komi(go_game *game, const sgf_property *property);

static go_game *go_game_sgf_player_name(go_game *game, const sgf_property *property);

static go_game *go_game_sgf_place(go_game *game, const sgf_property *property);

static go_game *go_game_sgf_ruleset(go_game *game, const sgf_property *property);

static go_game *go_game_sgf_source(go_game *game, const sgf_property *property);

static go_game *go_game_sgf_board_size(go_game *game, const sgf_property *property);

static go_game *go_game_sgf_color_point(go_game *game, const sgf_property *property);

static go_game *go_game_sgf_line(go_game *game, const sgf_property *property);

static go_game *go_game_sgf_move(go_game *game, const sgf_property *property);

static go_game *go_game_sgf_comment(go_game *game, const sgf_property *property);

static go_game *go_game_sgf_markup(go_game *game, const sgf_property *property);

static go_game *go_game_sgf_label(go_game *game, const sgf_property *property);

static go_game *go_game_sgf_player(go_game *game, const sgf_property *property);

static go_game *go_game_sgf_territory(go_game *game, const sgf_property *property);

static sgf_node *sgf_go_game_color_point(sgf_node *node, const go_game *game, const go_history_turn *turn, const go_history_turn_action *action, const sgf_parser *parser);

static sgf_node *sgf_go_game_move(sgf_node *node, const go_game *game, const go_history_turn *turn, const go_history_turn_action *action, const sgf_parser *parser);

static sgf_node *sgf_go_game_player(sgf_node *node, const go_game *game, const go_history_turn *turn, const go_history_turn_action *action, const sgf_parser *parser);

static sgf_node *sgf_go_game_comment(sgf_node *node, const go_game *game, const go_history_turn *turn, const go_history_turn_action *action, const sgf_parser *parser);

static sgf_node *sgf_go_game_markup(sgf_node *node, const go_game *game, const go_history_turn *turn, const go_history_turn_action *action, const sgf_parser *parser);

typedef struct sgf_property_handler sgf_property_handler;
struct sgf_property_handler {
	char *name;
	go_game *(*func)(go_game *, const sgf_property *);
};

/* You can handle AB and AW here too if you need to do handicap stuff.
   Just don't actually add any stones. */
static const sgf_property_handler root_property_handlers[] = {
	{ "CP", go_game_sgf_copyright },
	{ "EV", go_game_sgf_event },
	{ "GM", go_game_sgf_check_game },
	{ "GN", go_game_sgf_game_name },
	{ "HA", go_game_sgf_handicap },
	{ "KM", go_game_sgf_komi },
	{ "PB", go_game_sgf_player_name },
	{ "PC", go_game_sgf_place },
	{ "PW", go_game_sgf_player_name },
	{ "RU", go_game_sgf_ruleset },
	{ "SO", go_game_sgf_source },
	{ "SZ", go_game_sgf_board_size }
};

static const sgf_property_handler property_handlers[] = {
	{ "AB", go_game_sgf_color_point },
	{ "AE", go_game_sgf_color_point },
	{ "AR", go_game_sgf_line },
	{ "AW", go_game_sgf_color_point },
	{ "B", go_game_sgf_move },
	{ "C", go_game_sgf_comment },
	{ "CR", go_game_sgf_markup },
	{ "LB", go_game_sgf_label },
	{ "LN", go_game_sgf_line },
	{ "MA", go_game_sgf_markup },
	{ "PL", go_game_sgf_player },
	{ "SL", go_game_sgf_markup },
	{ "SQ", go_game_sgf_markup },
	{ "TB", go_game_sgf_territory },
	{ "TR", go_game_sgf_markup },
	{ "TW", go_game_sgf_territory },
	{ "W", go_game_sgf_move }
};

typedef struct go_history_turn_action_handler go_history_turn_action_handler;
struct go_history_turn_action_handler {
	unsigned int type;
	sgf_node *(*func)(sgf_node *, const go_game *, const go_history_turn *, const go_history_turn_action *, const sgf_parser *);
};

static const go_history_turn_action_handler action_handlers[] = {
	{ GO_HISTORY_TURN_ACTION_COLOR_POINT, sgf_go_game_color_point },
	{ GO_HISTORY_TURN_ACTION_MOVE, sgf_go_game_move },
	{ GO_HISTORY_TURN_ACTION_PASS, sgf_go_game_move },
	{ GO_HISTORY_TURN_ACTION_RESIGN, NULL },
	{ GO_HISTORY_TURN_ACTION_SET_PLAYER, sgf_go_game_player },
	{ GO_HISTORY_TURN_ACTION_COMMENT, sgf_go_game_comment },
	{ GO_HISTORY_TURN_ACTION_MARKUP, sgf_go_game_markup }
};

static int sgf_property_handler_name_cmp(const char *name, const sgf_property_handler *handler);

static sgf_property_handler *property_handlers_search(const sgf_property_handler *array, unsigned int num, const char *name);

static int go_history_turn_action_handler_type_cmp(unsigned int *type, const go_history_turn_action_handler *handler);

static const go_history_turn_action_handler *action_handlers_search(unsigned int type);

static sgf_node *sgf_node_make_property(sgf_node *node, const sgf_parser *parser, const char *name, const sgf_property_value *value);

static sgf_game *sgf_game_from_go_game(sgf_game *g, const go_game *game);

static sgf_node *sgf_game_from_go_game_aux(sgf_node *node, const go_game *game, const go_history_turn *turn, const sgf_parser *parser);

static sgf_node *sgf_game_from_go_game_info(sgf_node *node, const go_game *game, const sgf_parser *parser);

static sgf_node *sgf_game_from_go_game_result(sgf_node *node, const go_game *game, const sgf_parser *parser);

static go_game_load_sgf_helper *go_game_load_sgf_helper_init(go_game_load_sgf_helper *helper);

static void go_game_load_sgf_helper_free(go_game_load_sgf_helper *helper);

static go_game_load_sgf_helper *go_game_load_sgf_helper_push(go_game_load_sgf_helper *helper, go_history_turn *turn);

static go_game_load_sgf_helper *go_game_load_sgf_helper_pop(go_game_load_sgf_helper *helper, go_history_turn **turn);

static void go_game_load_sgf_index_game_start(sgf_parser *parser);

static void go_game_load_sgf_game_start(sgf_parser *parser);

static void go_game_load_sgf_game_end(sgf_parser *parser);

static void go_game_load_sgf_gametree_start(sgf_parser *parser);

static void go_game_load_sgf_gametree_end(sgf_parser *parser);

static void go_game_load_sgf_node_start_root(sgf_parser *parser);

static void go_game_load_sgf_node_start(sgf_parser *parser);

static void go_game_load_sgf_node_end_root(sgf_parser *parser);

static void go_game_load_sgf_node_end(sgf_parser *parser);

static void go_game_load_sgf_property_root(sgf_parser *parser, const sgf_property *property);

static void go_game_load_sgf_property(sgf_parser *parser, const sgf_property *property);

static void go_games_load_sgf_game_start(sgf_parser *parser);

static void go_games_load_sgf_game_end(sgf_parser *parser);

static void go_games_load_sgf_shallow_game_start(sgf_parser *parser);

static void go_games_load_sgf_shallow_node_end(sgf_parser *parser);

static go_game *go_game_sgf_copyright(go_game *game, const sgf_property *property)
{
	if (go_game_info_set(game, "copyright", property->values[0].types[0].text) == NULL)
		return NULL;

	return game;
}

static go_game *go_game_sgf_event(go_game *game, const sgf_property *property)
{
	if (go_game_info_set(game, "event", property->values[0].types[0].text) == NULL)
		return NULL;

	return game;
}

static go_game *go_game_sgf_check_game(go_game *game, const sgf_property *property)
{
	if (property->values[0].types[0].number != SGF_GAME_GO)
		return NULL;

	return game;
}

static go_game *go_game_sgf_game_name(go_game *game, const sgf_property *property)
{
	if (go_game_info_set(game, "name", property->values[0].types[0].text) == NULL)
		return NULL;

	return game;
}

static go_game *go_game_sgf_handicap(go_game *game, const sgf_property *property)
{
	if (go_game_team_set_handicap(game, 0, property->values[0].types[0].number) == NULL)
		return NULL;

	return game;
}

static go_game *go_game_sgf_komi(go_game *game, const sgf_property *property)
{
	game->teams[1].komi = property->values[0].types[0].real;

	return game;
}

static go_game *go_game_sgf_player_name(go_game *game, const sgf_property *property)
{
	unsigned int index;

	if (strcmp(property->name, "PB") == 0)
		index = 0;
	else if (strcmp(property->name, "PW") == 0)
		index = 1;
	else
		return NULL;
	if (go_player_set_name(&game->players[index], property->values[0].types[0].simpletext) == NULL)
		return NULL;

	return game;
}

static go_game *go_game_sgf_place(go_game *game, const sgf_property *property)
{
	if (go_game_info_set(game, "place", property->values[0].types[0].text) == NULL)
		return NULL;

	return game;
}

static go_game *go_game_sgf_ruleset(go_game *game, const sgf_property *property)
{
	unsigned int i;

	game->ruleset = DEFAULT_RULESET;
	for (i = 0; i < sizeof(sgf_ruleset_names) / sizeof(*sgf_ruleset_names); i++) {
		if (strcmp(property->values[0].types[0].simpletext, sgf_ruleset_names[i].name) == 0)
			game->ruleset = *sgf_ruleset_names[i].ruleset;
	}

	return game;
}

static go_game *go_game_sgf_source(go_game *game, const sgf_property *property)
{
	if (go_game_info_set(game, "source", property->values[0].types[0].text) == NULL)
		return NULL;

	return game;
}

static go_game *go_game_sgf_board_size(go_game *game, const sgf_property *property)
{
	if ((property->values[0].flags & SGF_PROPERTY_VALUE_SINGLE) ||
	    (property->values[0].types[0].number == property->values[0].types[1].number)) {
		if (go_game_board_set_size_square(game, property->values[0].types[0].number) == NULL)
			return NULL;
	} else {
		if (go_game_board_set_size_rectangular(game, property->values[0].types[0].number, property->values[0].types[1].number) == NULL)
			return NULL;
	}

	return game;
}

static go_game *go_game_sgf_color_point(go_game *game, const sgf_property *property)
{
	unsigned int i, index, color;
	unsigned int x, y;
	
	if (strcmp(property->name, "AB") == 0)
		color = game->teams[0].color;
	else if (strcmp(property->name, "AW") == 0)
		color = game->teams[1].color;
	else if (strcmp(property->name, "AE") == 0)
		color = 0;
	else
		return NULL;
	for (i = 0; i < property->num_values; i++) {
		if (property->values[i].flags & SGF_PROPERTY_VALUE_SINGLE) {
			x = property->values[i].types[0].number % 52;
			y = property->values[i].types[0].number / 52;
			index = go_game_board_index(game, x, y);
			if (go_game_color_point(game, index, color) == NULL)
				return NULL;
		} else {
			for (y = property->values[i].types[0].number / 52; y < property->values[i].types[1].number / 52; y++) {
				for (x = property->values[i].types[0].number % 52; x < property->values[i].types[1].number % 52; x++) {
					index = go_game_board_index(game, x, y);
					if (go_game_color_point(game, index, color) == NULL)
						return NULL;
				}
			}
		}
	}

	return game;
}

static go_game *go_game_sgf_line(go_game *game, const sgf_property *property)
{
	unsigned int i, start, end;
	unsigned int x1, y1, x2, y2;
	int type;

	if (strcmp(property->name, "AR") == 0)
		type = GO_GAME_MARKUP_ARROW;
	else if (strcmp(property->name, "LN") == 0)
		type = GO_GAME_MARKUP_LINE;
	else
		return NULL;
	for (i = 0; i < property->num_values; i++) {
		x1 = property->values[i].types[0].number % 52;
		y1 = property->values[i].types[0].number / 52;
		x2 = property->values[i].types[1].number % 52;
		y2 = property->values[i].types[1].number / 52;
		start = go_game_board_index(game, x1, y1);
		end = go_game_board_index(game, x2, y2);
		if (go_game_line(game, start, end, type) == NULL)
			return NULL;
	}

	return game;
}

static go_game *go_game_sgf_move(go_game *game, const sgf_property *property)
{
	unsigned int index, team_index;

	if (strcmp(property->name, "B") == 0)
		team_index = 0;
	else if (strcmp(property->name, "W") == 0)
		team_index = 1;
	else
		return NULL;
	if (team_index != game->current_team) {
		if (go_game_set_current_player(game, team_index, 0) == NULL)
			return NULL;
	}
	if (property->values[0].flags & SGF_PROPERTY_VALUE_NONE)
		return go_game_pass(game);
	index = go_game_board_index(game, property->values[0].types[0].number % 52, property->values[0].types[0].number / 52);
	if (index >= game->board.num_points)
		return go_game_pass(game);
	if (go_game_move(game, index) == NULL)
		return NULL;

	return game;
}

static go_game *go_game_sgf_comment(go_game *game, const sgf_property *property)
{
	if (go_game_comment(game, property->values[0].types[0].text) == NULL)
		return NULL;

	return game;
}

static go_game *go_game_sgf_label(go_game *game, const sgf_property *property)
{
	unsigned int i, index;

	for (i = 0; i < property->num_values; i++) {
		index = go_game_board_index(game, property->values[i].types[0].number % 52, property->values[i].types[0].number / 52);
		if (go_game_label(game, index, property->values[i].types[1].simpletext) == NULL)
			return NULL;
	}

	return game;
}

static go_game *go_game_sgf_markup(go_game *game, const sgf_property *property)
{
	unsigned int i, index;
	unsigned int x, y;
	int type;

	if (strcmp(property->name, "CR") == 0)
		type = GO_GAME_MARKUP_CIRCLE;
	else if (strcmp(property->name, "MA") == 0)
		type = GO_GAME_MARKUP_X;
	else if (strcmp(property->name, "SL") == 0)
		type = GO_GAME_MARKUP_SELECTED;
	else if (strcmp(property->name, "SQ") == 0)
		type = GO_GAME_MARKUP_SQUARE;
	else if (strcmp(property->name, "TR") == 0)
		type = GO_GAME_MARKUP_TRIANGLE;
	else
		return NULL;
	for (i = 0; i < property->num_values; i++) {
		if (property->values[i].flags & SGF_PROPERTY_VALUE_SINGLE) {
			x = property->values[i].types[0].number % 52;
			y = property->values[i].types[0].number / 52;
			index = go_game_board_index(game, x, y);
			if (go_game_mark(game, index, type) == NULL)
				return NULL;
		} else {
			for (y = property->values[i].types[0].number / 52; y < property->values[i].types[1].number / 52; y++) {
				for (x = property->values[i].types[0].number % 52; x < property->values[i].types[1].number % 52; x++) {
					index = go_game_board_index(game, x, y);
					if (go_game_mark(game, index, type) == NULL)
						return NULL;
				}
			}
		}
	}

	return game;
}

static go_game *go_game_sgf_player(go_game *game, const sgf_property *property)
{
	unsigned int team_index;

	team_index = property->values[0].types[0].color - 1;
	if (go_game_set_current_player(game, team_index, 0) == NULL)
		return NULL;

	return game;
}

static go_game *go_game_sgf_territory(go_game *game, const sgf_property *property)
{
	unsigned int i, index, color;
	unsigned int x, y;

	if (strcmp(property->name, "TB") == 0)
		color = game->teams[0].color;
	else if (strcmp(property->name, "TW") == 0)
		color = game->teams[1].color;
	else
		return NULL;
	for (i = 0; i < property->num_values; i++) {
		if (property->values[i].flags & SGF_PROPERTY_VALUE_SINGLE) {
			x = property->values[i].types[0].number % 52;
			y = property->values[i].types[0].number / 52;
			index = go_game_board_index(game, x, y);
			if ((game->board.points[index].color != 0) &&
			    (game->board.points[index].color != color) &&
			    (!go_game_is_dead(game, index))) {
				if (go_game_toggle_dead(game, index) == NULL)
					return NULL;
			}
		} else {
			for (y = property->values[i].types[0].number / 52; y < property->values[i].types[1].number / 52; y++) {
				for (x = property->values[i].types[0].number % 52; x < property->values[i].types[1].number % 52; x++) {
					index = go_game_board_index(game, x, y);
					if ((game->board.points[index].color != 0) &&
					    (game->board.points[index].color != color) &&
					    (!go_game_is_dead(game, index))) {
						if (go_game_toggle_dead(game, index) == NULL)
							return NULL;
					}
				}
			}
		}
	}

	return game;
}

static sgf_node *sgf_go_game_color_point(sgf_node *node, const go_game *game, const go_history_turn *turn, const go_history_turn_action *action, const sgf_parser *parser)
{
	static const char *property_names[] = { "AE", "AB", "AW" };
	sgf_property_value value;
	unsigned int index, color, width;

	color = action->value.index_color.color;
	if (color >= sizeof(property_names) / sizeof(*property_names))
		return NULL;
	if (game->board_info.type == GO_GAME_BOARD_INFO_TYPE_SQUARE)
		width = game->board_info.size.square.size;
	else
		width = game->board_info.size.rectangular.width;
	index = (action->value.index_color.index / width) * 52 + (action->value.index_color.index % width);
	value.flags = SGF_PROPERTY_VALUE_SINGLE;
	value.types[0].number = index;
	if (sgf_node_make_property(node, parser, property_names[color], &value) == NULL)
		return NULL;

	return node;
}

static sgf_node *sgf_go_game_move(sgf_node *node, const go_game *game, const go_history_turn *turn, const go_history_turn_action *action, const sgf_parser *parser)
{
	static const char *property_names[] = { "B", "W" };
	sgf_property_value value;
	unsigned int index, team_index, width;

	if (game->board_info.type == GO_GAME_BOARD_INFO_TYPE_SQUARE)
		width = game->board_info.size.square.size;
	else
		width = game->board_info.size.rectangular.width;
	team_index = turn->game_state.current_team;
	index = (action->value.integer / width) * 52 + (action->value.integer % width);
	if (team_index > 1)
		team_index = 1;
	if (action->type != GO_HISTORY_TURN_ACTION_PASS) {
		value.flags = SGF_PROPERTY_VALUE_SINGLE;
		value.types[0].number = index;
	} else {
		value.flags = SGF_PROPERTY_VALUE_NONE;
	}
	if (sgf_node_make_property(node, parser, property_names[team_index], &value) == NULL)
		return NULL;
		/*
		if (sgf_node_make_property_none(property, parser, property_names[team_index]) == NULL)
			return NULL;
		*/
		/*
		if (sgf_property_init(&property) == NULL)
			return NULL;
		if (sgf_parser_property_set_name(parser, &property, property_names[team_index]) == NULL)
			return NULL;
		if (sgf_node_add_property(node, &property) == NULL) {
			sgf_property_free(&property);
			return NULL;
		}
		*/

	return node;
}

static sgf_node *sgf_go_game_player(sgf_node *node, const go_game *game, const go_history_turn *turn, const go_history_turn_action *action, const sgf_parser *parser)
{
	sgf_property_value value;
	unsigned int color;

	color = action->value.integer + 1;
	if (color > 2)
		color = 2;
	value.flags = SGF_PROPERTY_VALUE_SINGLE;
	value.types[0].color = color;
	if (sgf_node_make_property(node, parser, "PL", &value) == NULL)
		return NULL;

	return node;
}

static sgf_node *sgf_go_game_comment(sgf_node *node, const go_game *game, const go_history_turn *turn, const go_history_turn_action *action, const sgf_parser *parser)
{
	sgf_property *p;
	sgf_property_value value;
	char *s;
	size_t length;

	p = sgf_node_search_properties_by_name(node, "C");
	if (p == NULL) {
		value.flags = SGF_PROPERTY_VALUE_SINGLE;
		value.types[0].text = action->value.string;
		if (sgf_node_make_property(node, parser, "C", &value) == NULL)
			return NULL;
	} else {
		/* Append the string to the old property value type after a newline */
		length = strlen(p->values[0].types[0].text);
		s = (char *) realloc(p->values[0].types[0].text, length + strlen(action->value.string) + 2);
		if (s == NULL)
			return NULL;
		p->values[0].types[0].text = s;
		p->values[0].types[0].text[length] = '\n';
		strcpy(p->values[0].types[0].text + length + 1, action->value.string);
	}

	return node;
}

static sgf_node *sgf_go_game_markup(sgf_node *node, const go_game *game, const go_history_turn *turn, const go_history_turn_action *action, const sgf_parser *parser)
{
	static struct {
		int type;
		const char *name;
	} property_names[] = {
		{ GO_GAME_MARKUP_CLEAR, NULL },
		{ GO_GAME_MARKUP_CIRCLE, "CR" },
		{ GO_GAME_MARKUP_SELECTED, "SL" },
		{ GO_GAME_MARKUP_SQUARE, "SQ" },
		{ GO_GAME_MARKUP_TRIANGLE, "TR" },
		{ GO_GAME_MARKUP_X, "MA" },
		{ GO_GAME_MARKUP_LINE, "LN" },
		{ GO_GAME_MARKUP_ARROW, "AR" },
		{ GO_GAME_MARKUP_LABEL, "LB" }
	};
	sgf_property_value value;
	unsigned int i, width;

	for (i = 0; i < sizeof(property_names) / sizeof(*property_names); i++) {
		if (property_names[i].type == action->value.markup.type)
			break;
	}
	if (i >= sizeof(property_names) / sizeof(*property_names))
		return NULL;
	if (game->board_info.type == GO_GAME_BOARD_INFO_TYPE_SQUARE)
		width = game->board_info.size.square.size;
	else
		width = game->board_info.size.rectangular.width;
	switch (action->value.markup.type) {
		case GO_GAME_MARKUP_CIRCLE:
		case GO_GAME_MARKUP_SELECTED:
		case GO_GAME_MARKUP_SQUARE:
		case GO_GAME_MARKUP_TRIANGLE:
		case GO_GAME_MARKUP_X:
			value.flags = SGF_PROPERTY_VALUE_SINGLE;
			value.types[0].number = (action->value.markup.index / width) * 52 + (action->value.markup.index % width);
			break;
		case GO_GAME_MARKUP_LINE:
		case GO_GAME_MARKUP_ARROW:
			value.flags = SGF_PROPERTY_VALUE_COMPOSED;
			value.types[0].number = (action->value.markup.index / width) * 52 + (action->value.markup.index % width);
			value.types[1].number = (action->value.markup.data.endpoint / width) * 52 + (action->value.markup.data.endpoint % width);
			break;
		case GO_GAME_MARKUP_LABEL:
			value.flags = SGF_PROPERTY_VALUE_COMPOSED;
			value.types[0].number = (action->value.markup.index / width) * 52 + (action->value.markup.index % width);
			value.types[1].simpletext = action->value.markup.data.string;
			break;
	}
	if (property_names[i].name != NULL) {
		if (sgf_node_make_property(node, parser, property_names[i].name, &value) == NULL)
			return NULL;
	}

	return node;
}

static int sgf_property_handler_name_cmp(const char *name, const sgf_property_handler *handler)
{
	return strcmp(name, handler->name);
}

static sgf_property_handler *property_handlers_search(const sgf_property_handler *array, unsigned int num, const char *name)
{
	sgf_property_handler *handler;

	handler = (sgf_property_handler *) array_search((const void **) &array, &num, sizeof(sgf_property_handler), name, (int (*)(const void *, const void *)) sgf_property_handler_name_cmp);
	/* To put raw SGF properties in the history */
	/*
	if (handler == NULL)
		handler = unknown;
	*/

	return handler;
}

static int go_history_turn_action_handler_type_cmp(unsigned int *type, const go_history_turn_action_handler *handler)
{
	return (int) *type - (int) handler->type;
}

static const go_history_turn_action_handler *action_handlers_search(unsigned int type)
{
	static const void *array = (void *) action_handlers;
	static unsigned int num = sizeof(action_handlers) / sizeof(*action_handlers);

	return (const go_history_turn_action_handler *) array_search_sorted(&array, &num, sizeof(go_history_turn_action_handler), &type, (int (*)(const void *, const void *)) go_history_turn_action_handler_type_cmp);
}

static sgf_node *sgf_node_make_property(sgf_node *node, const sgf_parser *parser, const char *name, const sgf_property_value *value)
{
	sgf_property property;
	sgf_property_value new_value, old_value;

	if (sgf_property_init(&property) == NULL)
		return NULL;
	if (sgf_parser_property_set_name(parser, &property, name) == NULL)
		return NULL;
	old_value = *value;
	old_value.handlers = property.type->handlers;
	if (sgf_property_value_duplicate(&new_value, &old_value) == NULL) {
		sgf_property_free(&property);
		return NULL;
	}
	if (sgf_property_add_value(&property, &new_value) == NULL) {
		sgf_property_free(&property);
		return NULL;
	}
	if (sgf_node_add_property(node, &property) == NULL) {
		sgf_property_free(&property);
		return NULL;
	}

	return node;
}

static sgf_game *sgf_game_from_go_game(sgf_game *g, const go_game *game)
{
	sgf_parser parser;
	sgf_node *node;
	sgf_property_value value;
	const go_history_turn_action_handler *handler;
	char *ruleset_name;
	unsigned int i;
	int diff, tmp;

	node = (sgf_node *) malloc(sizeof(sgf_node));
	if (node == NULL)
		return NULL;
	if (sgf_node_init(node) == NULL) {
		free(node);
		return NULL;
	}
	*g = node;
	/* We don't do anything with the parser but use it to search property
	   types */
	if (sgf_parser_init(&parser) == NULL)
		return NULL;
	parser.game = SGF_GAME_GO;
	parser.format = 4;
	if ((game->num_players != 2) ||
	    (((game->board_info.type != GO_GAME_BOARD_INFO_TYPE_SQUARE) &&
	      (game->board_info.size.square.size > 52)) &&
	     ((game->board_info.type != GO_GAME_BOARD_INFO_TYPE_RECTANGULAR) &&
	      ((game->board_info.size.rectangular.width > 52) ||
	       (game->board_info.size.rectangular.height > 52))))) {
		return NULL;
	}
	value.flags = SGF_PROPERTY_VALUE_SINGLE;
	value.types[0].number = SGF_GAME_GO;
	if (sgf_node_make_property(node, &parser, "GM", &value) == NULL) {
		sgf_node_free(node);
		return NULL;
	}
	value.flags = SGF_PROPERTY_VALUE_SINGLE;
	value.types[0].number = 4;
	if (sgf_node_make_property(node, &parser, "FF", &value) == NULL) {
		sgf_node_free(node);
		return NULL;
	}
	if ((game->board_info.type == GO_GAME_BOARD_INFO_TYPE_SQUARE) ||
	    (game->board_info.size.rectangular.width ==
	     game->board_info.size.rectangular.height)) {
		value.flags = SGF_PROPERTY_VALUE_SINGLE;
		value.types[0].number = game->board_info.size.square.size;
	} else {
		value.flags = SGF_PROPERTY_VALUE_COMPOSED;
		value.types[0].number = game->board_info.size.rectangular.width;
		value.types[1].number = game->board_info.size.rectangular.height;
	}
	if (sgf_node_make_property(node, &parser, "SZ", &value) == NULL) {
		sgf_node_free(node);
		return NULL;
	}
	value.flags = SGF_PROPERTY_VALUE_COMPOSED;
	value.types[0].simpletext = "goclient";
	value.types[1].simpletext = "0.0";
	if (sgf_node_make_property(node, &parser, "AP", &value) == NULL) {
		sgf_node_free(node);
		return NULL;
	}
	diff = go_ruleset_cmp(&game->ruleset, &DEFAULT_RULESET);
	for (i = 0; i < sizeof(sgf_ruleset_names) / sizeof(*sgf_ruleset_names); i++) {
		tmp = go_ruleset_cmp(&game->ruleset, sgf_ruleset_names[i].ruleset);
		if (tmp <= diff) {
			ruleset_name = sgf_ruleset_names[i].name;
			diff = tmp;
		}
		if (tmp == 0)
			break;
	}
	value.flags = SGF_PROPERTY_VALUE_SINGLE;
	value.types[0].simpletext = ruleset_name;
	if (sgf_node_make_property(node, &parser, "RU", &value) == NULL) {
		sgf_node_free(node);
		return NULL;
	}
	if (sgf_game_from_go_game_info(node, game, &parser) == NULL) {
		sgf_node_free(node);
		return NULL;
	}
	if (sgf_game_from_go_game_result(node, game, &parser) == NULL) {
		sgf_node_free(node);
		return NULL;
	}
	for (i = 0; i < game->history.history->num_actions; i++) {
		handler = action_handlers_search(game->history.history->actions[i].type);
		if ((handler != NULL) && (handler->func != NULL)) {
			if (handler->func(node, game, game->history.history, &game->history.history->actions[i], &parser) == NULL) {
				sgf_node_free(node);
				return NULL;
			}
		}
	}
	if (sgf_game_from_go_game_aux(node, game, game->history.history->next, &parser) == NULL) {
		sgf_gametree_free(node);
		return NULL;
	}
	sgf_parser_free(&parser);

	return g;
}

static sgf_node *sgf_game_from_go_game_aux(sgf_node *node, const go_game *game, const go_history_turn *turn, const sgf_parser *parser)
{
	sgf_node *new_node;
	const go_history_turn_action_handler *handler;
	unsigned int i, num_variations;

	if (turn == NULL)
		return node;
	num_variations = 0;
	for (; turn != NULL; turn = turn->variation) {
		new_node = (sgf_node *) malloc(sizeof(sgf_node));
		if (new_node == NULL)
			return NULL;
		if (sgf_node_init(new_node) == NULL) {
			free(new_node);
			return NULL;
		}
		for (i = 0; i < turn->num_actions; i++) {
			handler = action_handlers_search(turn->actions[i].type);
			if ((handler != NULL) && (handler->func != NULL)) {
				if (handler->func(new_node, game, turn, &turn->actions[i], parser) == NULL)
					return NULL;
			}
		}
		if (num_variations == 0)
			node->next = new_node;
		else
			node->variation = new_node;
		node = new_node;
		num_variations++;
		if (sgf_game_from_go_game_aux(node, game, turn->next, parser) == NULL)
			return NULL;
	}

	return node;
}

static sgf_node *sgf_game_from_go_game_info(sgf_node *node, const go_game *game, const sgf_parser *parser)
{
	struct {
		char *name;
		sgf_property_value_type value;
	} properties[] = {
		{ "CP" }, { "EV" }, { "GN" }, { "HA" }, { "KM" }, { "PB" }, { "PC" },
		{ "PW" }, { "SO" }
	};
	sgf_property_value value;
	unsigned int i;

	properties[0].value.text = (char *) go_game_info_search(game, "copyright");
	properties[1].value.text = (char *) go_game_info_search(game, "event");
	properties[2].value.text = (char *) go_game_info_search(game, "name");
	if (game->teams[0].handicap > game->teams[1].handicap)
		properties[3].value.number = game->teams[0].handicap - game->teams[1].handicap;
	else
		properties[3].value.number = 0;
	properties[4].value.real = game->teams[1].komi - game->teams[0].komi;
	if (game->teams[0].name != NULL)
		properties[5].value.simpletext = game->teams[0].name;
	else
		properties[5].value.simpletext = game->players[0].name;
	properties[6].value.text = (char *) go_game_info_search(game, "place");
	if (game->teams[1].name != NULL)
		properties[7].value.simpletext = game->teams[1].name;
	else
		properties[7].value.simpletext = game->players[1].name;
	properties[8].value.text = (char *) go_game_info_search(game, "source");
	value.flags = SGF_PROPERTY_VALUE_SINGLE;
	for (i = 0; i < sizeof(properties) / sizeof(*properties); i++) {
		value.types[0] = properties[i].value;
		if (sgf_node_make_property(node, parser, properties[i].name, &value) == NULL) {
			/* It's possible that sgf_node_make_property returning NULL is not
			   a fatal error, like in the common case that one of the Text types
			   above is NULL */
			if (errno != 0)
				return NULL;
		}
	}

	return node;
}

static sgf_node *sgf_game_from_go_game_result(sgf_node *node, const go_game *game, const sgf_parser *parser)
{
	const go_history_turn *turn;
	sgf_property_value value;
	static char buffer[10];
	float difference;

	if (game->history.history == NULL)
		return node;
	for (turn = game->history.history; turn->next != NULL; turn = turn->next)
		;
	if (!go_game_is_over_turn(game, turn))
		snprintf(buffer, sizeof(buffer), "Void");
	else if (turn->game_state.team_states[1].is_resigned)
		snprintf(buffer, sizeof(buffer), "B+R");
	else if (turn->game_state.team_states[0].is_resigned)
		snprintf(buffer, sizeof(buffer), "W+R");
	/* Check if the sum of num_prisoners, komi, etc. don't equal score and do
	   "?" if not */
	else {
		difference = turn->game_state.team_states[0].score - turn->game_state.team_states[1].score;
		/* These might produce results in exponential notation (violating the
		   SGF spec) if the exponent of the score is less than -4. This should
		   be pretty rare. If it's a problem, do something like
		   sgf_property_value_type_compose_real in sgf.c */
		if (difference > 0)
			snprintf(buffer, sizeof(buffer), "B+%g", difference);
		else if (difference < 0)
			snprintf(buffer, sizeof(buffer), "W+%g", -difference);
		else
			snprintf(buffer, sizeof(buffer), "0");
	}
	value.flags = SGF_PROPERTY_VALUE_SINGLE;
	value.types[0].simpletext = buffer;
	if (sgf_node_make_property(node, parser, "RE", &value) == NULL) {
		sgf_node_free(node);
		return NULL;
	}

	return node;
}

static go_game_load_sgf_helper *go_game_load_sgf_helper_init(go_game_load_sgf_helper *helper)
{
	helper->game = NULL;
	helper->index = 0;
	if (sgf_node_init(&helper->root_node) == NULL)
		return NULL;
	array_init((void **) &helper->games, &helper->num_games);
	array_init((void **) &helper->stack, &helper->stack_size);

	return helper;
}

static void go_game_load_sgf_helper_free(go_game_load_sgf_helper *helper)
{
	array_free((void **) &helper->stack, &helper->stack_size, sizeof(go_history_turn *), NULL);

	return;
}

static go_game_load_sgf_helper *go_game_load_sgf_helper_push(go_game_load_sgf_helper *helper, go_history_turn *turn)
{
	if (array_insert((void **) &helper->stack, &helper->stack_size, sizeof(go_history_turn *), &turn, NULL) == NULL)
		return NULL;

	return helper;
}

static go_game_load_sgf_helper *go_game_load_sgf_helper_pop(go_game_load_sgf_helper *helper, go_history_turn **turn)
{
	if (helper->stack_size == 0)
		return NULL;
	if (turn != NULL)
		*turn = helper->stack[helper->stack_size - 1];
	if (array_remove_index_keep_order((void **) &helper->stack, &helper->stack_size, sizeof(go_history_turn *), helper->stack_size - 1) == NULL)
		return NULL;

	return helper;
}

static void go_game_load_sgf_index_game_start(sgf_parser *parser)
{
	go_game_load_sgf_helper *helper;

	helper = (go_game_load_sgf_helper *) parser->user_data;
	if (helper->index == 0) {
		sgf_parser_set_game_handlers(parser, go_game_load_sgf_game_start, go_game_load_sgf_game_end);
		sgf_parser_set_gametree_handlers(parser, go_game_load_sgf_gametree_start, go_game_load_sgf_gametree_end);
		sgf_parser_set_node_handlers(parser, go_game_load_sgf_node_start_root, go_game_load_sgf_node_end_root);
		sgf_parser_set_property_handler(parser, go_game_load_sgf_property_root);
		go_game_load_sgf_game_start(parser);
	} else {
		helper->index--;
	}

	return;
}

static void go_game_load_sgf_game_start(sgf_parser *parser)
{
	go_game_load_sgf_helper *helper;
	unsigned int i;

	helper = (go_game_load_sgf_helper *) parser->user_data;
	helper->game->ruleset = DEFAULT_RULESET;
	for (i = 0; i < 2; i++) {
		if (go_game_new_player(helper->game) == NULL) {
			sgf_parser_stop(parser);
			return;
		}
	}
	if (go_game_board_set_size_square(helper->game, DEFAULT_BOARD_SIZE) == NULL) {
		sgf_parser_stop(parser);
		return;
	}

	return;
}

static void go_game_load_sgf_game_end(sgf_parser *parser)
{
	sgf_parser_stop(parser);

	return;
}

static void go_game_load_sgf_gametree_start(sgf_parser *parser)
{
	go_game_load_sgf_helper *helper;

	helper = (go_game_load_sgf_helper *) parser->user_data;
	/* Save the current turn. At the very first gametree, it will be NULL. */
	if (go_game_load_sgf_helper_push(helper, helper->game->history.current_turn) == NULL) {
		sgf_parser_stop(parser);
		return;
	}

	return;
}

static void go_game_load_sgf_gametree_end(sgf_parser *parser)
{
	go_game_load_sgf_helper *helper;
	go_history_turn *turn;

	helper = (go_game_load_sgf_helper *) parser->user_data;
	if (go_game_load_sgf_helper_pop(helper, &turn) == NULL) {
		sgf_parser_stop(parser);
		return;
	}
	if (turn != NULL) {
		if (go_game_history_seek(helper->game, turn) == NULL) {
			sgf_parser_stop(parser);
			return;
		}
	} else {
		if (go_game_history_rewind(helper->game) == NULL) {
			sgf_parser_stop(parser);
			return;
		}
	}

	return;
}

static void go_game_load_sgf_node_start_root(sgf_parser *parser)
{
	go_game_load_sgf_helper *helper;

	helper = (go_game_load_sgf_helper *) parser->user_data;
	/* We need to cache the root node */
	if (sgf_node_init(&helper->root_node) == NULL) {
		sgf_parser_stop(parser);
		return;
	}
	go_game_load_sgf_node_start(parser);

	return;
}

static void go_game_load_sgf_node_start(sgf_parser *parser)
{
	go_game_load_sgf_helper *helper;

	helper = (go_game_load_sgf_helper *) parser->user_data;
	if (go_game_new_turn(helper->game) == NULL) {
		sgf_parser_stop(parser);
		return;
	}

	return;
}

static void go_game_load_sgf_node_end_root(sgf_parser *parser)
{
	go_game_load_sgf_helper *helper;
	sgf_property_handler *handler;
	unsigned int i;

	helper = (go_game_load_sgf_helper *) parser->user_data;
	/* Handle the root properties first. This is necessary because there can be
	   move and setup properties in the root node that depend on root
	   properties, and we can't be sure the root properties will come first. */
	for (i = 0; i < helper->root_node.num_properties; i++) {
		handler = property_handlers_search(root_property_handlers, sizeof(root_property_handlers) / sizeof(*root_property_handlers), helper->root_node.properties[i].name);
		if ((handler != NULL) && (handler->func != NULL)) {
			if (handler->func(helper->game, &helper->root_node.properties[i]) == NULL) {
				sgf_parser_stop(parser);
				return;
			}
		}
	}
	for (i = 0; i < helper->root_node.num_properties; i++) {
		handler = property_handlers_search(property_handlers, sizeof(property_handlers) / sizeof(*property_handlers), helper->root_node.properties[i].name);
		if ((handler != NULL) && (handler->func != NULL)) {
			if (handler->func(helper->game, &helper->root_node.properties[i]) == NULL) {
				sgf_parser_stop(parser);
				return;
			}
		}
	}
	/* This is the only node whose properties we need to cache, so forget it
	   now */
	sgf_node_free(&helper->root_node);
	sgf_parser_set_node_handlers(parser, go_game_load_sgf_node_start, go_game_load_sgf_node_end);
	sgf_parser_set_property_handler(parser, go_game_load_sgf_property);

	return;
}

static void go_game_load_sgf_node_end(sgf_parser *parser)
{
	go_game_load_sgf_helper *helper;

	helper = (go_game_load_sgf_helper *) parser->user_data;
	/* Delete the turn if it's empty, unless it's the root */
	if ((helper->game->history.current_turn->num_actions == 0) &&
	    (helper->game->history.current_turn->prev != NULL)) {
		if (go_game_history_undo(helper->game) == NULL) {
			sgf_parser_stop(parser);
			return;
		}
	}

	return;
}

static void go_game_load_sgf_property_root(sgf_parser *parser, const sgf_property *property)
{
	go_game_load_sgf_helper *helper;
	sgf_property p;

	helper = (go_game_load_sgf_helper *) parser->user_data;
	/* Just cache the property for later processing in
	   go_game_load_sgf_node_end_root */
	if (sgf_property_duplicate(&p, property) == NULL) {
		sgf_parser_stop(parser);
		return;
	}
	if (sgf_node_add_property(&helper->root_node, &p) == NULL) {
		sgf_property_free(&p);
		sgf_parser_stop(parser);
		return;
	}

	return;
}

static void go_game_load_sgf_property(sgf_parser *parser, const sgf_property *property)
{
	go_game_load_sgf_helper *helper;
	sgf_property_handler *handler;

	helper = (go_game_load_sgf_helper *) parser->user_data;
	handler = property_handlers_search(property_handlers, sizeof(property_handlers) / sizeof(*property_handlers), property->name);
	if ((handler != NULL) && (handler->func != NULL)) {
		if (handler->func(helper->game, property) == NULL) {
			sgf_parser_stop(parser);
			return;
		}
	}

	return;
}

go_game *go_game_load_sgf_index(go_game *game, const char *filename, unsigned int index)
{
	sgf_parser parser;
	go_game_load_sgf_helper helper;

	if (go_game_init(game) == NULL)
		return NULL;
	if (sgf_parser_init(&parser) == NULL) {
		go_game_free(game);
		return NULL;
	}
	if (sgf_parser_open_file(&parser, filename, "rb") == NULL)
		return NULL;
	if (go_game_load_sgf_helper_init(&helper) == NULL) {
		sgf_parser_free(&parser);
		return NULL;
	}
	helper.game = game;
	helper.index = index;
	sgf_parser_set_user_data(&parser, &helper);
	sgf_parser_set_game_handlers(&parser, go_game_load_sgf_index_game_start, NULL);
	/* The other handlers are set in go_game_load_sgf_index_game_start */
	if (sgf_parser_parse(&parser) == NULL) {
		go_game_free(game);
		go_game_load_sgf_helper_free(&helper);
		sgf_parser_free(&parser);
		return NULL;
	}
	sgf_parser_free(&parser);
	if (helper.index > 0) {
		/* The index was not reached */
		go_game_load_sgf_helper_free(&helper);
		return NULL;
	}
	go_game_load_sgf_helper_free(&helper);

	return game;
}

go_game *go_game_load_sgf(go_game *game, const char *filename)
{
	return go_game_load_sgf_index(game, filename, 0);
}

static void go_games_load_sgf_game_start(sgf_parser *parser)
{
	go_game_load_sgf_helper *helper;

	helper = (go_game_load_sgf_helper *) parser->user_data;
	if (go_game_init(helper->game) == NULL) {
		sgf_parser_stop(parser);
		return;
	}
	go_game_load_sgf_game_start(parser);
	sgf_parser_set_node_handlers(parser, go_game_load_sgf_node_start_root, go_game_load_sgf_node_end_root);
	sgf_parser_set_property_handler(parser, go_game_load_sgf_property_root);

	return;
}

static void go_games_load_sgf_game_end(sgf_parser *parser)
{
	go_game_load_sgf_helper *helper;

	helper = (go_game_load_sgf_helper *) parser->user_data;
	if (array_insert((void **) &helper->games, &helper->num_games, sizeof(go_game), helper->game, NULL) == NULL) {
		sgf_parser_stop(parser);
		return;
	}

	return;
}

go_game **go_games_load_sgf(go_game **games, unsigned int *num_games, const char *filename)
{
	sgf_parser parser;
	go_game_load_sgf_helper helper;
	go_game game;

	array_init((void **) games, num_games);
	if (sgf_parser_init(&parser) == NULL)
		return NULL;
	if (sgf_parser_open_file(&parser, filename, "rb") == NULL)
		return NULL;
	sgf_parser_set_game_handlers(&parser, go_games_load_sgf_game_start, go_games_load_sgf_game_end);
	sgf_parser_set_gametree_handlers(&parser, go_game_load_sgf_gametree_start, go_game_load_sgf_gametree_end);
	/* The node and property handlers are set in go_games_load_sgf_game_start */
	if (go_game_load_sgf_helper_init(&helper) == NULL) {
		sgf_parser_free(&parser);
		return NULL;
	}
	helper.game = &game;
	array_init((void **) &helper.games, &helper.num_games);
	sgf_parser_set_user_data(&parser, &helper);
	if (sgf_parser_parse(&parser) == NULL) {
		array_free((void **) &helper.games, &helper.num_games, sizeof(go_game), (void (*)(void *)) go_game_free);
		go_game_load_sgf_helper_free(&helper);
		sgf_parser_free(&parser);
		return NULL;
	}
	*games = helper.games;
	*num_games = helper.num_games;
	go_game_load_sgf_helper_free(&helper);
	sgf_parser_free(&parser);

	return games;
}

static void go_games_load_sgf_shallow_game_start(sgf_parser *parser)
{
	go_game_load_sgf_helper *helper;

	helper = (go_game_load_sgf_helper *) parser->user_data;
	if (go_game_init(helper->game) == NULL) {
		sgf_parser_stop(parser);
		return;
	}
	go_game_load_sgf_game_start(parser);
	sgf_parser_set_gametree_handlers(parser, go_game_load_sgf_gametree_start, go_game_load_sgf_gametree_end);
	sgf_parser_set_node_handlers(parser, go_game_load_sgf_node_start_root, go_games_load_sgf_shallow_node_end);
	sgf_parser_set_property_handler(parser, go_game_load_sgf_property_root);

	return;
}

static void go_games_load_sgf_shallow_node_end(sgf_parser *parser)
{
	go_game_load_sgf_node_end_root(parser);
	/* Ignore every node but the root */
	sgf_parser_set_gametree_handlers(parser, NULL, NULL);
	sgf_parser_set_node_handlers(parser, NULL, NULL);
	sgf_parser_set_property_handler(parser, NULL);

	return;
}

go_game **go_games_load_sgf_shallow(go_game **games, unsigned int *num_games, const char *filename)
{
	sgf_parser parser;
	go_game_load_sgf_helper helper;
	go_game game;

	array_init((void **) games, num_games);
	if (sgf_parser_init(&parser) == NULL)
		return NULL;
	if (sgf_parser_open_file(&parser, filename, "rb") == NULL)
		return NULL;
	sgf_parser_set_game_handlers(&parser, go_games_load_sgf_shallow_game_start, go_games_load_sgf_game_end);
	/* The gametree, node and property handlers are set in
	   go_games_load_sgf_shallow_game_start */
	if (go_game_load_sgf_helper_init(&helper) == NULL) {
		sgf_parser_free(&parser);
		return NULL;
	}
	helper.game = &game;
	array_init((void **) &helper.games, &helper.num_games);
	sgf_parser_set_user_data(&parser, &helper);
	if (sgf_parser_parse(&parser) == NULL) {
		array_free((void **) games, num_games, sizeof(go_game), (void (*)(void *)) go_game_free);
		go_game_load_sgf_helper_free(&helper);
		sgf_parser_free(&parser);
		return NULL;
	}
	*games = helper.games;
	*num_games = helper.num_games;
	go_game_load_sgf_helper_free(&helper);
	sgf_parser_free(&parser);

	return games;
}

const go_game *go_game_save_sgf(const go_game *game, const char *filename)
{
	sgf_parser parser;
	sgf_game g, *result;

	if (sgf_parser_init(&parser) == NULL)
		return NULL;
	if (sgf_game_from_go_game(&g, game) == NULL) {
		sgf_parser_free(&parser);
		return NULL;
	}
	if (sgf_parser_open_file(&parser, filename, "wb") == NULL) {
		sgf_game_free(&g);
		return NULL;
	}
	result = sgf_game_write(&g, &parser);
	sgf_parser_free(&parser);
	sgf_game_free(&g);
	if (result == NULL)
		return NULL;

	return game;
}
