/*
	game_ps.c
*/

#include "config.h"

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "game.h"

#define MAGNIFICATION (23.0 / 2.2 * 2.54 / 72.0)
/*
#define MAGNIFICATION (0.4)
*/
#define MAX_SCALE (50.0)

static void go_game_print_postscript(const go_game *game, FILE *fp);

static void print_header(FILE *fp, double boundingbox_width, double boundingbox_height);

static void print_procedures(FILE *fp);

static void print_star_points(FILE *fp, const go_game *game);

static void print_stones(FILE *fp, const go_game *game);

static void print_string_escaped(FILE *fp, const char *s);

static void print_markup(FILE *fp, const go_game *game);

static void print_footer(FILE *fp);

static double distance(const go_point_location *a, const go_point_location *b);

static void go_game_print_postscript_graph(const go_game *game, FILE *fp);

static void go_game_print_postscript_rectangular(const go_game *game, FILE *fp);

static void (*go_game_print_postscript_square)(const go_game *, FILE *) = go_game_print_postscript_rectangular;

static void go_game_print_postscript(const go_game *game, FILE *fp)
{
	switch (game->board_info.type) {
		case GO_GAME_BOARD_INFO_TYPE_SQUARE:
			go_game_print_postscript_square(game, fp);
			break;
		case GO_GAME_BOARD_INFO_TYPE_RECTANGULAR:
			go_game_print_postscript_rectangular(game, fp);
			break;
		default:
			go_game_print_postscript_graph(game, fp);
			break;
	}

	return;
}

static void print_header(FILE *fp, double boundingbox_width, double boundingbox_height)
{
	boundingbox_width *= MAGNIFICATION;
	boundingbox_height *= MAGNIFICATION;
	fprintf(fp, "\
%%!PS-Adobe-3.0 EPSF-3.0\n\
%%%%BoundingBox: 0 0 %u %u\n\
%%%%HiResBoundingBox: 0 0 %.4f %.4f\n\
%%%%EndComments\n", (unsigned int) ceil(boundingbox_width),
                (unsigned int) ceil(boundingbox_height),
                boundingbox_width, boundingbox_height);
	fprintf(fp, "\
\n\
/DeviceRGB setcolorspace\n\
0 0 0 setcolor\n");
	fprintf(fp, "\
\n\
/magnification %g def\n", MAGNIFICATION);

	return;
}

static void print_procedures(FILE *fp)
{
	fprintf(fp, "\
\n\
/invert { 1 exch sub } bind def\n");
	fprintf(fp, "\
\n\
/drawstone {\n\
\tgsave\n\
\t\tnewpath\n\
\t\txy stoneradius stoneborder 2 div sub 0 360 arc closepath\n\
\t\tgsave setcolor fill\n\
\t\tgrestore 0.0 0.0 0.0 setcolor stoneborder setlinewidth stroke\n\
\tgrestore\n\
} bind def\n");
	fprintf(fp, "\
\n\
/drawstarpoint {\n\
\tgsave\n\
\t\tnewpath\n\
\t\txy starpointradius 0 360 arc closepath fill\n\
\t\tgrestore\n\
} bind def\n");
	fprintf(fp, "\
\n\
/drawcircle {\n\
\tgsave\n\
\t\tnewpath\n\
\t\txy markupradius 0.8 mul 0 360 arc closepath\n\
\t\tgsave dup invert dup dup setcolor markupwidth 1.5 mul setlinewidth stroke\n\
\t\tgrestore dup dup setcolor markupwidth setlinewidth stroke\n\
\tgrestore\n\
} bind def\n");
	fprintf(fp, "\
\n\
/drawsquare {\n\
\tgsave\n\
\t\t0 setlinejoin\n\
\t\t/len markupradius 45 sin mul 2 mul def\n\
\t\tnewpath\n\
\t\txy exch len 2 div add exch len 2 div add moveto\n\
\t\tlen neg 0 rlineto 0 len neg rlineto len 0 rlineto closepath\n\
\t\tgsave dup invert dup dup setcolor markupwidth 1.5 mul setlinewidth stroke\n\
\t\tgrestore dup dup setcolor markupwidth setlinewidth stroke\n\
\tgrestore\n\
} bind def\n");
	fprintf(fp, "\
\n\
/drawtriangle {\n\
\tgsave\n\
\t\t0 setlinejoin\n\
\t\tnewpath\n\
\t\txy 1 index 1 index 1 index 1 index\n\
\t\tmarkupradius 90 sin mul add exch markupradius 90 cos mul add exch moveto\n\
\t\tmarkupradius 210 sin mul add exch markupradius 210 cos mul add exch lineto\n\
\t\tmarkupradius 330 sin mul add exch markupradius 330 cos mul add exch lineto\n\
\t\tclosepath\n");
	fprintf(fp, "\
\t\tgsave dup invert dup dup setcolor markupwidth 1.5 mul setlinewidth stroke\n\
\t\tgrestore dup dup setcolor markupwidth setlinewidth stroke\n\
\tgrestore\n\
} bind def\n");
	fprintf(fp, "\
\n\
/drawx {\n\
\tgsave\n\
\t\t2 setlinecap\n\
\t\t/len markupradius 45 sin mul 2 mul def\n\
\t\tnewpath\n\
\t\txy exch len 2 div add exch len 2 div add moveto\n\
\t\tlen neg len neg rlineto 0 len rmoveto len len neg rlineto\n\
\t\tgsave dup invert dup dup setcolor markupwidth 1.5 mul setlinewidth stroke\n\
\t\tgrestore dup dup setcolor markupwidth setlinewidth stroke\n\
\tgrestore\n\
} bind def\n");
/*
	fprintf(fp, "\
\n\
/drawmove {\n\
\tgsave\n\
\t\tnewpath\n\
\t\txy markupwidth 2 div 0 360 arc closepath\n\
\t\tgsave dup invert dup dup setcolor markupwidth 1.5 mul setlinewidth stroke\n\
\t\tgrestore dup dup setcolor markupwidth setlinewidth stroke\n\
\tgrestore\n\
} bind def\n");
*/
/*
	fprintf(fp, "\
\n\
/drawmove {\n\
\tgsave\n\
\t\t1 setlinecap\n\
\t\t/len markupwidth 45 sin mul 2 mul def\n\
\t\tnewpath\n\
\t\txy exch len 2 div add exch len 2 div add moveto\n\
\t\tlen neg len neg rlineto 0 len rmoveto len len neg rlineto\n\
\t\tgsave dup invert dup dup setcolor markupwidth 1.5 mul setlinewidth stroke\n\
\t\tgrestore dup dup setcolor markupwidth setlinewidth stroke\n\
\tgrestore\n\
} bind def\n");
*/
	fprintf(fp, "\
\n\
/drawmove {\n\
\tgsave\n\
\t\t1 setlinecap\n\
\t\tnewpath\n\
\t\txy exch markupwidth sub exch moveto\n\
\t\tmarkupwidth 2 mul 0 rlineto markupwidth neg markupwidth rmoveto\n\
\t\t0 markupwidth 2 mul neg rlineto\n\
\t\tgsave dup invert dup dup setcolor markupwidth 1.5 mul setlinewidth stroke\n\
\t\tgrestore dup dup setcolor markupwidth setlinewidth stroke\n\
\tgrestore\n\
} bind def\n");
	fprintf(fp, "\
\n\
/drawline {\n\
\tgsave\n\
\t\t1 setlinecap\n\
\t\tnewpath\n\
\t\t1 index xy moveto\n\
\t\txy lineto\n\
\t\tpop setcolor markupwidth setlinewidth stroke\n\
\tgrestore\n\
} bind def\n");
	fprintf(fp, "\
\n\
/drawarrow {\n\
\tgsave\n\
\t\t1 setlinecap\n\
\t\tnewpath\n\
\t\t1 index xy moveto\n\
\t\txy lineto\n\
\t\tpop setcolor markupwidth setlinewidth stroke\n\
\t\tnewpath\n\
\t\tfill\n\
\tgrestore\n\
} bind def\n");
	fprintf(fp, "\
\n\
/drawlabel {\n\
\tgsave\n\
\t\t1 setlinejoin\n\
\t\t/Helvetica findfont markupradius 2 mul scalefont setfont\n\
\t\tnewpath\n\
\t\txy moveto\n\
\t\tdup stringwidth pop 2 div neg markupradius 2 div neg rmoveto\n\
\t\tfalse charpath\n\
\t\tgsave dup invert dup dup setcolor markupwidth 0.5 mul setlinewidth stroke\n\
\t\tgrestore dup dup setcolor fill\n\
\tgrestore\n\
} bind def\n");

	return;
}

static void print_star_points(FILE *fp, const go_game *game)
{
	unsigned int i;

	if (game->board_info.num_star_points == 0)
		return;
	putc('\n', fp);
	fprintf(fp, "[");
	for (i = 0; i < game->board_info.num_star_points; i++)
		fprintf(fp, " %u", game->board_info.star_points[i]);
	fprintf(fp, " ]\n\
{\n\
\tnewpath xy starpointradius 0 360 arc closepath fill\n\
} forall\n");

	return;
}

static void print_stones(FILE *fp, const go_game *game)
{
	unsigned int i, color;

	for (i = 0; i < game->board.num_points; i++) {
		if (game->board.points[i].color != 0)
			break;
	}
	if (i < game->board.num_points) {
		putc('\n', fp);
		for (; i < game->board.num_points; i++) {
			color = game->board.points[i].color;
			if ((color > 0) && (color <= game->num_teams)) {
				fprintf(fp, "%g %g %g %u drawstone\n",
				        game->teams[color - 1].display_color.r,
				        game->teams[color - 1].display_color.g,
				        game->teams[color - 1].display_color.b,
						i);
			}
		}
	}

	return;
}

static void print_string_escaped(FILE *fp, const char *s)
{
	const char *escapes = "\\()";
	size_t length;

	for (;;) {
		length = strcspn(s, escapes);
		if (length > 0)
			fwrite(s, sizeof(char), length, fp);
		s += length;
		if (*s == '\0')
			break;
		fprintf(fp, "\\%c", *s);
		s++;
	}

	return;
}

static void print_markup(FILE *fp, const go_game *game)
{
	const static struct {
		int type;
		char *name;
	} markup_procedure_names[] = {
		{ GO_GAME_MARKUP_CIRCLE, "drawcircle" },
		{ GO_GAME_MARKUP_SELECTED, NULL },
		{ GO_GAME_MARKUP_SQUARE, "drawsquare" },
		{ GO_GAME_MARKUP_TRIANGLE, "drawtriangle" },
		{ GO_GAME_MARKUP_X, "drawx" },
		{ GO_GAME_MARKUP_MOVE, "drawmove" },
		{ GO_GAME_MARKUP_LINE, "drawline" },
		{ GO_GAME_MARKUP_ARROW, "drawarrow" },
		{ GO_GAME_MARKUP_LABEL, "drawlabel" }
	};
	unsigned int i, markup_index, color;
	float markup_color;

	for (i = 0; i < game->num_markup; i++) {
		for (markup_index = 0; markup_index < sizeof(markup_procedure_names) / sizeof(*markup_procedure_names); markup_index++) {
			if (markup_procedure_names[markup_index].type == game->markup[i].type)
				break;
		}
		if (markup_index >= sizeof(markup_procedure_names) / sizeof(*markup_procedure_names))
			continue;
		color = game->board.points[game->markup[i].index].color;
		if ((color == 0) || (color >= game->num_teams)) {
			markup_color = 0.0;
		} else {
			if (game->teams[color - 1].display_color.r +
			    game->teams[color - 1].display_color.g +
			    game->teams[color - 1].display_color.b < 1.5)
				markup_color = 1.0;
			else
				markup_color = 0.0;
		}
		switch(game->markup[i].type) {
			case GO_GAME_MARKUP_CIRCLE:
			case GO_GAME_MARKUP_SQUARE:
			case GO_GAME_MARKUP_TRIANGLE:
			case GO_GAME_MARKUP_X:
			case GO_GAME_MARKUP_MOVE:
				fprintf(fp, "%g %u %s\n",
			            markup_color, game->markup[i].index,
						markup_procedure_names[markup_index].name);
				break;
			case GO_GAME_MARKUP_LINE:
				fprintf(fp, "%g %g %g %u %u %s\n",
						0.2, 0.2, 0.75,
			            game->markup[i].index, game->markup[i].data.endpoint,
						markup_procedure_names[markup_index].name);
				break;
			case GO_GAME_MARKUP_ARROW:
				fprintf(fp, "%g %g %g %u %u %s\n",
						0.75, 0.2, 0.2,
			            game->markup[i].index, game->markup[i].data.endpoint,
						markup_procedure_names[markup_index].name);
				break;
			case GO_GAME_MARKUP_LABEL:
				fprintf(fp, "%g (", markup_color);
				print_string_escaped(fp, game->markup[i].data.string);
				fprintf(fp, ") %u %s\n",
			            game->markup[i].index,
						markup_procedure_names[markup_index].name);
				break;
		}
	}

	return;
}

static void print_footer(FILE *fp)
{
	fprintf(fp, "\
\n\
%%%%EOF\n");

	return;
}

static double distance(const go_point_location *a, const go_point_location *b)
{
	return sqrt((a->x - b->x) * (a->x - b->x) +
	            (a->y - b->y) * (a->y - b->y) +
	            (a->z - b->z) * (a->z - b->z));
}

static void go_game_print_postscript_graph(const go_game *game, FILE *fp)
{
	static const float point_spacing = 2.2, edge_spacing = 1.4,
	                   stone_radius = 1.1, stone_border = 0.1,
	                   line_width = 0.1, star_point_radius = 0.2,
	                   markup_radius = 0.7, markup_width = 0.2;
	double boundingbox_width, boundingbox_height;
	float dist, min_dist, scaling;
	go_point_location *locations, extents;
	unsigned int i, j;

	locations = (go_point_location *) malloc(game->board.num_points * sizeof(go_point_location));
	if (locations == NULL)
		return;
	for (i = 0; i < game->board.num_points; i++) {
		locations[i].index = i;
		go_game_board_get_point_location(game, i, &locations[i].x, &locations[i].y, &locations[i].z);
		/* Flatten three-dimensional boards */
		locations[i].z = 0.0;
	}
	go_point_locations_normalize(locations, game->board.num_points, &extents);
	min_dist = point_spacing;
	for (i = 0; i < game->board.num_points; i++) {
		for (j = i + 1; j < game->board.num_points; j++) {
			dist = distance(&locations[i], &locations[j]);
			if ((dist > 0.0) && (dist < min_dist))
				min_dist = dist;
		}
	}
	scaling = point_spacing / min_dist;
	if (scaling > MAX_SCALE)
		scaling = MAX_SCALE;
	boundingbox_width = (edge_spacing * 2 + extents.x * scaling) / 2.54 * 72;
	boundingbox_height = (edge_spacing * 2 + extents.y * scaling) / 2.54 * 72;
	print_header(fp, boundingbox_width, boundingbox_height);
	fprintf(fp, "\
\n\
/cm {2.54 div 72 mul} def\n\
\n\
/pointspacing %g def\n\
/edgespacing %g cm def\n\
/stoneradius %g cm def\n\
/stoneborder %g cm def\n\
/linewidth %g cm def\n\
/starpointradius %g cm def\n\
/markupradius %g cm def\n\
/markupwidth %g cm def\n", point_spacing,
	            edge_spacing, stone_radius, stone_border, line_width,
	            star_point_radius, markup_radius, markup_width);
	fprintf(fp, "\
\n\
/locations [");
	for (i = 0; i < game->board.num_points; i++)
		fprintf(fp, " %g %g", locations[i].x, locations[i].y);
	fprintf(fp, " ] def\n");
	free(locations);
	fprintf(fp, "\
\n\
/scaling %g def\n\
\n\
/xy {\n\
\tdup add dup\n\
\tlocations exch get scaling mul cm exch\n\
\tlocations exch 1 add get scaling mul cm\n\
} bind def\n", scaling);
	print_procedures(fp);
	fprintf(fp, "\
\n\
magnification magnification scale\n\
edgespacing edgespacing translate\n");
	fprintf(fp, "\
\n\
linewidth setlinewidth\n\
1 setlinecap\n\
\n");
	for (i = 0; i < game->board.num_points; i++) {
		if (game->board.points[i].num_adjacent == 0) {
			fprintf(fp, "%u drawstarpoint\n", i);
		} else {
			for (j = 0; j < game->board.points[i].num_adjacent; j++) {
				if (game->board.points[i].adjacent[j] > i) {
					fprintf(fp, "newpath %u xy moveto\n", i);
					for (; j < game->board.points[i].num_adjacent; j++) {
						if (game->board.points[i].adjacent[j] > i)
							fprintf(fp, "gsave %u xy lineto stroke grestore\n", game->board.points[i].adjacent[j]);
					}
					break;
				}
			}
		}
	}
	print_star_points(fp, game);
	print_stones(fp, game);
	print_markup(fp, game);
	print_footer(fp);

	return;
}

static void go_game_print_postscript_rectangular(const go_game *game, FILE *fp)
{
	static const float x_spacing = 2.2, y_spacing = 2.2, edge_spacing = 1.4,
	                   stone_radius = 1.1, stone_border = 0.1,
	                   line_width = 0.1, star_point_radius = 0.2,
	                   markup_radius = 0.7, markup_width = 0.2;
	double boundingbox_width, boundingbox_height;
	unsigned int width, height;

	if (game->board_info.type == GO_GAME_BOARD_INFO_TYPE_SQUARE) {
		width = game->board_info.size.square.size;
		height = game->board_info.size.square.size;
	} else {
		width = game->board_info.size.rectangular.width;
		height = game->board_info.size.rectangular.height;
	}
	boundingbox_width = ((width - 1) * x_spacing + edge_spacing * 2) / 2.54 * 72;
	boundingbox_height = ((height - 1) * y_spacing + edge_spacing * 2) / 2.54 * 72;
	print_header(fp, boundingbox_width, boundingbox_height);
	fprintf(fp, "\
\n\
/width %u def\n\
/height %u def\n\
\n\
/cm {2.54 div 72 mul} def\n\
\n\
/xspacing %g cm def\n\
/yspacing %g cm def\n\
/edgespacing %g cm def\n\
/stoneradius %g cm def\n\
/stoneborder %g cm def\n\
/linewidth %g cm def\n\
/starpointradius %g cm def\n\
/markupradius %g cm def\n\
/markupwidth %g cm def\n", width, height, x_spacing, y_spacing,
	            edge_spacing, stone_radius, stone_border, line_width,
	            star_point_radius, markup_radius, markup_width);
	fprintf(fp, "\
\n\
/xy {\n\
\tdup\n\
\texch width mod xspacing mul\n\
\texch width 1 sub exch width idiv sub yspacing mul\n\
} bind def\n");
	print_procedures(fp);
	fprintf(fp, "\
\n\
magnification magnification scale\n");
/* Board background */
	fprintf(fp, "\
\n\
gsave\n\
0.94 0.69 0.42 setcolor\n\
0 0 moveto %f cm 0 rlineto 0 %f cm rlineto %f cm 0 rlineto closepath fill\n\
grestore\n",
	        ((width - 1) * x_spacing + edge_spacing * 2),
	        ((height - 1) * y_spacing + edge_spacing * 2),
	        -((width - 1) * x_spacing + edge_spacing * 2));
	fprintf(fp, "\
edgespacing edgespacing translate\n");
	fprintf(fp, "\
\n\
linewidth setlinewidth\n\
\n\
newpath\n\
0 0 moveto\n\
xspacing width 1 sub mul 0 rlineto\n\
0 yspacing height 1 sub mul rlineto\n\
xspacing width 1 sub mul neg 0 rlineto\n\
closepath\n\
stroke\n\
newpath\n\
1 1 width 2 sub {\n\
\txspacing mul 0 moveto\n\
\t0 yspacing height 1 sub mul rlineto\n\
} for\n\
1 1 height 2 sub {\n\
\t0 exch yspacing mul moveto\n\
\txspacing width 1 sub mul 0 rlineto\n\
} for\n\
stroke\n");
	print_star_points(fp, game);
	print_stones(fp, game);
	print_markup(fp, game);
	print_footer(fp);

	return;
}

const go_game *go_game_save_postscript(const go_game *game, const char *filename)
{
	FILE *fp;

	fp = fopen(filename, "wb");
	if (fp == NULL)
		return NULL;
	go_game_print_postscript(game, fp);
	if (fclose(fp) == EOF)
		return NULL;

	return game;
}
