/*
	game.h
*/

#ifndef _GAME_H
#define _GAME_H

#include "config.h"

#include <stdarg.h>

#include "board.h"
#include "ruleset.h"

enum {
	GO_GAME_BOARD_INFO_TYPE_GRAPH = 0,
	GO_GAME_BOARD_INFO_TYPE_SQUARE = 1,
	GO_GAME_BOARD_INFO_TYPE_RECTANGULAR = 2
};

enum {
	GO_HISTORY_TURN_ACTION_COLOR_POINT,
	GO_HISTORY_TURN_ACTION_MOVE,
	GO_HISTORY_TURN_ACTION_PASS,
	GO_HISTORY_TURN_ACTION_RESIGN,
	GO_HISTORY_TURN_ACTION_SET_PLAYER,
	GO_HISTORY_TURN_ACTION_COMMENT,
	GO_HISTORY_TURN_ACTION_DEAD,
	GO_HISTORY_TURN_ACTION_MARKUP
};

enum {
	GO_GAME_MARKUP_CLEAR,
	GO_GAME_MARKUP_CIRCLE,
	GO_GAME_MARKUP_SELECTED,
	GO_GAME_MARKUP_SQUARE,
	GO_GAME_MARKUP_TRIANGLE,
	/* GO_GAME_MARKUP_TERRITORY, */
	GO_GAME_MARKUP_X,
	GO_GAME_MARKUP_MOVE,
	GO_GAME_MARKUP_LINE,
	GO_GAME_MARKUP_ARROW,
	GO_GAME_MARKUP_LABEL
};

typedef struct go_player go_player;
struct go_player {
	char *name;
	/* Rank */
};

typedef struct go_team go_team;
struct go_team {
	char *name;
	/* Can the color be inferred from the index? */
	unsigned int color;
	unsigned int handicap;
	unsigned int num_players;
	unsigned int *players;
	struct {
		float r;
		float g;
		float b;
	} display_color;
	char display_char;
	char alt_display_char;
	unsigned int current_player;
	unsigned int next_player;
	unsigned int is_resigned;
	float komi;
	float territory;
	float num_occupied_points;
	float num_prisoners;
	float score;
};

typedef struct go_board_state go_board_state;
struct go_board_state {
	unsigned int num_points;
	unsigned int *colors;
};

typedef struct go_team_state go_team_state;
struct go_team_state {
	unsigned int current_player;
	unsigned int next_player;
	float num_prisoners;
	float score;
	unsigned int is_resigned;
};

typedef struct go_game_markup go_game_markup;
struct go_game_markup {
	int type;
	unsigned int index;
	union {
		unsigned int endpoint;
		char *string;
	} data;
};

typedef struct go_game_state go_game_state;
struct go_game_state {
	go_board_state board_state;
	go_team_state *team_states;
	unsigned int current_team;
	unsigned int next_team;
	unsigned int turn_num;
	unsigned int num_passes;
	unsigned int num_resignations;
	unsigned int handicap_team;
	unsigned int handicap_counter;
	unsigned int num_dead_strings;
	unsigned int *dead_strings;
	unsigned int num_markup;
	go_game_markup *markup;
};

typedef struct go_history_turn_action go_history_turn_action;
struct go_history_turn_action {
	int type;
	union {
		unsigned int integer;
		struct {
			unsigned int index;
			unsigned int color;
		} index_color;
		struct {
			unsigned int team_index;
			unsigned int player_index;
		} player;
		char *string;
		go_game_markup markup;
	} value;
};

typedef struct go_history_turn go_history_turn;
struct go_history_turn {
	go_game_state game_state;
	unsigned int num_actions;
	go_history_turn_action *actions;
	go_history_turn *prev;
	go_history_turn *next;
	go_history_turn *variation;
	/* Maybe make another next that is the currently selected variation
	   (not necessarily the main one) */
};

typedef struct go_history_hash_table_bucket go_history_hash_table_bucket;
struct go_history_hash_table_bucket {
	unsigned int num_turns;
	go_history_turn **turns;
};

typedef struct go_history go_history;
struct go_history {
	go_history_turn *history;
	go_history_turn *current_turn;
	unsigned int hash_table_size;
	unsigned int hash_table_num_nodes;
	go_history_hash_table_bucket *hash_table;
};

typedef struct go_point_location go_point_location;
struct go_point_location {
	unsigned int index;
	float x;
	float y;
	float z;
};

typedef struct go_game_info go_game_info;
struct go_game_info {
	char *name;
	char *value;
};

typedef struct go_game go_game;
struct go_game {
	go_board board;
	go_ruleset ruleset;
	go_history history;
	struct {
		unsigned int type;
		union {
			struct {
				unsigned int size;
			} graph;
			struct {
				unsigned int size;
			} square;
			struct {
				unsigned int width;
				unsigned int height;
			} rectangular;
		} size;
		unsigned int num_star_points;
		unsigned int *star_points;
		unsigned int num_point_locations;
		go_point_location *point_locations;
	} board_info;
	unsigned int num_info;
	go_game_info *info;
	unsigned int num_teams;
	go_team *teams;
	unsigned int current_team;
	unsigned int next_team;
	unsigned int num_players;
	go_player *players;
	unsigned int turn_num;
	unsigned int num_passes;
	unsigned int num_resignations;
	unsigned int handicap_team;
	unsigned int handicap_counter;
	unsigned int num_dead_strings;
	unsigned int *dead_strings;
	unsigned int num_markup;
	go_game_markup *markup;
};

go_player *go_player_set_name(go_player *player, const char *name);

go_team *go_team_set_name(go_team *team, const char *name);

go_game *go_game_init(go_game *game);

void go_game_free(go_game *game);

unsigned int go_game_board_index(go_game *game, ...);

go_game *go_game_board_set_size_graph(go_game *game, unsigned int size);

go_game *go_game_board_set_size_square(go_game *game, unsigned int size);

go_game *go_game_board_set_size_rectangular(go_game *game, unsigned int width, unsigned int height);

go_game *go_game_history_seek(go_game *game, go_history_turn *turn);

go_game *go_game_history_rewind(go_game *game);

go_game *go_game_history_end(go_game *game);

go_game *go_game_history_back(go_game *game);

go_game *go_game_history_forward(go_game *game);

go_game *go_game_history_undo(go_game *game);

go_game *go_game_board_set_point_location(go_game *game, unsigned int index, float x, float y, float z);

void go_game_board_get_point_location(const go_game *game, unsigned int index, float *x, float *y, float *z);

go_point_location *go_point_locations_normalize(go_point_location *locations, unsigned int num_locations, go_point_location *extents);

go_game *go_game_board_add_star_point(go_game *game, unsigned int index);

go_game *go_game_board_remove_star_point(go_game *game, unsigned int index);

go_game *go_game_info_set(go_game *game, const char *name, const char *value);

const char *go_game_info_search(const go_game *game, const char *name);

go_player *go_game_new_player(go_game *game);

go_player *go_game_new_player_team(go_game *game, unsigned int index);

go_game *go_game_remove_player(go_game *game, unsigned int index);

go_team *go_game_new_team(go_game *game);

go_game *go_game_remove_team(go_game *game, unsigned int index);

go_game *go_game_team_set_handicap(go_game *game, unsigned int index, unsigned int handicap);

go_game *go_game_start(go_game *game);

go_game *go_game_end(go_game *game);

go_game *go_game_new_turn(go_game *game);

go_game *go_game_color_point(go_game *game, unsigned int index, unsigned int color);

int go_game_move_is_legal(const go_game *game, unsigned int index);

go_game *go_game_move(go_game *game, unsigned int index);

int go_game_pass_is_legal(const go_game *game);

go_game *go_game_pass(go_game *game);

go_game *go_game_resign(go_game *game);

go_game *go_game_set_current_player(go_game *game, unsigned int team_index, unsigned int player_index);

go_game *go_game_comment(go_game *game, const char *s);

go_game *go_game_toggle_dead(go_game *game, unsigned int index);

int go_game_is_dead(const go_game *game, unsigned int index);

go_game *go_game_clear_markup(go_game *game);

go_game *go_game_mark(go_game *game, unsigned int index, int type);

go_game *go_game_line(go_game *game, unsigned int start, unsigned int end, int type);

go_game *go_game_label(go_game *game, unsigned int index, const char *string);

go_game *go_game_count_score(go_game *game);

int go_game_is_over(const go_game *game);

int go_game_is_over_turn(const go_game *game, const go_history_turn *turn);

go_game *go_game_load_xml(go_game *game, const char *filename);

const go_game *go_game_save_xml(const go_game *game, const char *filename);

go_game *go_game_load_sgf_index(go_game *game, const char *filename, unsigned int index);

go_game *go_game_load_sgf(go_game *game, const char *filename);

go_game **go_games_load_sgf(go_game **games, unsigned int *num_games, const char *filename);

go_game **go_games_load_sgf_shallow(go_game **games, unsigned int *num_games, const char *filename);

const go_game *go_game_save_sgf(const go_game *game, const char *filename);

const go_game *go_game_save_postscript(const go_game *game, const char *filename);

#endif
