/*
	game.c
*/

#include "config.h"

#include <stdarg.h>
#include <stdlib.h>
#include <string.h>

#include "array.h"
#include "board.h"
#include "game.h"
#include "ruleset.h"

/* These are defined in history.c */

extern void go_game_markup_free(go_game_markup *markup);

extern go_game *go_game_from_state(go_game *game, const go_game_state *game_state);

extern go_game_state *go_game_state_from_game(go_game_state *game_state, const go_game *game);

extern go_history *go_history_init(go_history *history);

extern void go_history_free(go_history *history);

extern go_history_turn *go_history_new_turn(go_history *history);

extern go_history *go_history_delete_turn(go_history *history, go_history_turn *turn);

extern void go_history_search_board(const go_history *history, const go_board *board, go_history_turn ***array, unsigned int *num);

extern go_history *go_history_add_action(go_history *history, const go_history_turn_action *action);

extern go_history *go_history_hash_table_add(go_history *history, const go_history_turn *turn);

extern go_history *go_history_hash_table_remove(go_history *history, const go_history_turn *turn);

static int index_cmp(const unsigned int *a, const unsigned int *b);

static go_player *go_player_init(go_player *player);

static void go_player_free(go_player *player);

static go_team *go_team_init(go_team *team);

static void go_team_free(go_team *team);

static go_team *go_team_add_player(go_team *team, unsigned int index);

static go_team *go_team_remove_player(go_team *team, unsigned int index);

static go_game *go_game_set_state(go_game *game);

static go_game *go_game_unset_state(go_game *game);

static go_game *go_game_get_state(go_game *game);

static int go_point_location_cmp(const go_point_location *a, const go_point_location *b);

static int go_point_location_index_cmp(const unsigned int *index, const go_point_location *location);

static go_point_location *go_game_board_get_point_location_graph(const go_game *game, unsigned int index);

static go_point_location *go_game_board_get_point_location_square(const go_game *game, unsigned int index);

static go_point_location *go_game_board_get_point_location_rectangular(const go_game *game, unsigned int index);

static void go_game_info_free(go_game_info *info);

static int go_game_info_name_cmp(const char *name, const go_game_info *info);

static go_game *go_game_next_player(go_game *game);

static int go_board_board_state_cmp(const go_board *board, const go_board_state *state);

static int go_game_is_ko_violation(const go_game *game);

static go_game *go_game_do_move(go_game *game, unsigned int index, unsigned int color, int check);

static go_game *go_game_add_dead_string(go_game *game, unsigned int index);

static go_game *go_game_remove_dead_string(go_game *game, unsigned int index);

static int go_game_markup_cmp(const go_game_markup *a, const go_game_markup *b);

static go_game *go_game_add_markup(go_game *game, const go_game_markup *markup);

static int index_cmp(const unsigned int *a, const unsigned int *b)
{
	return *(int *) a - *(int *) b;
}

static go_player *go_player_init(go_player *player)
{
	player->name = NULL;

	return player;
}

static void go_player_free(go_player *player)
{
	free(player->name);

	return;
}

go_player *go_player_set_name(go_player *player, const char *name)
{
	char *p;

	p = (char *) realloc(player->name, strlen(name) + 1);
	if (p == NULL)
		return NULL;
	strcpy(p, name);
	player->name = p;

	return player;
}

go_team *go_team_set_name(go_team *team, const char *name)
{
	char *p;

	p = (char *) realloc(team->name, strlen(name) + 1);
	if (p == NULL)
		return NULL;
	strcpy(p, name);
	team->name = p;

	return team;
}

static go_team *go_team_init(go_team *team)
{
	team->name = NULL;
	team->color = 0;
	team->handicap = 0;
	array_init((void **) &team->players, &team->num_players);
	team->display_color.r = 0.0;
	team->display_color.g = 0.0;
	team->display_color.b = 0.0;
	team->display_char = '?';
	team->alt_display_char = '?';
	team->current_player = 0;
	team->next_player = 0;
	team->is_resigned = 0;
	team->komi = 0.0;
	team->territory = 0.0;
	team->num_occupied_points = 0.0;
	team->num_prisoners = 0.0;
	team->score = 0.0;

	return team;
}

static void go_team_free(go_team *team)
{
	free(team->name);
	array_free((void **) &team->players, &team->num_players, sizeof(unsigned int), NULL);

	return;
}

static go_team *go_team_add_player(go_team *team, unsigned int index)
{
	if (array_insert_no_duplicates((void **) &team->players, &team->num_players, sizeof(unsigned int), &index, (int (*)(const void *, const void *)) index_cmp) == NULL)
		return NULL;

	return team;
}

static go_team *go_team_remove_player(go_team *team, unsigned int index)
{
	if (array_remove_keep_order((void **) &team->players, &team->num_players, sizeof(unsigned int), &index, (int (*)(const void *, const void *)) index_cmp) == NULL)
		return NULL;

	return team;
}

go_game *go_game_init(go_game *game)
{
	if (go_board_init(&game->board) == NULL)
		return NULL;
	game->ruleset = go_ruleset_logical;
	if (go_history_init(&game->history) == NULL)
		return NULL;
	if (go_game_board_set_size_graph(game, 0) == NULL)
		return NULL;
	array_init((void **) &game->board_info.star_points, &game->board_info.num_star_points);
	array_init((void **) &game->board_info.point_locations, &game->board_info.num_point_locations);
	array_init((void **) &game->info, &game->num_info);
	/*
	array_init((void **) &game->info.dates, &game->info.num_dates);
	*/
	array_init((void **) &game->teams, &game->num_teams);
	game->current_team = 0;
	game->next_team = 0;
	array_init((void **) &game->players, &game->num_players);
	game->turn_num = 0;
	game->num_passes = 0;
	game->num_resignations = 0;
	game->handicap_team = 0;
	game->handicap_counter = 0;
	array_init((void **) &game->dead_strings, &game->num_dead_strings);
	array_init((void **) &game->markup, &game->num_markup);

	return game;
}

void go_game_free(go_game *game)
{
	go_board_free(&game->board);
	go_history_free(&game->history);
	array_free((void **) &game->board_info.star_points, &game->board_info.num_star_points, sizeof(unsigned int), NULL);
	array_free((void **) &game->board_info.point_locations, &game->board_info.num_point_locations, sizeof(go_point_location), NULL);
	array_free((void **) &game->info, &game->num_info, sizeof(go_game_info), (void (*)(void *)) go_game_info_free);
	/*
	array_free((void **) &game->info.dates, &game->info.num_dates, sizeof(time_t), NULL);
	*/
	array_free((void **) &game->teams, &game->num_teams, sizeof(go_team), (void (*)(void *)) go_team_free);
	array_free((void **) &game->players, &game->num_players, sizeof(go_player), (void (*)(void *)) go_player_free);
	/* The markup is freed with the history */

	return;
}

unsigned int go_game_board_index(go_game *game, ...)
{
	va_list ap;
	unsigned int index, x, y;

	va_start(ap, game);
	switch (game->board_info.type) {
		case GO_GAME_BOARD_INFO_TYPE_SQUARE:
			x = va_arg(ap, unsigned int);
			y = va_arg(ap, unsigned int);
			index = y * game->board_info.size.square.size + x;
			break;
		case GO_GAME_BOARD_INFO_TYPE_RECTANGULAR:
			x = va_arg(ap, unsigned int);
			y = va_arg(ap, unsigned int);
			index = y * game->board_info.size.rectangular.width + x;
			break;
		default:
			index = va_arg(ap, unsigned int);
			break;
	}
	va_end(ap);

	return index;
}

go_game *go_game_board_set_size_graph(go_game *game, unsigned int size)
{
	if (go_board_set_num_points(&game->board, size) == NULL)
		return NULL;
	game->board_info.type = GO_GAME_BOARD_INFO_TYPE_GRAPH;
	game->board_info.size.graph.size = size;

	return game;
}

go_game *go_game_board_set_size_square(go_game *game, unsigned int size)
{
	if (go_game_board_set_size_rectangular(game, size, size) == NULL)
		return NULL;
	game->board_info.type = GO_GAME_BOARD_INFO_TYPE_SQUARE;
	game->board_info.size.square.size = size;

	return game;
}

go_game *go_game_board_set_size_rectangular(go_game *game, unsigned int width, unsigned int height)
{
	unsigned int star_points[9];
	unsigned int i, x, y, index, x_corner, y_corner;

	if (go_board_set_num_points(&game->board, width * height) == NULL)
		return NULL;
	for (y = 0; y < height; y++) {
		for (x = 0; x < width - 1; x++) {
			index = y * width + x;
			if (go_board_connect_points(&game->board, index, index + 1) == NULL)
				return NULL;
		}
	}
	for (x = 0; x < width; x++) {
		for (y = 0; y < height - 1; y++) {
			index = y * width + x;
			if (go_board_connect_points(&game->board, index, index + width) == NULL)
				return NULL;
		}
	}
	game->board_info.type = GO_GAME_BOARD_INFO_TYPE_RECTANGULAR;
	game->board_info.size.rectangular.width = width;
	game->board_info.size.rectangular.height = height;
	array_free((void **) &game->board_info.star_points, &game->board_info.num_star_points, sizeof(unsigned int), NULL);
	array_init((void **) &game->board_info.star_points, &game->board_info.num_star_points);
	if (width < 13)
		x_corner = 2;
	else
		x_corner = 3;
	if (height < 13)
		y_corner = 2;
	else
		y_corner = 3;
	i = 0;
	if ((width >= 5) && (height >= 5)) {
		if ((width % 2 != 0) && (height % 2 != 0) &&
		    ((width == height) || ((width >= 19) && (height >= 19))))
			star_points[i++] = go_game_board_index(game, width / 2, height / 2);
		if ((width % 2 != 0) && (width >= 19)) {
			star_points[i++] = go_game_board_index(game, width / 2, y_corner);
			star_points[i++] = go_game_board_index(game, width / 2, height - y_corner - 1);
		}
		if ((height % 2 != 0) && (height >= 19)) {
			star_points[i++] = go_game_board_index(game, x_corner, height / 2);
			star_points[i++] = go_game_board_index(game, width - x_corner - 1, height / 2);
		}
		if ((width >= 9) && (height >= 9)) {
			star_points[i++] = go_game_board_index(game, x_corner, y_corner);
			star_points[i++] = go_game_board_index(game, width - x_corner - 1, y_corner);
			star_points[i++] = go_game_board_index(game, x_corner, height - y_corner - 1);
			star_points[i++] = go_game_board_index(game, width - x_corner - 1, height - y_corner - 1);
		}
	}
	for (; i > 0; i--) {
		if (go_game_board_add_star_point(game, star_points[i - 1]) == NULL) {
			go_board_free(&game->board);
			array_free((void **) &game->board_info.star_points, &game->board_info.num_star_points, sizeof(unsigned int), NULL);
			return NULL;
		}
	}

	return game;
}

static go_game *go_game_set_state(go_game *game)
{
	if (go_game_state_from_game(&game->history.current_turn->game_state, game) == NULL)
		return NULL;
	if (go_history_hash_table_add(&game->history, game->history.current_turn) == NULL)
		return NULL;

	return game;
}

static go_game *go_game_unset_state(go_game *game)
{
	if (go_history_hash_table_remove(&game->history, game->history.current_turn) == NULL)
		return NULL;

	return game;
}

static go_game *go_game_get_state(go_game *game)
{
	if (go_game_from_state(game, &game->history.current_turn->game_state) == NULL)
		return NULL;

	return game;
}

go_game *go_game_history_seek(go_game *game, go_history_turn *turn)
{
	if (turn == NULL)
		return NULL;
	game->history.current_turn = turn;
	if (go_game_get_state(game) == NULL)
		return NULL;

	return game;
}

go_game *go_game_history_rewind(go_game *game)
{
	return go_game_history_seek(game, game->history.history);
}

go_game *go_game_history_end(go_game *game)
{
	go_history_turn *turn;

	for (turn = game->history.current_turn; turn->next != NULL; turn = turn->next)
		;

	return go_game_history_seek(game, turn);
}

go_game *go_game_history_back(go_game *game)
{
	return go_game_history_seek(game, game->history.current_turn->prev);
}

go_game *go_game_history_forward(go_game *game)
{
	return go_game_history_seek(game, game->history.current_turn->next);
}

go_game *go_game_history_undo(go_game *game)
{
	go_history_turn *turn;

	turn = game->history.current_turn;
	if (turn->prev != NULL) {
		if (go_game_history_seek(game, turn->prev) == NULL)
			return NULL;
		if (go_history_delete_turn(&game->history, turn) == NULL)
			return NULL;
	} else {
		/* Undoing the first turn restarts the game */
		game->history.current_turn = NULL;
		if (go_history_delete_turn(&game->history, turn) == NULL)
			return NULL;
		if (go_game_start(game) == NULL)
			return NULL;
	}

	return game;
}

static int go_point_location_cmp(const go_point_location *a, const go_point_location *b)
{
	return (int) a->index - (int) b->index;
}

go_game *go_game_board_set_point_location(go_game *game, unsigned int index, float x, float y, float z)
{
	go_point_location location;

	if (index >= game->board.num_points)
		return NULL;
	/* This function does nothing for board types other than graph */
	if (game->board_info.type != GO_GAME_BOARD_INFO_TYPE_GRAPH)
		return game;
	location.index = index;
	location.x = x;
	location.y = y;
	location.z = z;
	if (array_insert_sorted_no_duplicates((void **) &game->board_info.point_locations, &game->board_info.num_point_locations, sizeof(go_point_location), &location, (int (*)(const void *, const void *)) go_point_location_cmp) == NULL)
		return NULL;

	return game;
}

void go_game_board_get_point_location(const go_game *game, unsigned int index, float *x, float *y, float *z)
{
	go_point_location *location;

	switch (game->board_info.type) {
		case GO_GAME_BOARD_INFO_TYPE_SQUARE:
			location = go_game_board_get_point_location_square(game, index);
			break;
		case GO_GAME_BOARD_INFO_TYPE_RECTANGULAR:
			location = go_game_board_get_point_location_rectangular(game, index);
			break;
		default:
			location = go_game_board_get_point_location_graph(game, index);
			break;
	}
	if (location == NULL) {
		*x = 0.0;
		*y = 0.0;
		*z = 0.0;
	} else {
		*x = location->x;
		*y = location->y;
		*z = location->z;
	}

	return;
}

static int go_point_location_index_cmp(const unsigned int *index, const go_point_location *location)
{
	return *(int *) index - (int) location->index;
}

static go_point_location *go_game_board_get_point_location_graph(const go_game *game, unsigned int index)
{
	return (go_point_location *) array_search_sorted((const void **) &game->board_info.point_locations, &game->board_info.num_point_locations, sizeof(go_point_location), &index, (int (*)(const void *, const void *)) go_point_location_index_cmp);
}

static go_point_location *go_game_board_get_point_location_square(const go_game *game, unsigned int index)
{
	return go_game_board_get_point_location_rectangular(game, index);
}

static go_point_location *go_game_board_get_point_location_rectangular(const go_game *game, unsigned int index)
{
	static go_point_location location;
	unsigned int width, height;

	if (game->board_info.type == GO_GAME_BOARD_INFO_TYPE_SQUARE) {
		width = game->board_info.size.square.size;
		height = game->board_info.size.square.size;
	} else {
		width = game->board_info.size.rectangular.width;
		height = game->board_info.size.rectangular.height;
	}
	location.x = (float) (index % width) / (width - 1);
	location.y = (float) ((width - 1) - (index / width)) / (height - 1);
	location.z = 0.0;

	return &location;
}

go_point_location *go_point_locations_normalize(go_point_location *locations, unsigned int num_locations, go_point_location *extents)
{
	float min_x, min_y, max_x, max_y, min_z, max_z, scale;
	unsigned int i;

	if (num_locations == 0)
		return locations;
	min_x = locations[0].x;
	max_x = locations[0].x;
	min_y = locations[0].y;
	max_y = locations[0].y;
	min_z = locations[0].z;
	max_z = locations[0].z;
	for (i = 1; i < num_locations; i++) {
		if (locations[i].x < min_x)
			min_x = locations[i].x;
		else if (locations[i].x > max_x)
			max_x = locations[i].x;
		if (locations[i].y < min_y)
			min_y = locations[i].y;
		else if (locations[i].y > max_y)
			max_y = locations[i].y;
		if (locations[i].z < min_z)
			min_z = locations[i].z;
		else if (locations[i].z > max_z)
			max_z = locations[i].z;
	}
	scale = max_x - min_x;
	if (max_y - min_y > scale)
		scale = max_y - min_y;
	if (max_z - min_z > scale)
		scale = max_z - min_z;
	if (scale == 0.0) {
		for (i = 0; i < num_locations; i++) {
			locations[i].x = 0.0;
			locations[i].y = 0.0;
			locations[i].z = 0.0;
		}
	} else {
		for (i = 0; i < num_locations; i++) {
			locations[i].x = (locations[i].x - min_x) / scale;
			locations[i].y = (locations[i].y - min_y) / scale;
			locations[i].z = (locations[i].z - min_z) / scale;
		}
	}
	if (extents != NULL) {
		extents->x = (max_x - min_x) / scale;
		extents->y = (max_y - min_y) / scale;
		extents->z = (max_z - min_z) / scale;
	}

	return locations;
}

go_game *go_game_board_add_star_point(go_game *game, unsigned int index)
{
	if (array_insert_sorted_no_duplicates((void **) &game->board_info.star_points, &game->board_info.num_star_points, sizeof(unsigned int), &index, (int (*)(const void *, const void *)) index_cmp) == NULL)
		return NULL;

	return game;
}

go_game *go_game_board_remove_star_point(go_game *game, unsigned int index)
{
	if (array_remove_sorted((void **) &game->board_info.star_points, &game->board_info.num_star_points, sizeof(unsigned int), &index, (int (*)(const void *, const void *)) index_cmp) == NULL)
		return NULL;

	return game;
}

static void go_game_info_free(go_game_info *info)
{
	free(info->name);
	free(info->value);

	return;
}

static int go_game_info_cmp(const go_game_info *a, const go_game_info *b)
{
	return strcmp(a->name, b->name);
}

static int go_game_info_name_cmp(const char *name, const go_game_info *info)
{
	return strcmp(name, info->name);
}

go_game *go_game_info_set(go_game *game, const char *name, const char *value)
{
	go_game_info info, *p;

	p = array_search_sorted((const void **) &game->info, &game->num_info, sizeof(go_game_info), name, (int (*)(const void *, const void *)) go_game_info_name_cmp);
	if (p == NULL) {
		info.name = NULL;
		info.value = NULL;
		p = &info;
	}
	p->name = (char *) realloc(p->name, strlen(name) + 1);
	if (p->name == NULL)
		return NULL;
	strcpy(p->name, name);
	p->value = (char *) realloc(p->value, strlen(value) + 1);
	if (p->value == NULL) {
		free(p->name);
		return NULL;
	}
	strcpy(p->value, value);
	if (array_insert_sorted_no_duplicates((void **) &game->info, &game->num_info, sizeof(go_game_info), p, (int (*)(const void *, const void *)) go_game_info_cmp) == NULL) {
		go_game_info_free(p);
		return NULL;
	}

	return game;
}

const char *go_game_info_search(const go_game *game, const char *name)
{
	go_game_info *info;

	info = array_search_sorted((const void **) &game->info, &game->num_info, sizeof(go_game_info), name, (int (*)(const void *, const void *)) go_game_info_name_cmp);
	if (info == NULL)
		return NULL;

	return info->value;
}

go_player *go_game_new_player(go_game *game)
{
	static const char *player_names[] = {
		"Black", "White", "Red", "Blue"
	};
	go_team *team;
	go_player *player;
	unsigned int index;

	team = go_game_new_team(game);
	if (team == NULL)
		return NULL;
	index = team - game->teams;
	player = go_game_new_player_team(game, index);
	if (player == NULL) {
		go_game_remove_team(game, index);
		return NULL;
	}
	if (index < sizeof(player_names) / sizeof(*player_names)) {
		if (go_player_set_name(player, player_names[index]) == NULL) {
			go_game_remove_team(game, index);
			go_game_remove_player(game, player - game->players);
			go_player_free(player);
			return NULL;
		}
	}

	return player;
}

go_player *go_game_new_player_team(go_game *game, unsigned int index)
{
	go_player player, *p;

	if (index >= game->num_teams)
		return NULL;
	if (go_player_init(&player) == NULL)
		return NULL;
	p = array_insert((void **) &game->players, &game->num_players, sizeof(go_player), &player, NULL);
	if (p == NULL)
		return NULL;
	if (go_team_add_player(&game->teams[index], p - game->players) == NULL) {
		go_game_remove_player(game, p - game->players);
		return NULL;
	}

	return p;
}

go_game *go_game_remove_player(go_game *game, unsigned int index)
{
	unsigned int i, j;

	go_player_free(&game->players[index]);
	if (array_remove_index_keep_order((void **) &game->players, &game->num_players, sizeof(go_player), index) == NULL)
		return NULL;
	for (j = 0; j < game->num_teams; j++) {
		go_team_remove_player(&game->teams[j], index);
		for (i = 0; i < game->teams[j].num_players; i++) {
			if (game->teams[j].players[i] > index)
				game->teams[j].players[i]--;
		}
	}

	return game;
}

go_team *go_game_new_team(go_game *game)
{
	static const struct {
		struct {
			float r;
			float g;
			float b;
		} display_color;
		char display_char;
		char alt_display_char;
	} team_info[] = {
		{ { 0.0, 0.0, 0.0 }, 'X', 'x' },
		{ { 1.0, 1.0, 1.0 }, 'O', 'o' },
		{ { 1.0, 0.0, 0.0 }, 'R', 'r' },
		{ { 0.0, 0.0, 1.0 }, 'B', 'b' }
	};
	go_team team, *p;
	unsigned int index;

	if (go_team_init(&team) == NULL)
		return NULL;
	p = array_insert((void **) &game->teams, &game->num_teams, sizeof(go_team), &team, NULL);
	if (p == NULL) {
		go_team_free(&team);
		return NULL;
	}
	index = p - game->teams;
	p->color = index + 1;
	if (index < sizeof(team_info) / sizeof(*team_info)) {
		game->teams[index].display_color.r = team_info[index].display_color.r;
		game->teams[index].display_color.g = team_info[index].display_color.g;
		game->teams[index].display_color.b = team_info[index].display_color.b;
		game->teams[index].display_char = team_info[index].display_char;
		game->teams[index].alt_display_char = team_info[index].alt_display_char;
	}

	return p;
}

go_game *go_game_remove_team(go_game *game, unsigned int index)
{
	unsigned int i;

	/* Update the handicaps */
	if (go_game_team_set_handicap(game, index, 0) == NULL)
		return NULL;
	if (array_remove_index_keep_order((void **) &game->teams, &game->num_teams, sizeof(go_team), index) == NULL)
		return NULL;
	for (i = index; i < game->num_teams; i++)
		game->teams[i].color--;

	return game;
}

go_game *go_game_team_set_handicap(go_game *game, unsigned int index, unsigned int handicap)
{
	game->teams[index].handicap = handicap;
	if (handicap > game->handicap_counter) {
		game->handicap_team = index;
		game->handicap_counter = handicap;
		game->current_team = index;
		game->next_team = index;
	} else if (index == game->handicap_team) {
		game->handicap_counter = 0;
		for (index = 0; index < game->num_teams; index++) {
			if (game->teams[index].handicap > game->handicap_counter) {
				game->handicap_team = index;
				game->handicap_counter = game->teams[index].handicap;
				game->current_team = index;
				game->next_team = index;
			}
		}
	}

	return game;
}

static go_game *go_game_next_player(go_game *game)
{
	game->teams[game->current_team].next_player = (game->teams[game->current_team].next_player + 1) % game->teams[game->current_team].num_players;
	game->next_team = game->current_team;
	if (game->handicap_counter == 0) {
		/* All handicaps are done */
		do {
			game->next_team = (game->next_team + 1) % game->num_teams;
		} while (game->teams[game->next_team].is_resigned &&
		         (game->next_team != game->current_team));
	} else {
		do {
			game->next_team = (game->next_team + 1) % game->num_teams;
		} while (game->teams[game->next_team].is_resigned ||
		         (game->teams[game->next_team].handicap <
		          game->handicap_counter));
		if (game->next_team == game->handicap_team) {
			game->handicap_counter--;
			do {
				game->handicap_team = (game->handicap_team + 1) % game->num_teams;
			} while (game->teams[game->handicap_team].is_resigned ||
			         (game->teams[game->handicap_team].handicap <
			          game->handicap_counter));
			if (game->teams[game->handicap_team].handicap <
			    game->teams[game->next_team].handicap)
				game->next_team = game->handicap_team;
			else
				game->handicap_team = game->next_team;
		}
	}

	return game;
}

go_game *go_game_start(go_game *game)
{
	if (go_game_new_turn(game) == NULL)
		return NULL;
	/* Do handicaps, as much as possible */

	return game;
}

go_game *go_game_end(go_game *game)
{
	go_game_count_score(game);

	return game;
}

go_game *go_game_new_turn(go_game *game)
{
	go_history_turn *turn;

	turn = go_history_new_turn(&game->history);
	if (turn == NULL)
		return NULL;
	game->history.current_turn = turn;
	game->current_team = game->next_team;
	game->teams[game->current_team].current_player = game->teams[game->current_team].next_player;
	if (game->history.current_turn->prev != NULL) {
		game->turn_num++;
	}
	array_init((void **) &game->dead_strings, &game->num_dead_strings);
	array_init((void **) &game->markup, &game->num_markup);
	if (go_game_set_state(game) == NULL) {
		go_game_history_undo(game);
		return NULL;
	}

	return game;
}

go_game *go_game_color_point(go_game *game, unsigned int index, unsigned int color)
{
	static go_history_turn_action action = { GO_HISTORY_TURN_ACTION_COLOR_POINT };
	unsigned int i, string;

	if (game->board.points[index].color == color)
		return game;
	if (go_game_unset_state(game) == NULL)
		return NULL;
	if (go_board_color_point(&game->board, index, color) == NULL)
		return NULL;
	/* If this point is part of a dead string, mark all its neighbors in the
	   same string dead, in case a string has been cut */
	if (go_game_is_dead(game, index)) {
		string = game->board.points[index].string_index;
		for (i = 0; i < game->board.points[index].num_adjacent; i++) {
			if (go_board_string_contains_point(&game->board, string, game->board.points[index].adjacent[i])) {
				if (go_game_add_dead_string(game, game->board.points[index].adjacent[i]) == NULL)
				return NULL;
			}
		}
	}
	if (go_game_set_state(game) == NULL)
		return NULL;
	action.value.index_color.index = index;
	action.value.index_color.color = color;
	if (go_history_add_action(&game->history, &action) == NULL)
		return NULL;

	return game;
}

static int go_board_board_state_cmp(const go_board *board, const go_board_state *state)
{
	unsigned int i;
	int diff;

	diff = 0;
	for (i = 0; (i < board->num_points) && (diff == 0); i++)
		diff = (int) board->points[i].color - state->colors[i];

	return diff;
}

static int go_game_is_ko_violation(const go_game *game)
{
	go_history_turn **turns, *turn;
	unsigned int i, num_turns;

	if (game->ruleset.ko == GO_RULESET_KO_NONE)
		return 0;
	go_history_search_board(&game->history, &game->board, &turns, &num_turns);
	if (game->ruleset.ko == GO_RULESET_KO_SIMPLE) {
		turn = game->history.current_turn;
		for (i = 0; i < game->num_teams - game->num_resignations; i++) {
			turn = turn->prev;
			if (turn == NULL)
				return 0;
		}
		for (i = 0; i < num_turns; i++) {
			if ((turns[i] == turn) &&
			    (go_board_board_state_cmp(&game->board, &turn->game_state.board_state) == 0))
				return 1;
		}
	} else if ((game->ruleset.ko == GO_RULESET_KO_POSITIONAL_SUPERKO) ||
	           (game->ruleset.ko == GO_RULESET_KO_SITUATIONAL_SUPERKO)) {
		for (turn = game->history.current_turn->prev; turn != NULL; turn = turn->prev) {
			for (i = 0; i < num_turns; i++) {
				if (((game->ruleset.ko == GO_RULESET_KO_POSITIONAL_SUPERKO) ||
				     (turns[i]->game_state.current_team == game->current_team)) &&
				    (turns[i] == turn) &&
				    (go_board_board_state_cmp(&game->board, &turns[i]->game_state.board_state) == 0)) {
						return 1;
				}
			}
		}
	}

	return 0;
}

static go_game *go_game_do_move(go_game *game, unsigned int index, unsigned int color, int check)
{
	unsigned int i, string, team_index, adjacent_index;
	unsigned int num_captured, *captured;

	if (check &&
	    (game->board.points[index].color != 0) &&
	    (game->ruleset.play_on_colored_points == 0))
		return NULL;
	if (go_board_color_point(&game->board, index, color) == NULL)
		return NULL;
	/* If this point is part of a dead string, mark all its neighbors in the
	   same string dead, in case a string has been cut */
	if (go_game_is_dead(game, index)) {
		string = game->board.points[index].string_index;
		for (i = 0; i < game->board.points[index].num_adjacent; i++) {
			if (go_board_string_contains_point(&game->board, string, game->board.points[index].adjacent[i])) {
				if (go_game_add_dead_string(game, game->board.points[index].adjacent[i]) == NULL)
				return NULL;
			}
		}
	}
	array_init((void **) &captured, &num_captured);
	for (i = 0; i < game->board.points[index].num_adjacent; i++) {
		adjacent_index = game->board.points[index].adjacent[i];
		if ((game->board.points[adjacent_index].color != 0) &&
		    (game->board.points[adjacent_index].color != game->board.points[index].color) &&
			(array_search((const void **) &captured, &num_captured, sizeof(unsigned int), &game->board.points[adjacent_index].string_index, (int (*)(const void *, const void *)) index_cmp) == NULL) &&
		    (go_board_point_reaches_color(&game->board, adjacent_index,  0) == 0)) {
			/* The string has no liberties. Add it to the list of captured
			   strings. We need not call array_insert_no_duplicates because
			   we have already verified that this string is not in the array. */
			if (array_insert((void **) &captured, &num_captured, sizeof(unsigned int), &game->board.points[adjacent_index].string_index, NULL) == NULL) {
				array_free((void **) &captured, &num_captured, sizeof(unsigned int), NULL);
				return NULL;
			}
		}
	}
	if (go_board_point_reaches_color(&game->board, index, 0) == 0) {
		/* If no strings are captured or simultaneous capture is on,
		   this is suicide */
		if ((num_captured == 0) || (game->ruleset.simultaneous_capture)) {
			if (check && !game->ruleset.suicide_allowed) {
				go_game_get_state(game);
				array_free((void **) &captured, &num_captured, sizeof(unsigned int), NULL);
				return NULL;
			}
			if (go_board_color_string(&game->board, game->board.points[index].string_index, 0) == NULL) {
				array_free((void **) &captured, &num_captured, sizeof(unsigned int), NULL);
				return NULL;
			}
		}
	}
	for (i = 0; i < num_captured; i++) {
		if (game->ruleset.score_captures) {
			/* Prisoners go to every team but the owner of the captured string.
			   Another way to do this is to divide up the prisoners as
			   discussed at http://www.di.fc.ul.pt/~jpn/gv/gv.htm. */
			for (team_index = 0; team_index < game->num_teams; team_index++) {
				if (team_index != game->board.points[game->board.strings[captured[i]].point_index].color)
					game->teams[team_index].num_prisoners += game->board.strings[captured[i]].size;
			}
		}
		if (go_board_color_string(&game->board, captured[i], (game->ruleset.othello_capture) ? color : 0) == NULL) {
			array_free((void **) &captured, &num_captured, sizeof(unsigned int), NULL);
			return NULL;
		}
	}
	array_free((void **) &captured, &num_captured, sizeof(unsigned int), NULL);
	if (check && go_game_is_ko_violation(game)) {
		go_game_get_state(game);
		return NULL;
	}

	return game;
}

int go_game_move_is_legal(const go_game *game, unsigned int index)
{
	go_game *g;

	g = (go_game *) game;
	if (go_game_do_move(g, index, game->teams[game->current_team].color, 1) == NULL)
		return 0;
	go_game_get_state(g);

	return 1;
}

go_game *go_game_move(go_game *game, unsigned int index)
{
	static go_history_turn_action action = { GO_HISTORY_TURN_ACTION_MOVE };
	static go_game_markup markup;

	if (go_game_unset_state(game) == NULL)
		return NULL;
	if (go_game_do_move(game, index, game->teams[game->current_team].color, 0) == NULL)
		return NULL;
	game->num_passes = 0;
	if (go_game_next_player(game) == NULL)
		return NULL;
	markup.type = GO_GAME_MARKUP_MOVE;
	markup.index = index;
	if (go_game_add_markup(game, &markup) == NULL)
		return NULL;
	if (go_game_set_state(game) == NULL)
		return NULL;
	action.value.integer = index;
	if (go_history_add_action(&game->history, &action) == NULL)
		return NULL;

	return game;
}

int go_game_pass_is_legal(const go_game *game)
{
	return game->ruleset.pass_allowed;
}

/* Unsetting and resetting the state is not necessary when the board doesn't
   change. If you decide not to, you need to call go_game_state_from_game. */
go_game *go_game_pass(go_game *game)
{
	static go_history_turn_action action = { GO_HISTORY_TURN_ACTION_PASS };
	unsigned int i;

	if (!go_game_pass_is_legal(game))
		return NULL;
	if (go_game_unset_state(game) == NULL)
		return NULL;
	if (game->ruleset.score_passes) {
		for (i = 0; i < game->num_teams; i++) {
			if (i != game->current_team)
				game->teams[i].num_prisoners++;
		}
	}
	game->num_passes++;
	if (go_game_next_player(game) == NULL)
		return NULL;
	if (go_game_set_state(game) == NULL)
		return NULL;
	if (go_history_add_action(&game->history, &action) == NULL)
		return NULL;

	return game;
}

go_game *go_game_resign(go_game *game)
{
	static go_history_turn_action action = { GO_HISTORY_TURN_ACTION_RESIGN };

	if (go_game_unset_state(game) == NULL)
		return NULL;
	game->teams[game->current_team].is_resigned = 1;
	game->num_resignations++;
	if (go_game_next_player(game) == NULL)
		return NULL;
	if (go_game_set_state(game) == NULL)
		return NULL;
	if (go_history_add_action(&game->history, &action) == NULL)
		return NULL;

	return game;
}

go_game *go_game_set_current_player(go_game *game, unsigned int team_index, unsigned int player_index)
{
	static go_history_turn_action action = { GO_HISTORY_TURN_ACTION_SET_PLAYER };

	if ((team_index != game->current_team) || (team_index != game->next_team) ||
	    (player_index != game->teams[team_index].current_player) ||
	    (player_index != game->teams[team_index].next_player)) {
		if ((team_index >= game->num_teams) ||
		    (player_index >= game->teams[team_index].num_players))
			return NULL;
		if (go_game_unset_state(game) == NULL)
			return NULL;
		game->current_team = team_index;
		game->next_team = team_index;
		game->teams[team_index].current_player = player_index;
		game->teams[team_index].next_player = player_index;
		if (go_game_set_state(game) == NULL)
			return NULL;
		action.value.player.team_index = team_index;
		action.value.player.player_index = player_index;
		if (go_history_add_action(&game->history, &action) == NULL)
			return NULL;
	}

	return game;
}

go_game *go_game_comment(go_game *game, const char *s)
{
	static go_history_turn_action action = { GO_HISTORY_TURN_ACTION_COMMENT };

	action.value.string = (char *) malloc(strlen(s) + 1);
	if (action.value.string == NULL)
		return NULL;
	strcpy(action.value.string, s);
	if (go_history_add_action(&game->history, &action) == NULL) {
		free(action.value.string);
		return NULL;
	}

	return game;
}

static go_game *go_game_add_dead_string(go_game *game, unsigned int index)
{
	if (go_game_is_dead(game, index))
		return game;
	if (array_insert((void **) &game->dead_strings, &game->num_dead_strings, sizeof(unsigned int), &index, NULL) == NULL)
		return NULL;

	return game;
}

static go_game *go_game_remove_dead_string(go_game *game, unsigned int index)
{
	unsigned int i;

	for (i = 0; i < game->num_dead_strings; i++) {
		if (go_board_string_contains_point(&game->board, game->board.points[game->dead_strings[i]].string_index, index)) {
			if (array_remove_index((void **) &game->dead_strings, &game->num_dead_strings, sizeof(unsigned int), i) == NULL)
				return NULL;
			return game;
		}
	}

	return game;
}

go_game *go_game_toggle_dead(go_game *game, unsigned int index)
{
	static go_history_turn_action action = { GO_HISTORY_TURN_ACTION_DEAD };
	unsigned int i, multiplier;

	if (game->board.points[index].color == 0)
		return game;
	if (go_game_unset_state(game) == NULL)
		return NULL;
	if (go_game_is_dead(game, index)) {
		multiplier = -1;
		if (go_game_remove_dead_string(game, index) == NULL)
			return NULL;
	} else {
		multiplier = 1;
		if (go_game_add_dead_string(game, index) == NULL)
			return NULL;
	}
	if (game->ruleset.score_captures) {
		/* Prisoners are added to or subtracted from every team but the
		   owner of the captured string, as in go_game_do_move */
		for (i = 0; i < game->num_teams; i++) {
			if (i != game->board.points[index].color)
				game->teams[i].num_prisoners += multiplier * game->board.strings[game->board.points[index].string_index].size;
		}
	}
	if (go_game_set_state(game) == NULL)
		return NULL;
	action.value.integer = index;
	if (go_history_add_action(&game->history, &action) == NULL)
		return NULL;

	return game;
}

int go_game_is_dead(const go_game *game, unsigned int index)
{
	unsigned int i;

	for (i = 0; i < game->num_dead_strings; i++) {
		if (go_board_string_contains_point(&game->board, game->board.points[game->dead_strings[i]].string_index, index))
			return 1;
	}

	return 0;
}

go_game *go_game_clear_markup(go_game *game)
{
	static go_history_turn_action action = { GO_HISTORY_TURN_ACTION_MARKUP };
	static go_game_markup *markup = &action.value.markup;

	if (go_game_unset_state(game) == NULL)
		return NULL;
	/* We don't free the markup elements because they are still referred to in
	   the history */
	array_init((void **) &game->markup, &game->num_markup);
	markup->type = GO_GAME_MARKUP_CLEAR;
	if (go_game_set_state(game) == NULL)
		return NULL;
	if (go_history_add_action(&game->history, &action) == NULL)
		return NULL;

	return game;
}

go_game *go_game_mark(go_game *game, unsigned int index, int type)
{
	static go_history_turn_action action = { GO_HISTORY_TURN_ACTION_MARKUP };
	static go_game_markup *markup = &action.value.markup;

	if (index > game->board.num_points)
		return NULL;
	markup->type = type;
	markup->index = index;
	/* Don't allow duplicate markup */
	if (array_search((const void **) &game->markup, &game->num_markup, sizeof(go_game_markup), markup, (int (*)(const void *, const void *)) go_game_markup_cmp) != NULL)
		return NULL;
	if (go_game_unset_state(game) == NULL)
		return NULL;
	if (go_game_add_markup(game, markup) == NULL)
		return NULL;
	if (go_game_set_state(game) == NULL)
		return NULL;
	if (go_history_add_action(&game->history, &action) == NULL)
		return NULL;

	return game;
}

go_game *go_game_line(go_game *game, unsigned int start, unsigned int end, int type)
{
	static go_history_turn_action action = { GO_HISTORY_TURN_ACTION_MARKUP };
	static go_game_markup *markup = &action.value.markup;
	unsigned int i;

	if ((start > game->board.num_points) ||
	    (end > game->board.num_points))
		return NULL;
	if (start == end)
		return game;
	markup->type = type;
	markup->index = start;
	markup->data.endpoint = end;
	/* Don't allow duplicate markup */
	for (i = 0; i < game->num_markup; i++) {
		if (go_game_markup_cmp(&game->markup[i], markup) == 0)
			return game;
	}
	if (go_game_unset_state(game) == NULL)
		return NULL;
	if (go_game_add_markup(game, markup) == NULL)
		return NULL;
	if (go_game_set_state(game) == NULL)
		return NULL;
	if (go_history_add_action(&game->history, &action) == NULL)
		return NULL;

	return game;
}

go_game *go_game_label(go_game *game, unsigned int index, const char *string)
{
	static go_history_turn_action action = { GO_HISTORY_TURN_ACTION_MARKUP };
	static go_game_markup *markup = &action.value.markup;

	if (index > game->board.num_points)
		return NULL;
	if (go_game_unset_state(game) == NULL)
		return NULL;
	markup->type = GO_GAME_MARKUP_LABEL;
	markup->index = index;
	markup->data.string = (char *) malloc(strlen(string) + 1);
	if (markup->data.string == NULL)
		return NULL;
	strcpy(markup->data.string, string);
	if (go_game_add_markup(game, markup) == NULL)
		return NULL;
	if (go_game_set_state(game) == NULL)
		return NULL;
	if (go_history_add_action(&game->history, &action) == NULL)
		return NULL;

	return game;
}

go_game *go_game_count_score(go_game *game)
{
	unsigned int i, color, surrounding_color;

	for (i = 0; i < game->num_teams; i++) {
		game->teams[i].territory = 0.0;
		game->teams[i].num_occupied_points = 0.0;
	}
	/* Remove all dead strings temporarily. The state of the game is
	   restored below. */
	for (i = 0; i < game->num_dead_strings; i++) {
		if (go_board_color_string(&game->board, game->board.points[game->dead_strings[i]].string_index, 0) == NULL)
			return NULL;
	}
	for (i = 0; i < game->board.num_strings; i++) {
		if (game->board.strings[i].size == 0)
			continue;
		color = game->board.points[game->board.strings[i].point_index].color;
		if (color == 0) {
			if (game->ruleset.score_territory) {
				surrounding_color = go_board_string_reaches_one_color(&game->board, i);
				if (surrounding_color != 0)
					game->teams[surrounding_color - 1].territory += game->board.strings[i].size;
			}
		} else if (game->ruleset.score_occupied_points) {
			game->teams[color - 1].num_occupied_points += game->board.strings[i].size;
		}
	}
	/* Restore the strings we temporarily removed */
	if (go_game_get_state(game) == NULL)
		return NULL;
	for (i = 0; i < game->num_teams; i++)
		game->teams[i].score = game->teams[i].territory +
		                       game->teams[i].num_occupied_points +
		                       game->teams[i].num_prisoners +
		                       game->teams[i].komi;

	return game;
}

int go_game_is_over(const go_game *game)
{
	return go_game_is_over_turn(game, game->history.current_turn);
}

int go_game_is_over_turn(const go_game *game, const go_history_turn *turn)
{
	if (turn->game_state.num_passes >= game->num_teams - turn->game_state.num_resignations)
		return 1;
	if (turn->game_state.num_resignations >= game->num_teams - 1)
		return 1;
	/* Check time limits */

	return 0;
}

static int go_game_markup_cmp(const go_game_markup *a, const go_game_markup *b)
{
	int diff;

	/* Markup should be sorted first by index, if possible */
	if ((a->type != GO_GAME_MARKUP_CLEAR) && (b->type != GO_GAME_MARKUP_CLEAR)) {
		diff = (int) a->index - (int) b->index;
		if (diff != 0)
			return diff;
	}
	diff = a->type - b->type;
	if (diff != 0)
		return diff;
	switch (a->type) {
		case GO_GAME_MARKUP_LINE:
		case GO_GAME_MARKUP_ARROW:
			diff = (int) a->data.endpoint - (int) b->data.endpoint;
			break;
		case GO_GAME_MARKUP_LABEL:
			diff = strcmp(a->data.string, b->data.string);
			break;
	}

	return diff;
}

static go_game *go_game_add_markup(go_game *game, const go_game_markup *markup)
{
	if (array_insert_sorted_no_duplicates((void **) &game->markup, &game->num_markup, sizeof(go_game_markup), markup, (int (*)(const void *, const void *)) go_game_markup_cmp) == NULL)
		return NULL;

	return game;
}
