/*
	board.c
*/

#include "config.h"

#include <stdlib.h>

#include "board.h"
#include "array.h"

static int index_cmp(const unsigned int *a, const unsigned int *b);

static go_point *go_point_add_adjacent(go_point *point, unsigned int index);

static go_point *go_point_remove_adjacent(go_point *point, unsigned int index);

static go_string *go_string_add_neighbor(go_string *string, unsigned int index);

static unsigned int go_board_new_string(go_board *board);

static go_board *go_board_join_strings(go_board *board, unsigned int a, unsigned int b);

static int index_cmp(const unsigned int *a, const unsigned int *b)
{
	return *(int *) a - *(int *) b;
}

go_point *go_point_init(go_point *point)
{
	point->color = 0;
	array_init((void **) &point->adjacent, &point->num_adjacent);
	point->string_index = 0;
	point->string_next = 0;

	return point;
}

void go_point_free(go_point *point)
{
	array_free((void **) &point->adjacent, &point->num_adjacent, sizeof(unsigned int), NULL);

	return;
}

static go_point *go_point_add_adjacent(go_point *point, unsigned int index)
{
	if (array_insert_no_duplicates((void **) &point->adjacent, &point->num_adjacent, sizeof(unsigned int), &index, (int (*)(const void *, const void *)) index_cmp) == NULL)
		return NULL;

	return point;
}

static go_point *go_point_remove_adjacent(go_point *point, unsigned int index)
{
	if (array_remove((void **) &point->adjacent, &point->num_adjacent, sizeof(unsigned int), &index, (int (*)(const void *, const void *)) index_cmp) == NULL)
		return NULL;

	return point;
}

go_string *go_string_init(go_string *string)
{
	string->size = 0;
	string->point_index = 0;
	array_init((void **) &string->neighbors, &string->num_neighbors);

	return string;
}

void go_string_free(go_string *string)
{
	array_free((void **) &string->neighbors, &string->num_neighbors, sizeof(unsigned int), NULL);

	return;
}

static go_string *go_string_add_neighbor(go_string *string, unsigned int index)
{
	if (array_insert_sorted_no_duplicates((void **) &string->neighbors, &string->num_neighbors, sizeof(unsigned int), &index, (int (*)(const void *, const void *)) index_cmp) == NULL)
		return NULL;

	return string;
}

go_board *go_board_init(go_board *board)
{
	array_init((void **) &board->points, &board->num_points);
	array_init((void **) &board->strings, &board->num_strings);
	board->new_string = 0;

	return board;
}

void go_board_free(go_board *board)
{
	array_free((void **) &board->points, &board->num_points, sizeof(go_point), (void (*)(void *)) go_point_free);
	array_free((void **) &board->strings, &board->num_strings, sizeof(go_string), (void (*)(void *)) go_string_free);

	return;
}

go_board *go_board_set_num_points(go_board *board, unsigned int num_points)
{
	unsigned int i;

	go_board_free(board);
	if (go_board_init(board) == NULL)
		return NULL;
	board->num_points = num_points;
	if (num_points == 0)
		return board;
	board->points = (go_point *) malloc(num_points * sizeof(go_point));
	if (board->points == NULL)
		return NULL;
	for (i = 0; i < num_points; i++) {
		if (go_point_init(&board->points[i]) == NULL) {
			go_board_free(board);
			return NULL;
		}
	}
	if (go_board_calculate_strings(board) == NULL)
		return NULL;

	return board;
}

go_board *go_board_add_point(go_board *board)
{
	go_point point, *p;

	if (go_point_init(&point) == NULL)
		return NULL;
	p = (go_point *) array_insert((void **) &board->points, &board->num_points, sizeof(go_point), &point, NULL);
	if (p == NULL)
		return NULL;
	/* Make the new point into its own string */
	p->string_index = go_board_new_string(board);
	if (p->string_index == (unsigned int) -1) {
		go_board_remove_point(board, board->num_points - 1);
		return NULL;
	}
	p->string_next = board->num_points - 1;
	board->strings[p->string_index].size = 1;
	board->strings[p->string_index].point_index = board->num_points - 1;

	return board;
}

go_board *go_board_remove_point(go_board *board, unsigned int index)
{
	unsigned int i, j;

	if (array_remove_index_keep_order((void **) &board->points, &board->num_points, sizeof(go_point), index) == NULL)
		return NULL;
	for (j = 0; j < board->num_points; j++) {
		go_point_remove_adjacent(&board->points[j], index);
		for (i = 0; i < board->points[j].num_adjacent; i++) {
			if (board->points[j].adjacent[i] > index)
				board->points[j].adjacent[i]--;
		}
	}
	if (go_board_calculate_strings(board) == NULL)
		return NULL;

	return board;
}

go_board *go_board_connect_points(go_board *board, unsigned int a, unsigned int b)
{
	if (a == b)
		return NULL;
	if (go_point_add_adjacent(&board->points[a], b) == NULL)
		return NULL;
	if (go_point_add_adjacent(&board->points[b], a) == NULL)
		return NULL;
	if (board->points[a].color == board->points[b].color) {
		if (go_board_join_strings(board, board->points[a].string_index, board->points[b].string_index) == NULL)
			return NULL;
	} else {
		if (go_string_add_neighbor(&board->strings[board->points[a].string_index], b) == NULL)
			return NULL;
		if (go_string_add_neighbor(&board->strings[board->points[b].string_index], a) == NULL)
			return NULL;
	}

	return board;
}

go_board *go_board_disconnect_points(go_board *board, unsigned int a, unsigned int b)
{
	if (go_point_remove_adjacent(&board->points[a], b) == NULL)
		return NULL;
	if (go_point_remove_adjacent(&board->points[b], a) == NULL)
		return NULL;
	if (go_board_calculate_strings(board) == NULL)
		return NULL;

	return board;
}

static unsigned int go_board_new_string(go_board *board)
{
	go_string string, *p;
	unsigned int i;

	if (board->new_string < board->num_strings) {
		i = board->new_string;
		go_string_init(&board->strings[i]);
		/* Look for the next empty string */
		for (board->new_string++; board->new_string < board->num_strings; board->new_string++) {
			if (board->strings[board->new_string].size == 0)
				break;
		}
	} else {
		go_string_init(&string);
		p = array_insert((void **) &board->strings, &board->num_strings, sizeof(go_string), &string, NULL);
		if (p == NULL)
			return (unsigned int) -1;
		i = p - board->strings;
		board->new_string++;
	}

	return i;
}

static go_board *go_board_join_strings(go_board *board, unsigned int a, unsigned int b)
{
	go_string *small, *big;
	unsigned int i;

	if (a == b)
		return board;
	if (board->strings[a].size < board->strings[b].size) {
		small = &board->strings[a];
		big = &board->strings[b];
	} else {
		small = &board->strings[b];
		big = &board->strings[a];
	}
	i = small->point_index;
	for (;;) {
		board->points[i].string_index = big - board->strings;
		if (board->points[i].string_next == small->point_index)
			break;
		i = board->points[i].string_next;
	}
	board->points[i].string_next = board->points[big->point_index].string_next;
	board->points[big->point_index].string_next = small->point_index;
	/* The new joined string gets all the neighbors of the two strings which
	   were joined to make it */
	for (i = 0; i < small->num_neighbors; i++) {
		if (go_string_add_neighbor(big, small->neighbors[i]) == NULL)
			return NULL;
	}
	/* The new string might have some of its constituent points for neighbors.
	   We need to remove them. */
	for (i = 0; i < big->num_neighbors; i++) {
		if (board->points[big->neighbors[i]].color == board->points[big->point_index].color) {
			if (array_remove_index_keep_order((void **) &big->neighbors, &big->num_neighbors, sizeof(unsigned int), i) == NULL)
				return NULL;
			i--;
		}
	}
	big->size += small->size;
	go_string_free(small);
	/* We must reinitialize the string because it may be freed again.
	   Also, initializing the string sets its size to 0, which marks it
	   as unused. */
	go_string_init(small);
	if (small - board->strings < board->new_string)
		board->new_string = small - board->strings;

	return board;
}

go_board *go_board_calculate_strings(go_board *board)
{
	go_point *current, *adjacent;
	unsigned int i, j;

	array_free((void **) &board->strings, &board->num_strings, sizeof(go_string), (void (*)(void *)) go_string_free);
	array_init((void **) &board->strings, &board->num_strings);
	board->new_string = 0;
	for (i = 0; i < board->num_points; i++) {
		current = &board->points[i];
		/* Make the current point a string */
		current->string_index = go_board_new_string(board);
		if (current->string_index == (unsigned int) -1)
			return NULL;
		current->string_next = i;
		board->strings[current->string_index].size = 1;
		board->strings[current->string_index].point_index = i;
		for (j = 0; j < current->num_adjacent; j++) {
			if (current->adjacent[j] < i) {
				adjacent = &board->points[current->adjacent[j]];
				if (adjacent->color == current->color) {
					if (go_board_join_strings(board, current->string_index, adjacent->string_index) == NULL)
						return NULL;
				} else {
					if (go_string_add_neighbor(&board->strings[current->string_index], current->adjacent[j]) == NULL)
						return NULL;
					if (go_string_add_neighbor(&board->strings[adjacent->string_index], i) == NULL)
						return NULL;
				}
			}
		}
	}

	return board;
}

go_board *go_board_color_point(go_board *board, unsigned int index, unsigned int color)
{
	if (index >= board->num_points)
		return NULL;
	board->points[index].color = color;
	if (go_board_calculate_strings(board) == NULL)
		return NULL;

	return board;
}

go_board *go_board_color_string(go_board *board, unsigned int index, unsigned int color)
{
	unsigned int i;

	i = board->strings[index].point_index;
	do {
		board->points[i].color = color;
		i = board->points[i].string_next;
	} while (i != board->strings[index].point_index);
	if (go_board_calculate_strings(board) == NULL)
		return NULL;

	return board;
}

int go_board_point_reaches_color(const go_board *board, unsigned int index, unsigned int color)
{
	return go_board_string_reaches_color(board, board->points[index].string_index, color);
}

int go_board_string_reaches_color(const go_board *board, unsigned int index, unsigned int color)
{
	go_string *string;
	unsigned int i;

	string = &board->strings[index];
	if (board->points[string->point_index].color == color)
		return 0;
	for (i = 0; i < string->num_neighbors; i++) {
		if (board->points[string->neighbors[i]].color == color)
			return 1;
	}

	return 0;
}

int go_board_string_reaches_one_color(const go_board *board, unsigned int index)
{
	go_string *string;
	unsigned int i, color;

	string = &board->strings[index];
	if (string->num_neighbors == 0)
		return 0;
	color = board->points[string->neighbors[0]].color;
	for (i = 1; i < string->num_neighbors; i++)
		if (board->points[string->neighbors[i]].color != color)
			return 0;

	return color;
}

int go_board_string_contains_point(const go_board *board, unsigned int string_index, unsigned int point_index)
{
	unsigned int i;

	i = board->strings[string_index].point_index;
	do {
		if (i == point_index)
			return 1;
		i = board->points[i].string_next;
	} while (i != board->strings[string_index].point_index);

	return 0;
}

/*
void go_board_print_dot(const go_board *board, FILE *fp)
{
	unsigned char colors[][3] = {
		{ 0, 0, 0 }, { 48, 48, 48 }, { 208, 208, 208 }, { 208, 0, 0 },
		{ 0, 0, 208 }
	};
	unsigned int i, j;

	fprintf(fp, "\
digraph \"board\" {\n\
overlap=scale\n\
splines=true\n\
sep=.1\n\
node [style=filled,shape=circle,height=0.5,fixedsize=true]\n\
edge [dir=none]\n");
	for (i = 0; i < board->num_points; i++) {
		if (board->points[i].color == 0)
			fprintf(fp,
			        "\"%u\" [color=\"#%02X%02X%02X\",label=\"\",height=0.1]\n",
			        i, colors[board->points[i].color][0],
			        colors[board->points[i].color][1],
			        colors[board->points[i].color][2]);
		else if (board->points[i].color < sizeof(colors) / sizeof(*colors))
			fprintf(fp, "\"%u\" [color=\"#%02X%02X%02X\",label=\"\"]\n",
			        i, colors[board->points[i].color][0],
			        colors[board->points[i].color][1],
			        colors[board->points[i].color][2]);
		else
			fprintf(fp, "\"%u\" [color=\"#%02X%02X%02X\",label=\"?\"]\n",
			        i, 255, 0, 255);
	}
	for (i = 0; i < board->num_points; i++)
		for (j = 0; j < board->points[i].num_adjacent; j++)
			if (board->points[i].adjacent[j] < i)
				fprintf(fp, "\"%u\" -> \"%u\"\n",
				        i, board->points[i].adjacent[j]);
	fprintf(fp, "}\n");

	return;
}

void go_board_print_strings(const go_board *board, FILE *fp)
{
	unsigned int i, j;

	for (i = 0; i < board->num_strings; i++) {
		if (board->strings[i].size == 0)
			continue;
		fprintf(fp, "String %u (%u):", i, board->strings[i].size);
		for (j = 0; j < board->strings[i].num_neighbors; j++)
			fprintf(fp, " %u", board->strings[i].neighbors[j]);
		putc('\n', fp);
	}

	return;
}

void go_board_print_strings_rectangular(const go_board *board, FILE *fp, unsigned int width, unsigned int height)
{
	unsigned int x, y, string_index;
	int num_digits;

	if ((width == 0) || (height == 0) || (board->num_points != width * height))
		return;
	num_digits = (int) log10(height) + 1;
	fprintf(fp, "%*s", num_digits + 1, "");
	for (x = 0; (x < width) && (x < 'I' - 'A'); x++)
		fprintf(fp, " %c", x + 'A');
	for (; (x < width) && (x < 'Z' - 'A'); x++)
		fprintf(fp, " %c", x + 'A' + 1);
	for (; x < width; x++)
		fprintf(fp, "  ");
	putc('\n', fp);
	for (y = 0; y < height; y++) {
		fprintf(fp, "%*u", num_digits + 1, height - y);
		for (x = 0; x < width; x++) {
			string_index = board->points[y * width + x].string_index;
			fprintf(fp, "%2u", string_index % 100);
		}
		fprintf(fp, "%*u", num_digits + 1, height - y);
		putc('\n', fp);
	}
	fprintf(fp, "%*s", num_digits + 1, "");
	for (x = 0; (x < width) && (x < 'I' - 'A'); x++)
		fprintf(fp, " %c", x + 'A');
	for (; (x < width) && (x < 'Z' - 'A'); x++)
		fprintf(fp, " %c", x + 'A' + 1);
	for (; x < width; x++)
		fprintf(fp, "  ");
	putc('\n', fp);

	return;
}
*/
