/*
	array.c
*/

#include "config.h"

#include <stdlib.h>
#include <string.h>

#include "array.h"

void array_init(void **array, unsigned int *num)
{
	*array = NULL;
	*num = 0;

	return;
}

void array_free(void **array, unsigned int *num, size_t size, void (*f)(void *))
{
	unsigned int i;
	char *p;

	p = *(char **) array;
	if (f != NULL) {
		for (i = 0; i < *num; i++)
			f(p + i * size);
	}
	free(p);

	return;
}

void *array_duplicate(void **array, unsigned int *num, size_t size, const void *from, unsigned int from_num, void *(*duplicate)(void *, const void *))
{
	*array = malloc(from_num * size);
	if (*array == NULL)
		return NULL;
	if (duplicate == NULL) {
		memcpy(*array, from, from_num * size);
		*num = from_num;
	} else {
		for (*num = 0; *num < from_num; (*num)++) {
			if (duplicate(*(char **) array + *num * size, (char *) from + *num * size) == NULL)
				/* Don't try to free anything, because we don't know how to
				   free the individual elements */
				return NULL;
		}
	}

	return *array;
}

void array_apply(void **array, unsigned int *num, size_t size, void (*f)(void *, void *), void *data)
{
	char *p;
	unsigned int i;

	p = *(char **) array;
	for (i = 0; i < *num; i++)
		f(p + i * size, data);

	return;
}

void array_sort(void **array, const unsigned int *num, size_t size, int (*cmp)(const void *, const void *))
{
	qsort(*array, *num, size, cmp);

	return;
}

void *array_search(const void **array, const unsigned int *num, size_t size, const void *element, int (*cmp)(const void *, const void *))
{
	const char *p;
	unsigned int i;

	p = *(const char **) array;
	for (i = 0; i < *num; i++)
		if (cmp(element, p + i * size) == 0)
			return (void *) (p + i * size);

	return NULL;
}

void *array_insert_index(void **array, unsigned int *num, size_t size, const void *element, unsigned int index)
{
	char *p;

	if (index > *num)
		return NULL;
	p = (char *) realloc(*array, (*num + 1) * size);
	if (p == NULL)
		return NULL;
	*array = (void *) p;
	p += index * size;
	memmove(p + size, p, (*num - index) * size);
	memcpy(p, element, size);
	(*num)++;

	return (void *) p;
}

void *array_remove_index(void **array, unsigned int *num, size_t size, unsigned int index)
{
	char *p;

	if (index >= *num)
		return *array;
	p = *(char **) array;
	(*num)--;
	if (index < *num)
		memmove(p + index * size, p + *num * size, size);
	if (*num * size == 0) {
		free(p);
		p = (char *) malloc(0);
	} else {
		p = (char *) realloc(p, *num * size);
	}
	if (p == NULL)
		return NULL;
	*array = (void *) p;

	return *array;
}

void *array_remove_index_keep_order(void **array, unsigned int *num, size_t size, unsigned int index)
{
	char *p;

	if (index >= *num)
		return *array;
	p = *(char **) array;
	(*num)--;
	memmove(p + index * size, p + (index + 1) * size, (*num - index) * size);
	if (*num * size == 0) {
		free(p);
		p = (char *) malloc(0);
	} else {
		p = (char *) realloc(p, *num * size);
	}
	if (p == NULL)
		return NULL;
	*array = (void *) p;

	return *array;
}

void *array_insert(void **array, unsigned int *num, size_t size, const void *element, int (*cmp)(const void *, const void *))
{
	return array_insert_index(array, num, size, element, *num);
}

void *array_insert_no_duplicates(void **array, unsigned int *num, size_t size, const void *element, int (*cmp)(const void *, const void *))
{
	char *p;
	unsigned int i;

	p = *(char **) array;
	for (i = 0; i < *num; i++) {
		if (cmp(element, p + i * size) == 0) {
			/* Overwrite the old element. This way we can update elements
			   based on some comparison function. Don't do this is if
			   it might overwrite a pointer to allocated memory; check
			   first with array_search. */
			memcpy(p + i * size, element, size);
			return (void *) (p + i * size);
		}
	}

	return array_insert_index(array, num, size, element, *num);
}

void *array_remove(void **array, unsigned int *num, size_t size, const void *element, int (*cmp)(const void *, const void *))
{
	char *p;
	unsigned int i;

	p = *(char **) array;
	for (i = 0; i < *num; i++) {
		if (cmp(element, p + i * size) == 0) {
			if (array_remove_index(array, num, size, i) == NULL)
				return NULL;
			i--;
		}
	}

	return *array;
}

void *array_remove_keep_order(void **array, unsigned int *num, size_t size, const void *element, int (*cmp)(const void *, const void *))
{
	char *p;
	unsigned int i;

	p = *(char **) array;
	for (i = 0; i < *num; i++) {
		if (cmp(element, p + i * size) == 0) {
			if (array_remove_index_keep_order(array, num, size, i) == NULL)
				return NULL;
			i--;
		}
	}

	return *array;
}

void *array_search_sorted(const void **array, const unsigned int *num, size_t size, const void *element, int (*cmp)(const void *, const void *))
{
	return bsearch(element, *array, *num, size, cmp);
}

void *array_insert_sorted(void **array, unsigned int *num, size_t size, const void *element, int (*cmp)(const void *, const void *))
{
	char *p;
	unsigned int i, l, u;
	int diff;

	p = *(char **) array;
	i = 0;
	l = 0;
	u = *num;
	while (l < u) {
		i = (l + u) / 2;
		diff = cmp(element, p + i * size);
		if (diff < 0)
			u = i;
		else if (diff >= 0)
			l = i + 1;
	}
	i = (l + u) / 2;

	return array_insert_index(array, num, size, element, i);
}

void *array_insert_sorted_no_duplicates(void **array, unsigned int *num, size_t size, const void *element, int (*cmp)(const void *, const void *))
{
	unsigned int i, l, u;
	int diff;
	char *p;

	p = *(char **) array;
	i = 0;
	l = 0;
	u = *num;
	while (l < u) {
		i = (l + u) / 2;
		diff = cmp(element, p + i * size);
		if (diff < 0)
			u = i;
		else if (diff > 0)
			l = i + 1;
		else if (diff == 0) {
			/* Overwrite the old element. See the comment on
			   array_insert_no_duplicates above. */
			memcpy(p + i * size, element, size);
			return (void *) (p + i * size);
		}
	}
	i = (l + u) / 2;

	return array_insert_index(array, num, size, element, i);
}

void *array_remove_sorted(void **array, unsigned int *num, size_t size, const void *element, int (*cmp)(const void *, const void *))
{
	unsigned int i, l, u, matching;
	int diff;
	char *p;

	p = *(char **) array;
	i = 0;
	l = 0;
	u = *num;
	while (l < u) {
		i = (l + u) / 2;
		diff = cmp(element, p + i * size);
		if (diff <= 0)
			u = i;
		else if (diff > 0)
			l = i + 1;
	}
	i = (l + u) / 2;
	matching = 0;
	while ((i + matching < *num) &&
	       (cmp(element, p + (i + matching) * size) == 0))
		matching++;
	*num -= matching;
	memmove(p + i * size, p + (i + matching) * size, (*num - i) * size);
	if (*num * size == 0) {
		free(p);
		p = (char *) malloc(0);
	} else {
		p = (char *) realloc(p, *num * size);
	}
	*array = (void *) p;

	return *array;
}
