// This is the browser half of the fetch-rpc WebExtension, which exposes the
// browser's fetch API (https://developer.mozilla.org/en-US/docs/Web/API/Fetch_API)
// on a socket. The native half of the extension listens on a socket and passes
// parameters to this, the browser half, which is what actually does the fetch.
//
// Each received WebExtension message represents a request to do a fetch. Each
// request is tagged with an ID. When fetch returns a result, we tag it with the
// same ID before sending it back to the native half. The native half of the
// extension uses the ID to match up asynchronous requests and responses.
//
// Examples:
// A simple request:
//   {
//     "id": "11111111",
//     "request": {
//       "input": "https://example.com/"
//     }
//   }
// The result:
//   {
//     "id": "11111111",
//     "response": {
//       "body": "PCFk...Cg==",
//       "headers": {},
//       "ok": true,
//       "redirected": false,
//       "status": 200,
//       "statusText": "OK",
//       "type": "basic",
//       "url": "https://example.com/"
//     }
//   }
//
// A request that results in an error:
//   {
//     "id": "22222222",
//     "request": {
//       "input": "https://example.com:999/"
//     }
//   }
// The error result:
//   {
//     "id": "22222222",
//     "error": {
//       "message": "NetworkError when attempting to fetch resource.",
//       "name": "TypeError"
//     }
//   }

// https://developer.mozilla.org/en-US/docs/Mozilla/Add-ons/WebExtensions/Chrome_incompatibilities#Callbacks_and_the_chrome.*_namespace
browser = typeof(browser) !== "undefined" ? browser : chrome;

// Decode a base64-encoded string into an ArrayBuffer.
function base64_decode(enc_str) {
    // First step is to decode the base64. atob returns a byte string; i.e., a
    // string of 16-bit characters, each of whose character codes is restricted
    // to the range 0x00–0xff.
    let dec_str = atob(enc_str);
    // Next, copy those character codes into an array of 8-bit elements.
    let dec_array = new Uint8Array(dec_str.length);
    for (let i = 0; i < dec_str.length; i++) {
        dec_array[i] = dec_str.charCodeAt(i);
    }
    return dec_array.buffer;
}

// Encode an ArrayBuffer into a base64-encoded string.
function base64_encode(dec_buf) {
    let dec_array = new Uint8Array(dec_buf);
    // Copy the elements of the array into a new byte string. Use this
    // reduce-and-join technique instead of String.fromCharCode(...dec_array) to
    // avoid stack overflow errors on Chromium.
    let dec_str = dec_array.reduce((l, x) => { l.push(String.fromCharCode(x)); return l; }, []).join("");
    // base64-encode the byte string.
    return btoa(dec_str);
}

// A Mutex's lock function returns a promise that resolves to a function which,
// when called, allows the next call to lock to proceed.
// https://stackoverflow.com/a/51086893
function Mutex() {
    // Initially unlocked.
    let p = Promise.resolve();
    this.lock = function() {
        let old_p = p;
        let unlock;
        // Make a new promise for the *next* caller to wait on. Copy the new
        // promise's resolve function into the outer scope as "unlock".
        p = new Promise(resolve => unlock = resolve);
        // The caller gets a promise that allows them to unlock the *next*
        // caller.
        return old_p.then(() => unlock);
    }
}

// Enforce exclusive access to the onBeforeSendHeaders listener.
const headersMutex = new Mutex();

async function handle(params) {
    // Convert the params into a Request. Most of the params are strings and
    // need no special conversion.
    // https://developer.mozilla.org/en-US/docs/Web/API/Request/Request#Parameters
    const input = params.input;
    const init = params.init != null ? params.init : {};
    if (init.body != null) {
        init.body = base64_decode(init.body);
    }
    const request = new Request(input, init);
    // init.headers gets further treatment in the headersFn listener below.

    // We need to use a webRequest.onBeforeSendHeaders listener to override
    // certain header fields, including Host (creating a Request with them in
    // init.headers does not work). But onBeforeSendHeaders is a global setting
    // (applies to all requests) and we need to be able to set different headers
    // per request. We make it so that any onBeforeSendHeaders listener is only
    // used for a single request, by acquiring a lock here and releasing it
    // within the listener itself. The lock is acquired and released before any
    // network communication happens; i.e., it's fast.
    const headersUnlock = await headersMutex.lock();
    function headersFn(details) {
        try {
            let replacements = new Headers(init.headers != null ? init.headers : {});
            // Remove all browser headers that conflict with the requested headers.
            let requestHeaders = details.requestHeaders.filter(header => !replacements.has(header.name));
            // Append the requested headers in array form.
            // https://developer.mozilla.org/en-US/docs/Mozilla/Add-ons/WebExtensions/API/webRequest/HttpHeaders
            for (let [name, value] of replacements) {
                requestHeaders.push({name, value});
            }
            return {requestHeaders};
        } catch (error) {
            // In case of any error in the code above, play it safe and cancel
            // the request.
            console.log(`${browser.runtime.id}: error in onBeforeSendHeaders: ${error.message}`);
            return {cancel: true};
        } finally {
            // Now that the listener has been called, remove it and release the
            // lock to allow the next request to set a different listener.
            browser.webRequest.onBeforeSendHeaders.removeListener(headersFn);
            headersUnlock();
        }
    }

    try {
        // Set a listener that overrides the headers for this request.
        // https://developer.mozilla.org/en-US/docs/Mozilla/Add-ons/WebExtensions/API/webRequest/onBeforeSendHeaders
        browser.webRequest.onBeforeSendHeaders.addListener(
            headersFn,
            {urls: ["<all_urls>"]},
            ["blocking", "requestHeaders"]
        );

        // Now actually do the request and build a response object.
        let response = await fetch(request);
        return {
            // https://developer.mozilla.org/en-US/docs/Web/API/Response#Properties
            headers: response.headers,
            ok: response.ok,
            redirected: response.redirected,
            status: response.status,
            statusText: response.statusText,
            type: response.type,
            url: response.url,
            body: base64_encode(await response.arrayBuffer()),
        };
    } finally {
        // With certain errors (e.g. an invalid URL), our onBeforeSendHeaders
        // listener may never get called, and therefore never release its lock.
        // Ensure that the lock is released and listener removed in any case.
        // It's safe to release a lock or remove a listener more than once.
        browser.webRequest.onBeforeSendHeaders.removeListener(headersFn);
        headersUnlock();
    }
}

// Set a top-level error logger for webRequest, to aid debugging.
browser.webRequest.onErrorOccurred.addListener(
    details => console.log(`${browser.runtime.id}: webRequest error:`, details),
    {urls: ["<all_urls>"]}
);

// Connect to our native process.
const port = browser.runtime.connectNative("fetch.rpc");

port.onMessage.addListener(message => {
    handle(message.request)
        .then(response => ({response: response}))
        .catch(error => ({error: {name: error.name, message: error.message}}))
        .then(result => port.postMessage(Object.assign({id: message.id}, result)));
});

port.onDisconnect.addListener(p => {
    // https://developer.mozilla.org/en-US/docs/Mozilla/Add-ons/WebExtensions/API/runtime/Port#Type
    // "Note that in Google Chrome port.error is not supported: instead, use
    // runtime.lastError to get the error message."
    let error = p.error || browser.runtime.lastError;
    if (error) {
        console.log(`${browser.runtime.id}: disconnected because of error: ${error.message}`);
    }
});
